<?php
/**
 * SimplePie
 *
 * A PHP-Based RSS and Atom Feed Framework.
 * Takes the hard work out of managing a complete RSS/Atom solution.
 *
 * Please note: This file is automatically generated by a build script. The
 * full original source is always available from http://simplepie.org/
 *
 * Copyright (c) 2004-2023, Ryan Parman, Sam Sneddon, Ryan McCue, and contributors
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 * 	* Redistributions of source code must retain the above copyright notice, this list of
 * 	  conditions and the following disclaimer.
 *
 * 	* Redistributions in binary form must reproduce the above copyright notice, this list
 * 	  of conditions and the following disclaimer in the documentation and/or other materials
 * 	  provided with the distribution.
 *
 * 	* Neither the name of the SimplePie Team nor the names of its contributors may be used
 * 	  to endorse or promote products derived from this software without specific prior
 * 	  written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS
 * AND CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * @link http://simplepie.org/ SimplePie
 * @license http://www.opensource.org/licenses/bsd-license.php BSD License
 */

// SPDX-FileCopyrightText: 2004-2023 Ryan Parman, Sam Sneddon, Ryan McCue
// SPDX-License-Identifier: BSD-3-Clause

namespace SimplePie {

use InvalidArgumentException;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\UriFactoryInterface;
use Psr\SimpleCache\CacheInterface;
use SimplePie\Cache\Base;
use SimplePie\Cache\BaseDataCache;
use SimplePie\Cache\CallableNameFilter;
use SimplePie\Cache\DataCache;
use SimplePie\Cache\NameFilter;
use SimplePie\Cache\Psr16;
use SimplePie\Content\Type\Sniffer;
use SimplePie\Exception as SimplePieException;
use SimplePie\HTTP\Client;
use SimplePie\HTTP\ClientException;
use SimplePie\HTTP\FileClient;
use SimplePie\HTTP\Psr18Client;
use SimplePie\HTTP\Response;

/**
 * SimplePie
 */
class SimplePie
{
    /**
     * SimplePie Name
     */
    public const NAME = 'SimplePie';

    /**
     * SimplePie Version
     */
    public const VERSION = '1.9.0';

    /**
     * SimplePie Website URL
     */
    public const URL = 'http://simplepie.org';

    /**
     * SimplePie Linkback
     */
    public const LINKBACK = '<a href="' . self::URL . '" title="' . self::NAME . ' ' . self::VERSION . '">' . self::NAME . '</a>';

    /**
     * No Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_NONE = 0;

    /**
     * Feed Link Element Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_AUTODISCOVERY = 1;

    /**
     * Local Feed Extension Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_LOCAL_EXTENSION = 2;

    /**
     * Local Feed Body Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_LOCAL_BODY = 4;

    /**
     * Remote Feed Extension Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_REMOTE_EXTENSION = 8;

    /**
     * Remote Feed Body Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_REMOTE_BODY = 16;

    /**
     * All Feed Autodiscovery
     * @see SimplePie::set_autodiscovery_level()
     */
    public const LOCATOR_ALL = 31;

    /**
     * No known feed type
     */
    public const TYPE_NONE = 0;

    /**
     * RSS 0.90
     */
    public const TYPE_RSS_090 = 1;

    /**
     * RSS 0.91 (Netscape)
     */
    public const TYPE_RSS_091_NETSCAPE = 2;

    /**
     * RSS 0.91 (Userland)
     */
    public const TYPE_RSS_091_USERLAND = 4;

    /**
     * RSS 0.91 (both Netscape and Userland)
     */
    public const TYPE_RSS_091 = 6;

    /**
     * RSS 0.92
     */
    public const TYPE_RSS_092 = 8;

    /**
     * RSS 0.93
     */
    public const TYPE_RSS_093 = 16;

    /**
     * RSS 0.94
     */
    public const TYPE_RSS_094 = 32;

    /**
     * RSS 1.0
     */
    public const TYPE_RSS_10 = 64;

    /**
     * RSS 2.0
     */
    public const TYPE_RSS_20 = 128;

    /**
     * RDF-based RSS
     */
    public const TYPE_RSS_RDF = 65;

    /**
     * Non-RDF-based RSS (truly intended as syndication format)
     */
    public const TYPE_RSS_SYNDICATION = 190;

    /**
     * All RSS
     */
    public const TYPE_RSS_ALL = 255;

    /**
     * Atom 0.3
     */
    public const TYPE_ATOM_03 = 256;

    /**
     * Atom 1.0
     */
    public const TYPE_ATOM_10 = 512;

    /**
     * All Atom
     */
    public const TYPE_ATOM_ALL = 768;

    /**
     * All feed types
     */
    public const TYPE_ALL = 1023;

    /**
     * No construct
     */
    public const CONSTRUCT_NONE = 0;

    /**
     * Text construct
     */
    public const CONSTRUCT_TEXT = 1;

    /**
     * HTML construct
     */
    public const CONSTRUCT_HTML = 2;

    /**
     * XHTML construct
     */
    public const CONSTRUCT_XHTML = 4;

    /**
     * base64-encoded construct
     */
    public const CONSTRUCT_BASE64 = 8;

    /**
     * IRI construct
     */
    public const CONSTRUCT_IRI = 16;

    /**
     * A construct that might be HTML
     */
    public const CONSTRUCT_MAYBE_HTML = 32;

    /**
     * All constructs
     */
    public const CONSTRUCT_ALL = 63;

    /**
     * Don't change case
     */
    public const SAME_CASE = 1;

    /**
     * Change to lowercase
     */
    public const LOWERCASE = 2;

    /**
     * Change to uppercase
     */
    public const UPPERCASE = 4;

    /**
     * PCRE for HTML attributes
     */
    public const PCRE_HTML_ATTRIBUTE = '((?:[\x09\x0A\x0B\x0C\x0D\x20]+[^\x09\x0A\x0B\x0C\x0D\x20\x2F\x3E][^\x09\x0A\x0B\x0C\x0D\x20\x2F\x3D\x3E]*(?:[\x09\x0A\x0B\x0C\x0D\x20]*=[\x09\x0A\x0B\x0C\x0D\x20]*(?:"(?:[^"]*)"|\'(?:[^\']*)\'|(?:[^\x09\x0A\x0B\x0C\x0D\x20\x22\x27\x3E][^\x09\x0A\x0B\x0C\x0D\x20\x3E]*)?))?)*)[\x09\x0A\x0B\x0C\x0D\x20]*';

    /**
     * PCRE for XML attributes
     */
    public const PCRE_XML_ATTRIBUTE = '((?:\s+(?:(?:[^\s:]+:)?[^\s:]+)\s*=\s*(?:"(?:[^"]*)"|\'(?:[^\']*)\'))*)\s*';

    /**
     * XML Namespace
     */
    public const NAMESPACE_XML = 'http://www.w3.org/XML/1998/namespace';

    /**
     * Atom 1.0 Namespace
     */
    public const NAMESPACE_ATOM_10 = 'http://www.w3.org/2005/Atom';

    /**
     * Atom 0.3 Namespace
     */
    public const NAMESPACE_ATOM_03 = 'http://purl.org/atom/ns#';

    /**
     * RDF Namespace
     */
    public const NAMESPACE_RDF = 'http://www.w3.org/1999/02/22-rdf-syntax-ns#';

    /**
     * RSS 0.90 Namespace
     */
    public const NAMESPACE_RSS_090 = 'http://my.netscape.com/rdf/simple/0.9/';

    /**
     * RSS 1.0 Namespace
     */
    public const NAMESPACE_RSS_10 = 'http://purl.org/rss/1.0/';

    /**
     * RSS 1.0 Content Module Namespace
     */
    public const NAMESPACE_RSS_10_MODULES_CONTENT = 'http://purl.org/rss/1.0/modules/content/';

    /**
     * RSS 2.0 Namespace
     * (Stupid, I know, but I'm certain it will confuse people less with support.)
     */
    public const NAMESPACE_RSS_20 = '';

    /**
     * DC 1.0 Namespace
     */
    public const NAMESPACE_DC_10 = 'http://purl.org/dc/elements/1.0/';

    /**
     * DC 1.1 Namespace
     */
    public const NAMESPACE_DC_11 = 'http://purl.org/dc/elements/1.1/';

    /**
     * W3C Basic Geo (WGS84 lat/long) Vocabulary Namespace
     */
    public const NAMESPACE_W3C_BASIC_GEO = 'http://www.w3.org/2003/01/geo/wgs84_pos#';

    /**
     * GeoRSS Namespace
     */
    public const NAMESPACE_GEORSS = 'http://www.georss.org/georss';

    /**
     * Media RSS Namespace
     */
    public const NAMESPACE_MEDIARSS = 'http://search.yahoo.com/mrss/';

    /**
     * Wrong Media RSS Namespace. Caused by a long-standing typo in the spec.
     */
    public const NAMESPACE_MEDIARSS_WRONG = 'http://search.yahoo.com/mrss';

    /**
     * Wrong Media RSS Namespace #2. New namespace introduced in Media RSS 1.5.
     */
    public const NAMESPACE_MEDIARSS_WRONG2 = 'http://video.search.yahoo.com/mrss';

    /**
     * Wrong Media RSS Namespace #3. A possible typo of the Media RSS 1.5 namespace.
     */
    public const NAMESPACE_MEDIARSS_WRONG3 = 'http://video.search.yahoo.com/mrss/';

    /**
     * Wrong Media RSS Namespace #4. New spec location after the RSS Advisory Board takes it over, but not a valid namespace.
     */
    public const NAMESPACE_MEDIARSS_WRONG4 = 'http://www.rssboard.org/media-rss';

    /**
     * Wrong Media RSS Namespace #5. A possible typo of the RSS Advisory Board URL.
     */
    public const NAMESPACE_MEDIARSS_WRONG5 = 'http://www.rssboard.org/media-rss/';

    /**
     * iTunes RSS Namespace
     */
    public const NAMESPACE_ITUNES = 'http://www.itunes.com/dtds/podcast-1.0.dtd';

    /**
     * XHTML Namespace
     */
    public const NAMESPACE_XHTML = 'http://www.w3.org/1999/xhtml';

    /**
     * IANA Link Relations Registry
     */
    public const IANA_LINK_RELATIONS_REGISTRY = 'http://www.iana.org/assignments/relation/';

    /**
     * No file source
     */
    public const FILE_SOURCE_NONE = 0;

    /**
     * Remote file source
     */
    public const FILE_SOURCE_REMOTE = 1;

    /**
     * Local file source
     */
    public const FILE_SOURCE_LOCAL = 2;

    /**
     * fsockopen() file source
     */
    public const FILE_SOURCE_FSOCKOPEN = 4;

    /**
     * cURL file source
     */
    public const FILE_SOURCE_CURL = 8;

    /**
     * file_get_contents() file source
     */
    public const FILE_SOURCE_FILE_GET_CONTENTS = 16;

    /**
     * @internal Default value of the HTTP Accept header when fetching/locating feeds
     */
    public const DEFAULT_HTTP_ACCEPT_HEADER = 'application/atom+xml, application/rss+xml, application/rdf+xml;q=0.9, application/xml;q=0.8, text/xml;q=0.8, text/html;q=0.7, unknown/unknown;q=0.1, application/unknown;q=0.1, */*;q=0.1';

    /**
     * @var array<string, mixed> Raw data
     * @access private
     */
    public $data = [];

    /**
     * @var string|string[]|null Error string (or array when multiple feeds are initialized)
     * @access private
     */
    public $error = null;

    /**
     * @var int HTTP status code
     * @see SimplePie::status_code()
     * @access private
     */
    public $status_code = 0;

    /**
     * @var Sanitize instance of Sanitize class
     * @see SimplePie::set_sanitize_class()
     * @access private
     */
    public $sanitize;

    /**
     * @var string SimplePie Useragent
     * @see SimplePie::set_useragent()
     * @access private
     */
    public $useragent = '';

    /**
     * @var string Feed URL
     * @see SimplePie::set_feed_url()
     * @access private
     */
    public $feed_url;

    /**
     * @var ?string Original feed URL, or new feed URL iff HTTP 301 Moved Permanently
     * @see SimplePie::subscribe_url()
     * @access private
     */
    public $permanent_url = null;

    /**
     * @var File Instance of File class to use as a feed
     * @see SimplePie::set_file()
     */
    private $file;

    /**
     * @var string|false Raw feed data
     * @see SimplePie::set_raw_data()
     * @access private
     */
    public $raw_data;

    /**
     * @var int Timeout for fetching remote files
     * @see SimplePie::set_timeout()
     * @access private
     */
    public $timeout = 10;

    /**
     * @var array<int, mixed> Custom curl options
     * @see SimplePie::set_curl_options()
     * @access private
     */
    public $curl_options = [];

    /**
     * @var bool Forces fsockopen() to be used for remote files instead
     * of cURL, even if a new enough version is installed
     * @see SimplePie::force_fsockopen()
     * @access private
     */
    public $force_fsockopen = false;

    /**
     * @var bool Force the given data/URL to be treated as a feed no matter what
     * it appears like
     * @see SimplePie::force_feed()
     * @access private
     */
    public $force_feed = false;

    /**
     * @var bool Enable/Disable Caching
     * @see SimplePie::enable_cache()
     * @access private
     */
    private $enable_cache = true;

    /**
     * @var DataCache|null
     * @see SimplePie::set_cache()
     */
    private $cache = null;

    /**
     * @var NameFilter
     * @see SimplePie::set_cache_namefilter()
     */
    private $cache_namefilter;

    /**
     * @var bool Force SimplePie to fallback to expired cache, if enabled,
     * when feed is unavailable.
     * @see SimplePie::force_cache_fallback()
     * @access private
     */
    public $force_cache_fallback = false;

    /**
     * @var int Cache duration (in seconds)
     * @see SimplePie::set_cache_duration()
     * @access private
     */
    public $cache_duration = 3600;

    /**
     * @var int Auto-discovery cache duration (in seconds)
     * @see SimplePie::set_autodiscovery_cache_duration()
     * @access private
     */
    public $autodiscovery_cache_duration = 604800; // 7 Days.

    /**
     * @var string Cache location (relative to executing script)
     * @see SimplePie::set_cache_location()
     * @access private
     */
    public $cache_location = '/home/pillowan/.ccache';

    /**
     * @var string&(callable(string): string) Function that creates the cache filename
     * @see SimplePie::set_cache_name_function()
     * @access private
     */
    public $cache_name_function = 'md5';

    /**
     * @var bool Reorder feed by date descending
     * @see SimplePie::enable_order_by_date()
     * @access private
     */
    public $order_by_date = true;

    /**
     * @var mixed Force input encoding to be set to the follow value
     * (false, or anything type-cast to false, disables this feature)
     * @see SimplePie::set_input_encoding()
     * @access private
     */
    public $input_encoding = false;

    /**
     * @var self::LOCATOR_* Feed Autodiscovery Level
     * @see SimplePie::set_autodiscovery_level()
     * @access private
     */
    public $autodiscovery = self::LOCATOR_ALL;

    /**
     * Class registry object
     *
     * @var Registry
     */
    public $registry;

    /**
     * @var int Maximum number of feeds to check with autodiscovery
     * @see SimplePie::set_max_checked_feeds()
     * @access private
     */
    public $max_checked_feeds = 10;

    /**
     * @var array<Response>|null All the feeds found during the autodiscovery process
     * @see SimplePie::get_all_discovered_feeds()
     * @access private
     */
    public $all_discovered_feeds = [];

    /**
     * @var string Web-accessible path to the handler_image.php file.
     * @see SimplePie::set_image_handler()
     * @access private
     */
    public $image_handler = '';

    /**
     * @var array<string> Stores the URLs when multiple feeds are being initialized.
     * @see SimplePie::set_feed_url()
     * @access private
     */
    public $multifeed_url = [];

    /**
     * @var array<int, static> Stores SimplePie objects when multiple feeds initialized.
     * @access private
     */
    public $multifeed_objects = [];

    /**
     * @var array<mixed> Stores the get_object_vars() array for use with multifeeds.
     * @see SimplePie::set_feed_url()
     * @access private
     */
    public $config_settings = null;

    /**
     * @var int Stores the number of items to return per-feed with multifeeds.
     * @see SimplePie::set_item_limit()
     * @access private
     */
    public $item_limit = 0;

    /**
     * @var bool Stores if last-modified and/or etag headers were sent with the
     * request when checking a feed.
     */
    public $check_modified = false;

    /**
     * @var array<string> Stores the default attributes to be stripped by strip_attributes().
     * @see SimplePie::strip_attributes()
     * @access private
     */
    public $strip_attributes = ['bgsound', 'class', 'expr', 'id', 'style', 'onclick', 'onerror', 'onfinish', 'onmouseover', 'onmouseout', 'onfocus', 'onblur', 'lowsrc', 'dynsrc'];

    /**
     * @var array<string, array<string, string>> Stores the default attributes to add to different tags by add_attributes().
     * @see SimplePie::add_attributes()
     * @access private
     */
    public $add_attributes = ['audio' => ['preload' => 'none'], 'iframe' => ['sandbox' => 'allow-scripts allow-same-origin'], 'video' => ['preload' => 'none']];

    /**
     * @var array<string> Stores the default tags to be stripped by strip_htmltags().
     * @see SimplePie::strip_htmltags()
     * @access private
     */
    public $strip_htmltags = ['base', 'blink', 'body', 'doctype', 'embed', 'font', 'form', 'frame', 'frameset', 'html', 'iframe', 'input', 'marquee', 'meta', 'noscript', 'object', 'param', 'script', 'style'];

    /**
     * @var string[]|string Stores the default attributes to be renamed by rename_attributes().
     * @see SimplePie::rename_attributes()
     * @access private
     */
    public $rename_attributes = [];

    /**
     * @var bool Should we throw exceptions, or use the old-style error property?
     * @access private
     */
    public $enable_exceptions = false;

    /**
     * @var Client|null
     */
    private $http_client = null;

    /** @var bool Whether HTTP client has been injected */
    private $http_client_injected = false;

    /**
     * The SimplePie class contains feed level data and options
     *
     * To use SimplePie, create the SimplePie object with no parameters. You can
     * then set configuration options using the provided methods. After setting
     * them, you must initialise the feed using $feed->init(). At that point the
     * object's methods and properties will be available to you.
     *
     * Previously, it was possible to pass in the feed URL along with cache
     * options directly into the constructor. This has been removed as of 1.3 as
     * it caused a lot of confusion.
     *
     * @since 1.0 Preview Release
     */
    public function __construct()
    {
        if (version_compare(PHP_VERSION, '7.2', '<')) {
            exit('Please upgrade to PHP 7.2 or newer.');
        }

        $this->set_useragent();

        $this->set_cache_namefilter(new CallableNameFilter($this->cache_name_function));

        // Other objects, instances created here so we can set options on them
        $this->sanitize = new Sanitize();
        $this->registry = new Registry();

        if (func_num_args() > 0) {
            trigger_error('Passing parameters to the constructor is no longer supported. Please use set_feed_url(), set_cache_location(), and set_cache_duration() directly.', \E_USER_DEPRECATED);

            $args = func_get_args();
            switch (count($args)) {
                case 3:
                    $this->set_cache_duration($args[2]);
                    // no break
                case 2:
                    $this->set_cache_location($args[1]);
                    // no break
                case 1:
                    $this->set_feed_url($args[0]);
                    $this->init();
            }
        }
    }

    /**
     * Used for converting object to a string
     * @return string
     */
    public function __toString()
    {
        return md5(serialize($this->data));
    }

    /**
     * Remove items that link back to this before destroying this object
     * @return void
     */
    public function __destruct()
    {
        if (!gc_enabled()) {
            if (!empty($this->data['items'])) {
                foreach ($this->data['items'] as $item) {
                    $item->__destruct();
                }
                unset($item, $this->data['items']);
            }
            if (!empty($this->data['ordered_items'])) {
                foreach ($this->data['ordered_items'] as $item) {
                    $item->__destruct();
                }
                unset($item, $this->data['ordered_items']);
            }
        }
    }

    /**
     * Force the given data/URL to be treated as a feed
     *
     * This tells SimplePie to ignore the content-type provided by the server.
     * Be careful when using this option, as it will also disable autodiscovery.
     *
     * @since 1.1
     * @param bool $enable Force the given data/URL to be treated as a feed
     * @return void
     */
    public function force_feed(bool $enable = false)
    {
        $this->force_feed = $enable;
    }

    /**
     * Set the URL of the feed you want to parse
     *
     * This allows you to enter the URL of the feed you want to parse, or the
     * website you want to try to use auto-discovery on. This takes priority
     * over any set raw data.
     *
     * Deprecated since 1.9.0: You can set multiple feeds to mash together by passing an array instead
     * of a string for the $url. Remember that with each additional feed comes
     * additional processing and resources.
     *
     * @since 1.0 Preview Release
     * @see set_raw_data()
     * @param string|string[] $url This is the URL (or (deprecated) array of URLs) that you want to parse.
     * @return void
     */
    public function set_feed_url($url)
    {
        $this->multifeed_url = [];
        if (is_array($url)) {
            trigger_error('Fetching multiple feeds with single SimplePie instance is deprecated since SimplePie 1.9.0, create one SimplePie instance per feed and use SimplePie::merge_items to get a single list of items.', \E_USER_DEPRECATED);
            foreach ($url as $value) {
                $this->multifeed_url[] = $this->registry->call(Misc::class, 'fix_protocol', [$value, 1]);
            }
        } else {
            $this->feed_url = $this->registry->call(Misc::class, 'fix_protocol', [$url, 1]);
            $this->permanent_url = $this->feed_url;
        }
    }

    /**
     * Set an instance of {@see File} to use as a feed
     *
     * @deprecated since SimplePie 1.9.0, use \SimplePie\SimplePie::set_http_client() or \SimplePie\SimplePie::set_raw_data() instead.
     *
     * @param File &$file
     * @return bool True on success, false on failure
     */
    public function set_file(File &$file)
    {
        // trigger_error(sprintf('SimplePie\SimplePie::set_file() is deprecated since SimplePie 1.9.0, please use "SimplePie\SimplePie::set_http_client()" or "SimplePie\SimplePie::set_raw_data()" instead.'), \E_USER_DEPRECATED);

        $this->feed_url = $file->get_final_requested_uri();
        $this->permanent_url = $this->feed_url;
        $this->file = &$file;

        return true;
    }

    /**
     * Set the raw XML data to parse
     *
     * Allows you to use a string of RSS/Atom data instead of a remote feed.
     *
     * If you have a feed available as a string in PHP, you can tell SimplePie
     * to parse that data string instead of a remote feed. Any set feed URL
     * takes precedence.
     *
     * @since 1.0 Beta 3
     * @param string $data RSS or Atom data as a string.
     * @see set_feed_url()
     * @return void
     */
    public function set_raw_data(string $data)
    {
        $this->raw_data = $data;
    }

    /**
     * Set a PSR-18 client and PSR-17 factories
     *
     * Allows you to use your own HTTP client implementations.
     * This will become required with SimplePie 2.0.0.
     */
    final public function set_http_client(
        ClientInterface $http_client,
        RequestFactoryInterface $request_factory,
        UriFactoryInterface $uri_factory
    ): void {
        $this->http_client = new Psr18Client($http_client, $request_factory, $uri_factory);
    }

    /**
     * Set the default timeout for fetching remote feeds
     *
     * This allows you to change the maximum time the feed's server to respond
     * and send the feed back.
     *
     * @since 1.0 Beta 3
     * @param int $timeout The maximum number of seconds to spend waiting to retrieve a feed.
     * @return void
     */
    public function set_timeout(int $timeout = 10)
    {
        if ($this->http_client_injected) {
            throw new SimplePieException(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure timeout in your HTTP client instead.',
                __METHOD__,
                self::class
            ));
        }

        $this->timeout = (int) $timeout;

        // Reset a possible existing FileClient,
        // so a new client with the changed value will be created
        if (is_object($this->http_client) && $this->http_client instanceof FileClient) {
            $this->http_client = null;
        } elseif (is_object($this->http_client)) {
            // Trigger notice if a PSR-18 client was set
            trigger_error(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure the timeout in your HTTP client instead.',
                __METHOD__,
                get_class($this)
            ), \E_USER_NOTICE);
        }
    }

    /**
     * Set custom curl options
     *
     * This allows you to change default curl options
     *
     * @since 1.0 Beta 3
     * @param array<int, mixed> $curl_options Curl options to add to default settings
     * @return void
     */
    public function set_curl_options(array $curl_options = [])
    {
        if ($this->http_client_injected) {
            throw new SimplePieException(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure custom curl options in your HTTP client instead.',
                __METHOD__,
                self::class
            ));
        }

        $this->curl_options = $curl_options;

        // Reset a possible existing FileClient,
        // so a new client with the changed value will be created
        if (is_object($this->http_client) && $this->http_client instanceof FileClient) {
            $this->http_client = null;
        } elseif (is_object($this->http_client)) {
            // Trigger notice if a PSR-18 client was set
            trigger_error(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure the curl options in your HTTP client instead.',
                __METHOD__,
                get_class($this)
            ), \E_USER_NOTICE);
        }
    }

    /**
     * Force SimplePie to use fsockopen() instead of cURL
     *
     * @since 1.0 Beta 3
     * @param bool $enable Force fsockopen() to be used
     * @return void
     */
    public function force_fsockopen(bool $enable = false)
    {
        if ($this->http_client_injected) {
            throw new SimplePieException(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure fsockopen in your HTTP client instead.',
                __METHOD__,
                self::class
            ));
        }

        $this->force_fsockopen = $enable;

        // Reset a possible existing FileClient,
        // so a new client with the changed value will be created
        if (is_object($this->http_client) && $this->http_client instanceof FileClient) {
            $this->http_client = null;
        } elseif (is_object($this->http_client)) {
            // Trigger notice if a PSR-18 client was set
            trigger_error(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure fsockopen in your HTTP client instead.',
                __METHOD__,
                get_class($this)
            ), \E_USER_NOTICE);
        }
    }

    /**
     * Enable/disable caching in SimplePie.
     *
     * This option allows you to disable caching all-together in SimplePie.
     * However, disabling the cache can lead to longer load times.
     *
     * @since 1.0 Preview Release
     * @param bool $enable Enable caching
     * @return void
     */
    public function enable_cache(bool $enable = true)
    {
        $this->enable_cache = $enable;
    }

    /**
     * Set a PSR-16 implementation as cache
     *
     * @param CacheInterface $cache The PSR-16 cache implementation
     *
     * @return void
     */
    public function set_cache(CacheInterface $cache)
    {
        $this->cache = new Psr16($cache);
    }

    /**
     * SimplePie to continue to fall back to expired cache, if enabled, when
     * feed is unavailable.
     *
     * This tells SimplePie to ignore any file errors and fall back to cache
     * instead. This only works if caching is enabled and cached content
     * still exists.
     *
     * @deprecated since SimplePie 1.8.0, expired cache will not be used anymore.
     *
     * @param bool $enable Force use of cache on fail.
     * @return void
     */
    public function force_cache_fallback(bool $enable = false)
    {
        // @trigger_error(sprintf('SimplePie\SimplePie::force_cache_fallback() is deprecated since SimplePie 1.8.0, expired cache will not be used anymore.'), \E_USER_DEPRECATED);
        $this->force_cache_fallback = $enable;
    }

    /**
     * Set the length of time (in seconds) that the contents of a feed will be
     * cached
     *
     * @param int $seconds The feed content cache duration
     * @return void
     */
    public function set_cache_duration(int $seconds = 3600)
    {
        $this->cache_duration = $seconds;
    }

    /**
     * Set the length of time (in seconds) that the autodiscovered feed URL will
     * be cached
     *
     * @param int $seconds The autodiscovered feed URL cache duration.
     * @return void
     */
    public function set_autodiscovery_cache_duration(int $seconds = 604800)
    {
        $this->autodiscovery_cache_duration = $seconds;
    }

    /**
     * Set the file system location where the cached files should be stored
     *
     * @deprecated since SimplePie 1.8.0, use SimplePie::set_cache() instead.
     *
     * @param string $location The file system location.
     * @return void
     */
    public function set_cache_location(string $location = './cache')
    {
        // @trigger_error(sprintf('SimplePie\SimplePie::set_cache_location() is deprecated since SimplePie 1.8.0, please use "SimplePie\SimplePie::set_cache()" instead.'), \E_USER_DEPRECATED);
        $this->cache_location = $location;
    }

    /**
     * Return the filename (i.e. hash, without path and without extension) of the file to cache a given URL.
     *
     * @param string $url The URL of the feed to be cached.
     * @return string A filename (i.e. hash, without path and without extension).
     */
    public function get_cache_filename(string $url)
    {
        // Append custom parameters to the URL to avoid cache pollution in case of multiple calls with different parameters.
        $url .= $this->force_feed ? '#force_feed' : '';
        $options = [];
        if ($this->timeout != 10) {
            $options[CURLOPT_TIMEOUT] = $this->timeout;
        }
        if ($this->useragent !== Misc::get_default_useragent()) {
            $options[CURLOPT_USERAGENT] = $this->useragent;
        }
        if (!empty($this->curl_options)) {
            foreach ($this->curl_options as $k => $v) {
                $options[$k] = $v;
            }
        }
        if (!empty($options)) {
            ksort($options);
            $url .= '#' . urlencode(var_export($options, true));
        }

        return $this->cache_namefilter->filter($url);
    }

    /**
     * Set whether feed items should be sorted into reverse chronological order
     *
     * @param bool $enable Sort as reverse chronological order.
     * @return void
     */
    public function enable_order_by_date(bool $enable = true)
    {
        $this->order_by_date = $enable;
    }

    /**
     * Set the character encoding used to parse the feed
     *
     * This overrides the encoding reported by the feed, however it will fall
     * back to the normal encoding detection if the override fails
     *
     * @param string|false $encoding Character encoding
     * @return void
     */
    public function set_input_encoding($encoding = false)
    {
        if ($encoding) {
            $this->input_encoding = (string) $encoding;
        } else {
            $this->input_encoding = false;
        }
    }

    /**
     * Set how much feed autodiscovery to do
     *
     * @see self::LOCATOR_NONE
     * @see self::LOCATOR_AUTODISCOVERY
     * @see self::LOCATOR_LOCAL_EXTENSION
     * @see self::LOCATOR_LOCAL_BODY
     * @see self::LOCATOR_REMOTE_EXTENSION
     * @see self::LOCATOR_REMOTE_BODY
     * @see self::LOCATOR_ALL
     * @param self::LOCATOR_* $level Feed Autodiscovery Level (level can be a combination of the above constants, see bitwise OR operator)
     * @return void
     */
    public function set_autodiscovery_level(int $level = self::LOCATOR_ALL)
    {
        $this->autodiscovery = $level;
    }

    /**
     * Get the class registry
     *
     * Use this to override SimplePie's default classes
     *
     * @return Registry
     */
    public function &get_registry()
    {
        return $this->registry;
    }

    /**
     * Set which class SimplePie uses for caching
     *
     * @deprecated since SimplePie 1.3, use {@see set_cache()} instead
     *
     * @param class-string<Cache> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_cache_class(string $class = Cache::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::set_cache()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Cache::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for auto-discovery
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Locator> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_locator_class(string $class = Locator::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Locator::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for XML parsing
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Parser> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_parser_class(string $class = Parser::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Parser::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for remote file fetching
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<File> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_file_class(string $class = File::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(File::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for data sanitization
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Sanitize> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_sanitize_class(string $class = Sanitize::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Sanitize::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for handling feed items
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Item> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_item_class(string $class = Item::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Item::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for handling author data
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Author> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_author_class(string $class = Author::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Author::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for handling category data
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Category> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_category_class(string $class = Category::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Category::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for feed enclosures
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Enclosure> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_enclosure_class(string $class = Enclosure::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Enclosure::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for `<media:text>` captions
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Caption> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_caption_class(string $class = Caption::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Caption::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for `<media:copyright>`
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Copyright> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_copyright_class(string $class = Copyright::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Copyright::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for `<media:credit>`
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Credit> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_credit_class(string $class = Credit::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Credit::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for `<media:rating>`
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Rating> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_rating_class(string $class = Rating::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Rating::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for `<media:restriction>`
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Restriction> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_restriction_class(string $class = Restriction::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Restriction::class, $class, true);
    }

    /**
     * Set which class SimplePie uses for content-type sniffing
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Sniffer> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_content_type_sniffer_class(string $class = Sniffer::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Sniffer::class, $class, true);
    }

    /**
     * Set which class SimplePie uses item sources
     *
     * @deprecated since SimplePie 1.3, use {@see get_registry()} instead
     *
     * @param class-string<Source> $class Name of custom class
     *
     * @return bool True on success, false otherwise
     */
    public function set_source_class(string $class = Source::class)
    {
        trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.3, please use "SimplePie\SimplePie::get_registry()" instead.', __METHOD__), \E_USER_DEPRECATED);

        return $this->registry->register(Source::class, $class, true);
    }

    /**
     * Set the user agent string
     *
     * @param string $ua New user agent string.
     * @return void
     */
    public function set_useragent(?string $ua = null)
    {
        if ($this->http_client_injected) {
            throw new SimplePieException(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure user agent string in your HTTP client instead.',
                __METHOD__,
                self::class
            ));
        }

        if ($ua === null) {
            $ua = Misc::get_default_useragent();
        }

        $this->useragent = (string) $ua;

        // Reset a possible existing FileClient,
        // so a new client with the changed value will be created
        if (is_object($this->http_client) && $this->http_client instanceof FileClient) {
            $this->http_client = null;
        } elseif (is_object($this->http_client)) {
            // Trigger notice if a PSR-18 client was set
            trigger_error(sprintf(
                'Using "%s()" has no effect, because you already provided a HTTP client with "%s::set_http_client()". Configure the useragent in your HTTP client instead.',
                __METHOD__,
                get_class($this)
            ), \E_USER_NOTICE);
        }
    }

    /**
     * Set a namefilter to modify the cache filename with
     *
     * @param NameFilter $filter
     *
     * @return void
     */
    public function set_cache_namefilter(NameFilter $filter): void
    {
        $this->cache_namefilter = $filter;
    }

    /**
     * Set callback function to create cache filename with
     *
     * @deprecated since SimplePie 1.8.0, use {@see set_cache_namefilter()} instead
     *
     * @param (string&(callable(string): string))|null $function Callback function
     * @return void
     */
    public function set_cache_name_function(?string $function = null)
    {
        // trigger_error(sprintf('"%s()" is deprecated since SimplePie 1.8.0, please use "SimplePie\SimplePie::set_cache_namefilter()" instead.', __METHOD__), \E_USER_DEPRECATED);

        if ($function === null) {
            $function = 'md5';
        }

        $this->cache_name_function = $function;

        $this->set_cache_namefilter(new CallableNameFilter($this->cache_name_function));
    }

    /**
     * Set options to make SP as fast as possible
     *
     * Forgoes a substantial amount of data sanitization in favor of speed. This
     * turns SimplePie into a dumb parser of feeds.
     *
     * @param bool $set Whether to set them or not
     * @return void
     */
    public function set_stupidly_fast(bool $set = false)
    {
        if ($set) {
            $this->enable_order_by_date(false);
            $this->remove_div(false);
            $this->strip_comments(false);
            $this->strip_htmltags([]);
            $this->strip_attributes([]);
            $this->add_attributes([]);
            $this->set_image_handler(false);
            $this->set_https_domains([]);
        }
    }

    /**
     * Set maximum number of feeds to check with autodiscovery
     *
     * @param int $max Maximum number of feeds to check
     * @return void
     */
    public function set_max_checked_feeds(int $max = 10)
    {
        $this->max_checked_feeds = $max;
    }

    /**
     * @return void
     */
    public function remove_div(bool $enable = true)
    {
        $this->sanitize->remove_div($enable);
    }

    /**
     * @param string[]|string|false $tags Set a list of tags to strip, or set empty string to use default tags, or false to strip nothing.
     * @return void
     */
    public function strip_htmltags($tags = '', ?bool $encode = null)
    {
        if ($tags === '') {
            $tags = $this->strip_htmltags;
        }
        $this->sanitize->strip_htmltags($tags);
        if ($encode !== null) {
            $this->sanitize->encode_instead_of_strip($encode);
        }
    }

    /**
     * @return void
     */
    public function encode_instead_of_strip(bool $enable = true)
    {
        $this->sanitize->encode_instead_of_strip($enable);
    }

    /**
     * @param string[]|string $attribs
     * @return void
     */
    public function rename_attributes($attribs = '')
    {
        if ($attribs === '') {
            $attribs = $this->rename_attributes;
        }
        $this->sanitize->rename_attributes($attribs);
    }

    /**
     * @param string[]|string $attribs
     * @return void
     */
    public function strip_attributes($attribs = '')
    {
        if ($attribs === '') {
            $attribs = $this->strip_attributes;
        }
        $this->sanitize->strip_attributes($attribs);
    }

    /**
     * @param array<string, array<string, string>>|'' $attribs
     * @return void
     */
    public function add_attributes($attribs = '')
    {
        if ($attribs === '') {
            $attribs = $this->add_attributes;
        }
        $this->sanitize->add_attributes($attribs);
    }

    /**
     * Set the output encoding
     *
     * Allows you to override SimplePie's output to match that of your webpage.
     * This is useful for times when your webpages are not being served as
     * UTF-8. This setting will be obeyed by {@see handle_content_type()}, and
     * is similar to {@see set_input_encoding()}.
     *
     * It should be noted, however, that not all character encodings can support
     * all characters. If your page is being served as ISO-8859-1 and you try
     * to display a Japanese feed, you'll likely see garbled characters.
     * Because of this, it is highly recommended to ensure that your webpages
     * are served as UTF-8.
     *
     * The number of supported character encodings depends on whether your web
     * host supports {@link http://php.net/mbstring mbstring},
     * {@link http://php.net/iconv iconv}, or both. See
     * {@link http://simplepie.org/wiki/faq/Supported_Character_Encodings} for
     * more information.
     *
     * @param string $encoding
     * @return void
     */
    public function set_output_encoding(string $encoding = 'UTF-8')
    {
        $this->sanitize->set_output_encoding($encoding);
    }

    /**
     * @return void
     */
    public function strip_comments(bool $strip = false)
    {
        $this->sanitize->strip_comments($strip);
    }

    /**
     * Set element/attribute key/value pairs of HTML attributes
     * containing URLs that need to be resolved relative to the feed
     *
     * Defaults to |a|@href, |area|@href, |blockquote|@cite, |del|@cite,
     * |form|@action, |img|@longdesc, |img|@src, |input|@src, |ins|@cite,
     * |q|@cite
     *
     * @since 1.0
     * @param array<string, string|string[]>|null $element_attribute Element/attribute key/value pairs, null for default
     * @return void
     */
    public function set_url_replacements(?array $element_attribute = null)
    {
        $this->sanitize->set_url_replacements($element_attribute);
    }

    /**
     * Set the list of domains for which to force HTTPS.
     * @see Sanitize::set_https_domains()
     * @param array<string> $domains List of HTTPS domains. Example array('biz', 'example.com', 'example.org', 'www.example.net').
     * @return void
     */
    public function set_https_domains(array $domains = [])
    {
        $this->sanitize->set_https_domains($domains);
    }

    /**
     * Set the handler to enable the display of cached images.
     *
     * @param string|false $page Web-accessible path to the handler_image.php file.
     * @param string $qs The query string that the value should be passed to.
     * @return void
     */
    public function set_image_handler($page = false, string $qs = 'i')
    {
        if ($page !== false) {
            $this->sanitize->set_image_handler($page . '?' . $qs . '=');
        } else {
            $this->image_handler = '';
        }
    }

    /**
     * Set the limit for items returned per-feed with multifeeds
     *
     * @param int $limit The maximum number of items to return.
     * @return void
     */
    public function set_item_limit(int $limit = 0)
    {
        $this->item_limit = $limit;
    }

    /**
     * Enable throwing exceptions
     *
     * @param bool $enable Should we throw exceptions, or use the old-style error property?
     * @return void
     */
    public function enable_exceptions(bool $enable = true)
    {
        $this->enable_exceptions = $enable;
    }

    /**
     * Initialize the feed object
     *
     * This is what makes everything happen. Period. This is where all of the
     * configuration options get processed, feeds are fetched, cached, and
     * parsed, and all of that other good stuff.
     *
     * @return bool True if successful, false otherwise
     */
    public function init()
    {
        // Check absolute bare minimum requirements.
        if (!extension_loaded('xml') || !extension_loaded('pcre')) {
            $this->error = 'XML or PCRE extensions not loaded!';
            return false;
        }
        // Then check the xml extension is sane (i.e., libxml 2.7.x issue on PHP < 5.2.9 and libxml 2.7.0 to 2.7.2 on any version) if we don't have xmlreader.
        elseif (!extension_loaded('xmlreader')) {
            static $xml_is_sane = null;
            if ($xml_is_sane === null) {
                $parser_check = xml_parser_create();
                xml_parse_into_struct($parser_check, '<foo>&amp;</foo>', $values);
                if (\PHP_VERSION_ID < 80000) {
                    xml_parser_free($parser_check);
                }
                $xml_is_sane = isset($values[0]['value']);
            }
            if (!$xml_is_sane) {
                return false;
            }
        }

        // The default sanitize class gets set in the constructor, check if it has
        // changed.
        if ($this->registry->get_class(Sanitize::class) !== Sanitize::class) {
            $this->sanitize = $this->registry->create(Sanitize::class);
        }
        if (method_exists($this->sanitize, 'set_registry')) {
            $this->sanitize->set_registry($this->registry);
        }

        // Pass whatever was set with config options over to the sanitizer.
        // Pass the classes in for legacy support; new classes should use the registry instead
        $cache = $this->registry->get_class(Cache::class);
        \assert($cache !== null, 'Cache must be defined');
        $this->sanitize->pass_cache_data(
            $this->enable_cache,
            $this->cache_location,
            $this->cache_namefilter,
            $cache,
            $this->cache
        );

        $http_client = $this->get_http_client();

        if ($http_client instanceof Psr18Client) {
            $this->sanitize->set_http_client(
                $http_client->getHttpClient(),
                $http_client->getRequestFactory(),
                $http_client->getUriFactory()
            );
        }

        if (!empty($this->multifeed_url)) {
            $i = 0;
            $success = 0;
            $this->multifeed_objects = [];
            $this->error = [];
            foreach ($this->multifeed_url as $url) {
                $this->multifeed_objects[$i] = clone $this;
                $this->multifeed_objects[$i]->set_feed_url($url);
                $single_success = $this->multifeed_objects[$i]->init();
                $success |= $single_success;
                if (!$single_success) {
                    $this->error[$i] = $this->multifeed_objects[$i]->error();
                }
                $i++;
            }
            return (bool) $success;
        } elseif ($this->feed_url === null && $this->raw_data === null) {
            return false;
        }

        $this->error = null;
        $this->data = [];
        $this->check_modified = false;
        $this->multifeed_objects = [];
        $cache = false;

        if ($this->feed_url !== null) {
            $parsed_feed_url = $this->registry->call(Misc::class, 'parse_url', [$this->feed_url]);

            // Decide whether to enable caching
            if ($this->enable_cache && $parsed_feed_url['scheme'] !== '') {
                $cache = $this->get_cache($this->feed_url);
            }

            // Fetch the data into $this->raw_data
            if (($fetched = $this->fetch_data($cache)) === true) {
                return true;
            } elseif ($fetched === false) {
                return false;
            }

            [$headers, $sniffed] = $fetched;
        }

        // Empty response check
        if (empty($this->raw_data)) {
            $this->error = "A feed could not be found at `$this->feed_url`. Empty body.";
            $this->registry->call(Misc::class, 'error', [$this->error, E_USER_NOTICE, __FILE__, __LINE__]);
            return false;
        }

        // Set up array of possible encodings
        $encodings = [];

        // First check to see if input has been overridden.
        if ($this->input_encoding !== false) {
            $encodings[] = strtoupper($this->input_encoding);
        }

        $application_types = ['application/xml', 'application/xml-dtd', 'application/xml-external-parsed-entity'];
        $text_types = ['text/xml', 'text/xml-external-parsed-entity'];

        // RFC 3023 (only applies to sniffed content)
        if (isset($sniffed)) {
            if (in_array($sniffed, $application_types) || substr($sniffed, 0, 12) === 'application/' && substr($sniffed, -4) === '+xml') {
                if (isset($headers['content-type']) && preg_match('/;\x20?charset=([^;]*)/i', $headers['content-type'], $charset)) {
                    $encodings[] = strtoupper($charset[1]);
                }
                $encodings = array_merge($encodings, $this->registry->call(Misc::class, 'xml_encoding', [$this->raw_data, &$this->registry]));
                $encodings[] = 'UTF-8';
            } elseif (in_array($sniffed, $text_types) || substr($sniffed, 0, 5) === 'text/' && substr($sniffed, -4) === '+xml') {
                if (isset($headers['content-type']) && preg_match('/;\x20?charset=([^;]*)/i', $headers['content-type'], $charset)) {
                    $encodings[] = strtoupper($charset[1]);
                }
                $encodings[] = 'US-ASCII';
            }
            // Text MIME-type default
            elseif (substr($sniffed, 0, 5) === 'text/') {
                $encodings[] = 'UTF-8';
            }
        }

        // Fallback to XML 1.0 Appendix F.1/UTF-8/ISO-8859-1
        $encodings = array_merge($encodings, $this->registry->call(Misc::class, 'xml_encoding', [$this->raw_data, &$this->registry]));
        $encodings[] = 'UTF-8';
        $encodings[] = 'ISO-8859-1';

        // There's no point in trying an encoding twice
        $encodings = array_unique($encodings);

        // Loop through each possible encoding, till we return something, or run out of possibilities
        foreach ($encodings as $encoding) {
            // Change the encoding to UTF-8 (as we always use UTF-8 internally)
            if ($utf8_data = $this->registry->call(Misc::class, 'change_encoding', [$this->raw_data, $encoding, 'UTF-8'])) {
                // Create new parser
                $parser = $this->registry->create(Parser::class);

                // If it's parsed fine
                if ($parser->parse($utf8_data, 'UTF-8', $this->permanent_url ?? '')) {
                    $this->data = $parser->get_data();
                    if (!($this->get_type() & ~self::TYPE_NONE)) {
                        $this->error = "A feed could not be found at `$this->feed_url`. This does not appear to be a valid RSS or Atom feed.";
                        $this->registry->call(Misc::class, 'error', [$this->error, E_USER_NOTICE, __FILE__, __LINE__]);
                        return false;
                    }

                    if (isset($headers)) {
                        $this->data['headers'] = $headers;
                    }
                    $this->data['build'] = Misc::get_build();

                    // Cache the file if caching is enabled
                    $this->data['cache_expiration_time'] = $this->cache_duration + time();

                    if ($cache && !$cache->set_data($this->get_cache_filename($this->feed_url), $this->data, $this->cache_duration)) {
                        trigger_error("$this->cache_location is not writable. Make sure you've set the correct relative or absolute path, and that the location is server-writable.", E_USER_WARNING);
                    }
                    return true;
                }
            }
        }

        if (isset($parser)) {
            // We have an error, just set Misc::error to it and quit
            $this->error = $this->feed_url;
            $this->error .= sprintf(' is invalid XML, likely due to invalid characters. XML error: %s at line %d, column %d', $parser->get_error_string(), $parser->get_current_line(), $parser->get_current_column());
        } else {
            $this->error = 'The data could not be converted to UTF-8.';
            if (!extension_loaded('mbstring') && !extension_loaded('iconv') && !class_exists('\UConverter')) {
                $this->error .= ' You MUST have either the iconv, mbstring or intl (PHP 5.5+) extension installed and enabled.';
            } else {
                $missingExtensions = [];
                if (!extension_loaded('iconv')) {
                    $missingExtensions[] = 'iconv';
                }
                if (!extension_loaded('mbstring')) {
                    $missingExtensions[] = 'mbstring';
                }
                if (!class_exists('\UConverter')) {
                    $missingExtensions[] = 'intl (PHP 5.5+)';
                }
                $this->error .= ' Try installing/enabling the ' . implode(' or ', $missingExtensions) . ' extension.';
            }
        }

        $this->registry->call(Misc::class, 'error', [$this->error, E_USER_NOTICE, __FILE__, __LINE__]);

        return false;
    }

    /**
     * Fetch the data
     *
     * If the data is already cached, attempt to fetch it from there instead
     *
     * @param Base|DataCache|false $cache Cache handler, or false to not load from the cache
     * @return array{array<string, string>, string}|bool Returns true if the data was loaded from the cache, or an array of HTTP headers and sniffed type
     */
    protected function fetch_data(&$cache)
    {
        if ($cache instanceof Base) {
            // @trigger_error(sprintf('Providing $cache as "\SimplePie\Cache\Base" in %s() is deprecated since SimplePie 1.8.0, please provide "\SimplePie\Cache\DataCache" implementation instead.', __METHOD__), \E_USER_DEPRECATED);
            $cache = new BaseDataCache($cache);
        }

        // @phpstan-ignore-next-line Enforce PHPDoc type.
        if ($cache !== false && !$cache instanceof DataCache) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($cache) must be of type %s|false',
                __METHOD__,
                DataCache::class
            ), 1);
        }

        $cacheKey = $this->get_cache_filename($this->feed_url);

        // If it's enabled, use the cache
        if ($cache) {
            // Load the Cache
            $this->data = $cache->get_data($cacheKey, []);

            if (!empty($this->data)) {
                // If the cache is for an outdated build of SimplePie
                if (!isset($this->data['build']) || $this->data['build'] !== Misc::get_build()) {
                    $cache->delete_data($cacheKey);
                    $this->data = [];
                }
                // If we've hit a collision just rerun it with caching disabled
                elseif (isset($this->data['url']) && $this->data['url'] !== $this->feed_url) {
                    $cache = false;
                    $this->data = [];
                }
                // If we've got a non feed_url stored (if the page isn't actually a feed, or is a redirect) use that URL.
                elseif (isset($this->data['feed_url'])) {
                    // Do not need to do feed autodiscovery yet.
                    if ($this->data['feed_url'] !== $this->data['url']) {
                        $this->set_feed_url($this->data['feed_url']);
                        $this->data['url'] = $this->data['feed_url'];

                        $cache->set_data($this->get_cache_filename($this->feed_url), $this->data, $this->autodiscovery_cache_duration);

                        return $this->init();
                    }

                    $cache->delete_data($this->get_cache_filename($this->feed_url));
                    $this->data = [];
                }
                // Check if the cache has been updated
                elseif (!isset($this->data['cache_expiration_time']) || $this->data['cache_expiration_time'] < time()) {
                    // Want to know if we tried to send last-modified and/or etag headers
                    // when requesting this file. (Note that it's up to the file to
                    // support this, but we don't always send the headers either.)
                    $this->check_modified = true;
                    if (isset($this->data['headers']['last-modified']) || isset($this->data['headers']['etag'])) {
                        $headers = [
                            'Accept' => SimplePie::DEFAULT_HTTP_ACCEPT_HEADER,
                        ];
                        if (isset($this->data['headers']['last-modified'])) {
                            $headers['if-modified-since'] = $this->data['headers']['last-modified'];
                        }
                        if (isset($this->data['headers']['etag'])) {
                            $headers['if-none-match'] = $this->data['headers']['etag'];
                        }

                        try {
                            $file = $this->get_http_client()->request(Client::METHOD_GET, $this->feed_url, $headers);
                            $this->status_code = $file->get_status_code();
                        } catch (ClientException $th) {
                            $this->check_modified = false;
                            $this->status_code = 0;

                            if ($this->force_cache_fallback) {
                                $this->data['cache_expiration_time'] = $this->cache_duration + time();
                                $cache->set_data($cacheKey, $this->data, $this->cache_duration);

                                return true;
                            }

                            $failedFileReason = $th->getMessage();
                        }

                        if ($this->status_code === 304) {
                            // Set raw_data to false here too, to signify that the cache
                            // is still valid.
                            $this->raw_data = false;
                            $this->data['cache_expiration_time'] = $this->cache_duration + time();
                            $cache->set_data($cacheKey, $this->data, $this->cache_duration);

                            return true;
                        }
                    }
                }
                // If the cache is still valid, just return true
                else {
                    $this->raw_data = false;
                    return true;
                }
            }
            // If the cache is empty
            else {
                $this->data = [];
            }
        }

        // If we don't already have the file (it'll only exist if we've opened it to check if the cache has been modified), open it.
        if (!isset($file)) {
            if ($this->file instanceof File && $this->file->get_final_requested_uri() === $this->feed_url) {
                $file = &$this->file;
            } elseif (isset($failedFileReason)) {
                // Do not try to fetch again if we already failed once.
                // If the file connection had an error, set SimplePie::error to that and quit
                $this->error = $failedFileReason;

                return !empty($this->data);
            } else {
                $headers = [
                    'Accept' => SimplePie::DEFAULT_HTTP_ACCEPT_HEADER,
                ];
                try {
                    $file = $this->get_http_client()->request(Client::METHOD_GET, $this->feed_url, $headers);
                } catch (ClientException $th) {
                    // If the file connection has an error, set SimplePie::error to that and quit
                    $this->error = $th->getMessage();

                    return !empty($this->data);
                }
            }
        }
        $this->status_code = $file->get_status_code();

        // If the file connection has an error, set SimplePie::error to that and quit
        if (!(!Misc::is_remote_uri($file->get_final_requested_uri()) || ($file->get_status_code() === 200 || $file->get_status_code() > 206 && $file->get_status_code() < 300))) {
            $this->error = 'Retrieved unsupported status code "' . $this->status_code . '"';
            return !empty($this->data);
        }

        if (!$this->force_feed) {
            // Check if the supplied URL is a feed, if it isn't, look for it.
            $locate = $this->registry->create(Locator::class, [
                (!$file instanceof File) ? File::fromResponse($file) : $file,
                $this->timeout,
                $this->useragent,
                $this->max_checked_feeds,
                $this->force_fsockopen,
                $this->curl_options
            ]);

            $http_client = $this->get_http_client();

            if ($http_client instanceof Psr18Client) {
                $locate->set_http_client(
                    $http_client->getHttpClient(),
                    $http_client->getRequestFactory(),
                    $http_client->getUriFactory()
                );
            }

            if (!$locate->is_feed($file)) {
                $copyStatusCode = $file->get_status_code();
                $copyContentType = $file->get_header_line('content-type');
                try {
                    $microformats = false;
                    if (class_exists('DOMXpath') && function_exists('Mf2\parse')) {
                        $doc = new \DOMDocument();
                        @$doc->loadHTML($file->get_body_content());
                        $xpath = new \DOMXpath($doc);
                        // Check for both h-feed and h-entry, as both a feed with no entries
                        // and a list of entries without an h-feed wrapper are both valid.
                        $query = '//*[contains(concat(" ", @class, " "), " h-feed ") or '.
                            'contains(concat(" ", @class, " "), " h-entry ")]';

                        /** @var \DOMNodeList<\DOMElement> $result */
                        $result = $xpath->query($query);
                        $microformats = $result->length !== 0;
                    }
                    // Now also do feed discovery, but if microformats were found don't
                    // overwrite the current value of file.
                    $discovered = $locate->find(
                        $this->autodiscovery,
                        $this->all_discovered_feeds
                    );
                    if ($microformats) {
                        $hub = $locate->get_rel_link('hub');
                        $self = $locate->get_rel_link('self');
                        if ($hub || $self) {
                            $file = $this->store_links($file, $hub, $self);
                        }
                        // Push the current file onto all_discovered feeds so the user can
                        // be shown this as one of the options.
                        if ($this->all_discovered_feeds !== null) {
                            $this->all_discovered_feeds[] = $file;
                        }
                    } else {
                        if ($discovered) {
                            $file = $discovered;
                        } else {
                            // We need to unset this so that if SimplePie::set_file() has
                            // been called that object is untouched
                            unset($file);
                            $this->error = "A feed could not be found at `$this->feed_url`; the status code is `$copyStatusCode` and content-type is `$copyContentType`";
                            $this->registry->call(Misc::class, 'error', [$this->error, E_USER_NOTICE, __FILE__, __LINE__]);
                            return false;
                        }
                    }
                } catch (SimplePieException $e) {
                    // We need to unset this so that if SimplePie::set_file() has been called that object is untouched
                    unset($file);
                    // This is usually because DOMDocument doesn't exist
                    $this->error = $e->getMessage();
                    $this->registry->call(Misc::class, 'error', [$this->error, E_USER_NOTICE, $e->getFile(), $e->getLine()]);
                    return false;
                }

                if ($cache) {
                    $this->data = [
                        'url' => $this->feed_url,
                        'feed_url' => $file->get_final_requested_uri(),
                        'build' => Misc::get_build(),
                        'cache_expiration_time' => $this->cache_duration + time(),
                    ];

                    if (!$cache->set_data($cacheKey, $this->data, $this->cache_duration)) {
                        trigger_error("$this->cache_location is not writable. Make sure you've set the correct relative or absolute path, and that the location is server-writable.", E_USER_WARNING);
                    }
                }
            }
            $this->feed_url = $file->get_final_requested_uri();
            $locate = null;
        }

        $this->raw_data = $file->get_body_content();
        $this->permanent_url = $file->get_permanent_uri();

        $headers = [];
        foreach ($file->get_headers() as $key => $values) {
            $headers[$key] = implode(', ', $values);
        }

        $sniffer = $this->registry->create(Sniffer::class, [&$file]);
        $sniffed = $sniffer->get_type();

        return [$headers, $sniffed];
    }

    /**
     * Get the error message for the occurred error
     *
     * @return string|string[]|null Error message, or array of messages for multifeeds
     */
    public function error()
    {
        return $this->error;
    }

    /**
     * Get the last HTTP status code
     *
     * @return int Status code
     */
    public function status_code()
    {
        return $this->status_code;
    }

    /**
     * Get the raw XML
     *
     * This is the same as the old `$feed->enable_xml_dump(true)`, but returns
     * the data instead of printing it.
     *
     * @return string|false Raw XML data, false if the cache is used
     */
    public function get_raw_data()
    {
        return $this->raw_data;
    }

    /**
     * Get the character encoding used for output
     *
     * @since Preview Release
     * @return string
     */
    public function get_encoding()
    {
        return $this->sanitize->output_encoding;
    }

    /**
     * Send the content-type header with correct encoding
     *
     * This method ensures that the SimplePie-enabled page is being served with
     * the correct {@link http://www.iana.org/assignments/media-types/ mime-type}
     * and character encoding HTTP headers (character encoding determined by the
     * {@see set_output_encoding} config option).
     *
     * This won't work properly if any content or whitespace has already been
     * sent to the browser, because it relies on PHP's
     * {@link http://php.net/header header()} function, and these are the
     * circumstances under which the function works.
     *
     * Because it's setting these settings for the entire page (as is the nature
     * of HTTP headers), this should only be used once per page (again, at the
     * top).
     *
     * @param string $mime MIME type to serve the page as
     * @return void
     */
    public function handle_content_type(string $mime = 'text/html')
    {
        if (!headers_sent()) {
            $header = "Content-type: $mime;";
            if ($this->get_encoding()) {
                $header .= ' charset=' . $this->get_encoding();
            } else {
                $header .= ' charset=UTF-8';
            }
            header($header);
        }
    }

    /**
     * Get the type of the feed
     *
     * This returns a self::TYPE_* constant, which can be tested against
     * using {@link http://php.net/language.operators.bitwise bitwise operators}
     *
     * @since 0.8 (usage changed to using constants in 1.0)
     * @see self::TYPE_NONE Unknown.
     * @see self::TYPE_RSS_090 RSS 0.90.
     * @see self::TYPE_RSS_091_NETSCAPE RSS 0.91 (Netscape).
     * @see self::TYPE_RSS_091_USERLAND RSS 0.91 (Userland).
     * @see self::TYPE_RSS_091 RSS 0.91.
     * @see self::TYPE_RSS_092 RSS 0.92.
     * @see self::TYPE_RSS_093 RSS 0.93.
     * @see self::TYPE_RSS_094 RSS 0.94.
     * @see self::TYPE_RSS_10 RSS 1.0.
     * @see self::TYPE_RSS_20 RSS 2.0.x.
     * @see self::TYPE_RSS_RDF RDF-based RSS.
     * @see self::TYPE_RSS_SYNDICATION Non-RDF-based RSS (truly intended as syndication format).
     * @see self::TYPE_RSS_ALL Any version of RSS.
     * @see self::TYPE_ATOM_03 Atom 0.3.
     * @see self::TYPE_ATOM_10 Atom 1.0.
     * @see self::TYPE_ATOM_ALL Any version of Atom.
     * @see self::TYPE_ALL Any known/supported feed type.
     * @return int-mask-of<self::TYPE_*> constant
     */
    public function get_type()
    {
        if (!isset($this->data['type'])) {
            $this->data['type'] = self::TYPE_ALL;
            if (isset($this->data['child'][self::NAMESPACE_ATOM_10]['feed'])) {
                $this->data['type'] &= self::TYPE_ATOM_10;
            } elseif (isset($this->data['child'][self::NAMESPACE_ATOM_03]['feed'])) {
                $this->data['type'] &= self::TYPE_ATOM_03;
            } elseif (isset($this->data['child'][self::NAMESPACE_RDF]['RDF'])) {
                if (isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_10]['channel'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_10]['image'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_10]['item'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_10]['textinput'])) {
                    $this->data['type'] &= self::TYPE_RSS_10;
                }
                if (isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_090]['channel'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_090]['image'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_090]['item'])
                || isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][self::NAMESPACE_RSS_090]['textinput'])) {
                    $this->data['type'] &= self::TYPE_RSS_090;
                }
            } elseif (isset($this->data['child'][self::NAMESPACE_RSS_20]['rss'])) {
                $this->data['type'] &= self::TYPE_RSS_ALL;
                if (isset($this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['attribs']['']['version'])) {
                    switch (trim($this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['attribs']['']['version'])) {
                        case '0.91':
                            $this->data['type'] &= self::TYPE_RSS_091;
                            if (isset($this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['child'][self::NAMESPACE_RSS_20]['skiphours']['hour'][0]['data'])) {
                                switch (trim($this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['child'][self::NAMESPACE_RSS_20]['skiphours']['hour'][0]['data'])) {
                                    case '0':
                                        $this->data['type'] &= self::TYPE_RSS_091_NETSCAPE;
                                        break;

                                    case '24':
                                        $this->data['type'] &= self::TYPE_RSS_091_USERLAND;
                                        break;
                                }
                            }
                            break;

                        case '0.92':
                            $this->data['type'] &= self::TYPE_RSS_092;
                            break;

                        case '0.93':
                            $this->data['type'] &= self::TYPE_RSS_093;
                            break;

                        case '0.94':
                            $this->data['type'] &= self::TYPE_RSS_094;
                            break;

                        case '2.0':
                            $this->data['type'] &= self::TYPE_RSS_20;
                            break;
                    }
                }
            } else {
                $this->data['type'] = self::TYPE_NONE;
            }
        }
        return $this->data['type'];
    }

    /**
     * Get the URL for the feed
     *
     * When the 'permanent' mode is enabled, returns the original feed URL,
     * except in the case of an `HTTP 301 Moved Permanently` status response,
     * in which case the location of the first redirection is returned.
     *
     * When the 'permanent' mode is disabled (default),
     * may or may not be different from the URL passed to {@see set_feed_url()},
     * depending on whether auto-discovery was used, and whether there were
     * any redirects along the way.
     *
     * @since Preview Release (previously called `get_feed_url()` since SimplePie 0.8.)
     * @todo Support <itunes:new-feed-url>
     * @todo Also, |atom:link|@rel=self
     * @param bool $permanent Permanent mode to return only the original URL or the first redirection
     * iff it is a 301 redirection
     * @return string|null
     */
    public function subscribe_url(bool $permanent = false)
    {
        if ($permanent) {
            if ($this->permanent_url !== null) {
                // sanitize encodes ampersands which are required when used in a url.
                return str_replace(
                    '&amp;',
                    '&',
                    $this->sanitize(
                        $this->permanent_url,
                        self::CONSTRUCT_IRI
                    )
                );
            }
        } else {
            if ($this->feed_url !== null) {
                return str_replace(
                    '&amp;',
                    '&',
                    $this->sanitize(
                        $this->feed_url,
                        self::CONSTRUCT_IRI
                    )
                );
            }
        }
        return null;
    }

    /**
     * Get data for an feed-level element
     *
     * This method allows you to get access to ANY element/attribute that is a
     * sub-element of the opening feed tag.
     *
     * The return value is an indexed array of elements matching the given
     * namespace and tag name. Each element has `attribs`, `data` and `child`
     * subkeys. For `attribs` and `child`, these contain namespace subkeys.
     * `attribs` then has one level of associative name => value data (where
     * `value` is a string) after the namespace. `child` has tag-indexed keys
     * after the namespace, each member of which is an indexed array matching
     * this same format.
     *
     * For example:
     * <pre>
     * // This is probably a bad example because we already support
     * // <media:content> natively, but it shows you how to parse through
     * // the nodes.
     * $group = $item->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'group');
     * $content = $group[0]['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['content'];
     * $file = $content[0]['attribs']['']['url'];
     * echo $file;
     * </pre>
     *
     * @since 1.0
     * @see http://simplepie.org/wiki/faq/supported_xml_namespaces
     * @param string $namespace The URL of the XML namespace of the elements you're trying to access
     * @param string $tag Tag name
     * @return array<array<string, mixed>>|null
     */
    public function get_feed_tags(string $namespace, string $tag)
    {
        $type = $this->get_type();
        if ($type & self::TYPE_ATOM_10) {
            if (isset($this->data['child'][self::NAMESPACE_ATOM_10]['feed'][0]['child'][$namespace][$tag])) {
                return $this->data['child'][self::NAMESPACE_ATOM_10]['feed'][0]['child'][$namespace][$tag];
            }
        }
        if ($type & self::TYPE_ATOM_03) {
            if (isset($this->data['child'][self::NAMESPACE_ATOM_03]['feed'][0]['child'][$namespace][$tag])) {
                return $this->data['child'][self::NAMESPACE_ATOM_03]['feed'][0]['child'][$namespace][$tag];
            }
        }
        if ($type & self::TYPE_RSS_RDF) {
            if (isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][$namespace][$tag])) {
                return $this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['child'][$namespace][$tag];
            }
        }
        if ($type & self::TYPE_RSS_SYNDICATION) {
            if (isset($this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['child'][$namespace][$tag])) {
                return $this->data['child'][self::NAMESPACE_RSS_20]['rss'][0]['child'][$namespace][$tag];
            }
        }
        return null;
    }

    /**
     * Get data for an channel-level element
     *
     * This method allows you to get access to ANY element/attribute in the
     * channel/header section of the feed.
     *
     * See {@see SimplePie::get_feed_tags()} for a description of the return value
     *
     * @since 1.0
     * @see http://simplepie.org/wiki/faq/supported_xml_namespaces
     * @param string $namespace The URL of the XML namespace of the elements you're trying to access
     * @param string $tag Tag name
     * @return array<array<string, mixed>>|null
     */
    public function get_channel_tags(string $namespace, string $tag)
    {
        $type = $this->get_type();
        if ($type & self::TYPE_ATOM_ALL) {
            if ($return = $this->get_feed_tags($namespace, $tag)) {
                return $return;
            }
        }
        if ($type & self::TYPE_RSS_10) {
            if ($channel = $this->get_feed_tags(self::NAMESPACE_RSS_10, 'channel')) {
                if (isset($channel[0]['child'][$namespace][$tag])) {
                    return $channel[0]['child'][$namespace][$tag];
                }
            }
        }
        if ($type & self::TYPE_RSS_090) {
            if ($channel = $this->get_feed_tags(self::NAMESPACE_RSS_090, 'channel')) {
                if (isset($channel[0]['child'][$namespace][$tag])) {
                    return $channel[0]['child'][$namespace][$tag];
                }
            }
        }
        if ($type & self::TYPE_RSS_SYNDICATION) {
            if ($channel = $this->get_feed_tags(self::NAMESPACE_RSS_20, 'channel')) {
                if (isset($channel[0]['child'][$namespace][$tag])) {
                    return $channel[0]['child'][$namespace][$tag];
                }
            }
        }
        return null;
    }

    /**
     * Get data for an channel-level element
     *
     * This method allows you to get access to ANY element/attribute in the
     * image/logo section of the feed.
     *
     * See {@see SimplePie::get_feed_tags()} for a description of the return value
     *
     * @since 1.0
     * @see http://simplepie.org/wiki/faq/supported_xml_namespaces
     * @param string $namespace The URL of the XML namespace of the elements you're trying to access
     * @param string $tag Tag name
     * @return array<array<string, mixed>>|null
     */
    public function get_image_tags(string $namespace, string $tag)
    {
        $type = $this->get_type();
        if ($type & self::TYPE_RSS_10) {
            if ($image = $this->get_feed_tags(self::NAMESPACE_RSS_10, 'image')) {
                if (isset($image[0]['child'][$namespace][$tag])) {
                    return $image[0]['child'][$namespace][$tag];
                }
            }
        }
        if ($type & self::TYPE_RSS_090) {
            if ($image = $this->get_feed_tags(self::NAMESPACE_RSS_090, 'image')) {
                if (isset($image[0]['child'][$namespace][$tag])) {
                    return $image[0]['child'][$namespace][$tag];
                }
            }
        }
        if ($type & self::TYPE_RSS_SYNDICATION) {
            if ($image = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'image')) {
                if (isset($image[0]['child'][$namespace][$tag])) {
                    return $image[0]['child'][$namespace][$tag];
                }
            }
        }
        return null;
    }

    /**
     * Get the base URL value from the feed
     *
     * Uses `<xml:base>` if available,
     * otherwise uses the first 'self' link or the first 'alternate' link of the feed,
     * or failing that, the URL of the feed itself.
     *
     * @see get_link
     * @see subscribe_url
     *
     * @param array<string, mixed> $element
     * @return string
     */
    public function get_base(array $element = [])
    {
        if (!empty($element['xml_base_explicit']) && isset($element['xml_base'])) {
            return $element['xml_base'];
        }
        if (($link = $this->get_link(0, 'alternate')) !== null) {
            return $link;
        }
        if (($link = $this->get_link(0, 'self')) !== null) {
            return $link;
        }

        return $this->subscribe_url() ?? '';
    }

    /**
     * Sanitize feed data
     *
     * @access private
     * @see Sanitize::sanitize()
     * @param string $data Data to sanitize
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @param string $base Base URL to resolve URLs against
     * @return string Sanitized data
     */
    public function sanitize(string $data, int $type, string $base = '')
    {
        try {
            // This really returns string|false but changing encoding is uncommon and we are going to deprecate it, so let’s just lie to PHPStan in the interest of cleaner annotations.
            return $this->sanitize->sanitize($data, $type, $base);
        } catch (SimplePieException $e) {
            if (!$this->enable_exceptions) {
                $this->error = $e->getMessage();
                $this->registry->call(Misc::class, 'error', [$this->error, E_USER_WARNING, $e->getFile(), $e->getLine()]);
                return '';
            }

            throw $e;
        }
    }

    /**
     * Get the title of the feed
     *
     * Uses `<atom:title>`, `<title>` or `<dc:title>`
     *
     * @since 1.0 (previously called `get_feed_title` since 0.8)
     * @return string|null
     */
    public function get_title()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'title')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'title')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_10, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_090, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_11, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_10, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * Get a category for the feed
     *
     * @since Unknown
     * @param int $key The category that you want to return. Remember that arrays begin with 0, not 1
     * @return Category|null
     */
    public function get_category(int $key = 0)
    {
        $categories = $this->get_categories();
        if (isset($categories[$key])) {
            return $categories[$key];
        }

        return null;
    }

    /**
     * Get all categories for the feed
     *
     * Uses `<atom:category>`, `<category>` or `<dc:subject>`
     *
     * @since Unknown
     * @return array<Category>|null List of {@see Category} objects
     */
    public function get_categories()
    {
        $categories = [];

        foreach ((array) $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'category') as $category) {
            $term = null;
            $scheme = null;
            $label = null;
            if (isset($category['attribs']['']['term'])) {
                $term = $this->sanitize($category['attribs']['']['term'], self::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['scheme'])) {
                $scheme = $this->sanitize($category['attribs']['']['scheme'], self::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['label'])) {
                $label = $this->sanitize($category['attribs']['']['label'], self::CONSTRUCT_TEXT);
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_RSS_20, 'category') as $category) {
            // This is really the label, but keep this as the term also for BC.
            // Label will also work on retrieving because that falls back to term.
            $term = $this->sanitize($category['data'], self::CONSTRUCT_TEXT);
            if (isset($category['attribs']['']['domain'])) {
                $scheme = $this->sanitize($category['attribs']['']['domain'], self::CONSTRUCT_TEXT);
            } else {
                $scheme = null;
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, null]);
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_DC_11, 'subject') as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], self::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_DC_10, 'subject') as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], self::CONSTRUCT_TEXT), null, null]);
        }

        if (!empty($categories)) {
            return array_unique($categories);
        }

        return null;
    }

    /**
     * Get an author for the feed
     *
     * @since 1.1
     * @param int $key The author that you want to return. Remember that arrays begin with 0, not 1
     * @return Author|null
     */
    public function get_author(int $key = 0)
    {
        $authors = $this->get_authors();
        if (isset($authors[$key])) {
            return $authors[$key];
        }

        return null;
    }

    /**
     * Get all authors for the feed
     *
     * Uses `<atom:author>`, `<author>`, `<dc:creator>` or `<itunes:author>`
     *
     * @since 1.1
     * @return array<Author>|null List of {@see Author} objects
     */
    public function get_authors()
    {
        $authors = [];
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'author') as $author) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($author['child'][self::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($author['child'][self::NAMESPACE_ATOM_10]['name'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if (isset($author['child'][self::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $author['child'][self::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], self::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($author['child'][self::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($author['child'][self::NAMESPACE_ATOM_10]['email'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        if ($author = $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'author')) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($author[0]['child'][self::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($author[0]['child'][self::NAMESPACE_ATOM_03]['name'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if (isset($author[0]['child'][self::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $author[0]['child'][self::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], self::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($author[0]['child'][self::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($author[0]['child'][self::NAMESPACE_ATOM_03]['email'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_DC_11, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], self::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_DC_10, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], self::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_ITUNES, 'author') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], self::CONSTRUCT_TEXT), null, null]);
        }

        if (!empty($authors)) {
            return array_unique($authors);
        }

        return null;
    }

    /**
     * Get a contributor for the feed
     *
     * @since 1.1
     * @param int $key The contrbutor that you want to return. Remember that arrays begin with 0, not 1
     * @return Author|null
     */
    public function get_contributor(int $key = 0)
    {
        $contributors = $this->get_contributors();
        if (isset($contributors[$key])) {
            return $contributors[$key];
        }

        return null;
    }

    /**
     * Get all contributors for the feed
     *
     * Uses `<atom:contributor>`
     *
     * @since 1.1
     * @return array<Author>|null List of {@see Author} objects
     */
    public function get_contributors()
    {
        $contributors = [];
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'contributor') as $contributor) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($contributor['child'][self::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][self::NAMESPACE_ATOM_10]['name'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][self::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $contributor['child'][self::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], self::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($contributor['child'][self::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][self::NAMESPACE_ATOM_10]['email'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        foreach ((array) $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'contributor') as $contributor) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($contributor['child'][self::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][self::NAMESPACE_ATOM_03]['name'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][self::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $contributor['child'][self::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], self::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($contributor['child'][self::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][self::NAMESPACE_ATOM_03]['email'][0]['data'], self::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }

        if (!empty($contributors)) {
            return array_unique($contributors);
        }

        return null;
    }

    /**
     * Get a single link for the feed
     *
     * @since 1.0 (previously called `get_feed_link` since Preview Release, `get_feed_permalink()` since 0.8)
     * @param int $key The link that you want to return. Remember that arrays begin with 0, not 1
     * @param string $rel The relationship of the link to return
     * @return string|null Link URL
     */
    public function get_link(int $key = 0, string $rel = 'alternate')
    {
        $links = $this->get_links($rel);
        if (isset($links[$key])) {
            return $links[$key];
        }

        return null;
    }

    /**
     * Get the permalink for the item
     *
     * Returns the first link available with a relationship of "alternate".
     * Identical to {@see get_link()} with key 0
     *
     * @see get_link
     * @since 1.0 (previously called `get_feed_link` since Preview Release, `get_feed_permalink()` since 0.8)
     * @internal Added for parity between the parent-level and the item/entry-level.
     * @return string|null Link URL
     */
    public function get_permalink()
    {
        return $this->get_link(0);
    }

    /**
     * Get all links for the feed
     *
     * Uses `<atom:link>` or `<link>`
     *
     * @since Beta 2
     * @param string $rel The relationship of links to return
     * @return array<string>|null Links found for the feed (strings)
     */
    public function get_links(string $rel = 'alternate')
    {
        if (!isset($this->data['links'])) {
            $this->data['links'] = [];
            if ($links = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'link')) {
                foreach ($links as $link) {
                    if (isset($link['attribs']['']['href'])) {
                        $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                        $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], self::CONSTRUCT_IRI, $this->get_base($link));
                    }
                }
            }
            if ($links = $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'link')) {
                foreach ($links as $link) {
                    if (isset($link['attribs']['']['href'])) {
                        $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                        $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], self::CONSTRUCT_IRI, $this->get_base($link));
                    }
                }
            }
            if ($links = $this->get_channel_tags(self::NAMESPACE_RSS_10, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], self::CONSTRUCT_IRI, $this->get_base($links[0]));
            }
            if ($links = $this->get_channel_tags(self::NAMESPACE_RSS_090, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], self::CONSTRUCT_IRI, $this->get_base($links[0]));
            }
            if ($links = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], self::CONSTRUCT_IRI, $this->get_base($links[0]));
            }

            $keys = array_keys($this->data['links']);
            foreach ($keys as $key) {
                if ($this->registry->call(Misc::class, 'is_isegment_nz_nc', [$key])) {
                    if (isset($this->data['links'][self::IANA_LINK_RELATIONS_REGISTRY . $key])) {
                        $this->data['links'][self::IANA_LINK_RELATIONS_REGISTRY . $key] = array_merge($this->data['links'][$key], $this->data['links'][self::IANA_LINK_RELATIONS_REGISTRY . $key]);
                        $this->data['links'][$key] = &$this->data['links'][self::IANA_LINK_RELATIONS_REGISTRY . $key];
                    } else {
                        $this->data['links'][self::IANA_LINK_RELATIONS_REGISTRY . $key] = &$this->data['links'][$key];
                    }
                } elseif (substr($key, 0, 41) === self::IANA_LINK_RELATIONS_REGISTRY) {
                    $this->data['links'][substr($key, 41)] = &$this->data['links'][$key];
                }
                $this->data['links'][$key] = array_unique($this->data['links'][$key]);
            }
        }

        if (isset($this->data['headers']['link'])) {
            $link_headers = $this->data['headers']['link'];
            if (is_array($link_headers)) {
                $link_headers = implode(',', $link_headers);
            }
            // https://datatracker.ietf.org/doc/html/rfc8288
            if (is_string($link_headers) &&
                preg_match_all('/<(?P<uri>[^>]+)>\s*;\s*rel\s*=\s*(?P<quote>"?)' . preg_quote($rel) . '(?P=quote)\s*(?=,|$)/i', $link_headers, $matches)) {
                return $matches['uri'];
            }
        }

        if (isset($this->data['links'][$rel])) {
            return $this->data['links'][$rel];
        }

        return null;
    }

    /**
     * @return ?array<Response>
     */
    public function get_all_discovered_feeds()
    {
        return $this->all_discovered_feeds;
    }

    /**
     * Get the content for the item
     *
     * Uses `<atom:subtitle>`, `<atom:tagline>`, `<description>`,
     * `<dc:description>`, `<itunes:summary>` or `<itunes:subtitle>`
     *
     * @since 1.0 (previously called `get_feed_description()` since 0.8)
     * @return string|null
     */
    public function get_description()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'subtitle')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'tagline')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_10, 'description')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_090, 'description')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'description')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_11, 'description')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_10, 'description')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ITUNES, 'summary')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ITUNES, 'subtitle')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_HTML, $this->get_base($return[0]));
        }

        return null;
    }

    /**
     * Get the copyright info for the feed
     *
     * Uses `<atom:rights>`, `<atom:copyright>` or `<dc:rights>`
     *
     * @since 1.0 (previously called `get_feed_copyright()` since 0.8)
     * @return string|null
     */
    public function get_copyright()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'rights')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_03, 'copyright')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'copyright')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_11, 'rights')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_10, 'rights')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * Get the language for the feed
     *
     * Uses `<language>`, `<dc:language>`, or @xml_lang
     *
     * @since 1.0 (previously called `get_feed_language()` since 0.8)
     * @return string|null
     */
    public function get_language()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'language')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_11, 'language')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_DC_10, 'language')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif (isset($this->data['child'][self::NAMESPACE_ATOM_10]['feed'][0]['xml_lang'])) {
            return $this->sanitize($this->data['child'][self::NAMESPACE_ATOM_10]['feed'][0]['xml_lang'], self::CONSTRUCT_TEXT);
        } elseif (isset($this->data['child'][self::NAMESPACE_ATOM_03]['feed'][0]['xml_lang'])) {
            return $this->sanitize($this->data['child'][self::NAMESPACE_ATOM_03]['feed'][0]['xml_lang'], self::CONSTRUCT_TEXT);
        } elseif (isset($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['xml_lang'])) {
            return $this->sanitize($this->data['child'][self::NAMESPACE_RDF]['RDF'][0]['xml_lang'], self::CONSTRUCT_TEXT);
        } elseif (isset($this->data['headers']['content-language'])) {
            return $this->sanitize($this->data['headers']['content-language'], self::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * Get the latitude coordinates for the item
     *
     * Compatible with the W3C WGS84 Basic Geo and GeoRSS specifications
     *
     * Uses `<geo:lat>` or `<georss:point>`
     *
     * @since 1.0
     * @link http://www.w3.org/2003/01/geo/ W3C WGS84 Basic Geo
     * @link http://www.georss.org/ GeoRSS
     * @return float|null
     */
    public function get_latitude()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_W3C_BASIC_GEO, 'lat')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_channel_tags(self::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[1];
        }

        return null;
    }

    /**
     * Get the longitude coordinates for the feed
     *
     * Compatible with the W3C WGS84 Basic Geo and GeoRSS specifications
     *
     * Uses `<geo:long>`, `<geo:lon>` or `<georss:point>`
     *
     * @since 1.0
     * @link http://www.w3.org/2003/01/geo/ W3C WGS84 Basic Geo
     * @link http://www.georss.org/ GeoRSS
     * @return float|null
     */
    public function get_longitude()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_W3C_BASIC_GEO, 'long')) {
            return (float) $return[0]['data'];
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_W3C_BASIC_GEO, 'lon')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_channel_tags(self::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[2];
        }

        return null;
    }

    /**
     * Get the feed logo's title
     *
     * RSS 0.9.0, 1.0 and 2.0 feeds are allowed to have a "feed logo" title.
     *
     * Uses `<image><title>` or `<image><dc:title>`
     *
     * @return string|null
     */
    public function get_image_title()
    {
        if ($return = $this->get_image_tags(self::NAMESPACE_RSS_10, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_090, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_20, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_DC_11, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_DC_10, 'title')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * Get the feed logo's URL
     *
     * RSS 0.9.0, 2.0, Atom 1.0, and feeds with iTunes RSS tags are allowed to
     * have a "feed logo" URL. This points directly to the image itself.
     *
     * Uses `<itunes:image>`, `<atom:logo>`, `<atom:icon>`,
     * `<image><title>` or `<image><dc:title>`
     *
     * @return string|null
     */
    public function get_image_url()
    {
        if ($return = $this->get_channel_tags(self::NAMESPACE_ITUNES, 'image')) {
            return $this->sanitize($return[0]['attribs']['']['href'], self::CONSTRUCT_IRI);
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'logo')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_channel_tags(self::NAMESPACE_ATOM_10, 'icon')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_10, 'url')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_090, 'url')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_20, 'url')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        }

        return null;
    }

    /**
     * Get the feed logo's link
     *
     * RSS 0.9.0, 1.0 and 2.0 feeds are allowed to have a "feed logo" link. This
     * points to a human-readable page that the image should link to.
     *
     * Uses `<itunes:image>`, `<atom:logo>`, `<atom:icon>`,
     * `<image><title>` or `<image><dc:title>`
     *
     * @return string|null
     */
    public function get_image_link()
    {
        if ($return = $this->get_image_tags(self::NAMESPACE_RSS_10, 'link')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_090, 'link')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_image_tags(self::NAMESPACE_RSS_20, 'link')) {
            return $this->sanitize($return[0]['data'], self::CONSTRUCT_IRI, $this->get_base($return[0]));
        }

        return null;
    }

    /**
     * Get the feed logo's link
     *
     * RSS 2.0 feeds are allowed to have a "feed logo" width.
     *
     * Uses `<image><width>` or defaults to 88 if no width is specified and
     * the feed is an RSS 2.0 feed.
     *
     * @return int|null
     */
    public function get_image_width()
    {
        if ($return = $this->get_image_tags(self::NAMESPACE_RSS_20, 'width')) {
            return intval($return[0]['data']);
        } elseif ($this->get_type() & self::TYPE_RSS_SYNDICATION && $this->get_image_tags(self::NAMESPACE_RSS_20, 'url')) {
            return 88;
        }

        return null;
    }

    /**
     * Get the feed logo's height
     *
     * RSS 2.0 feeds are allowed to have a "feed logo" height.
     *
     * Uses `<image><height>` or defaults to 31 if no height is specified and
     * the feed is an RSS 2.0 feed.
     *
     * @return int|null
     */
    public function get_image_height()
    {
        if ($return = $this->get_image_tags(self::NAMESPACE_RSS_20, 'height')) {
            return intval($return[0]['data']);
        } elseif ($this->get_type() & self::TYPE_RSS_SYNDICATION && $this->get_image_tags(self::NAMESPACE_RSS_20, 'url')) {
            return 31;
        }

        return null;
    }

    /**
     * Get the number of items in the feed
     *
     * This is well-suited for {@link http://php.net/for for()} loops with
     * {@see get_item()}
     *
     * @param int $max Maximum value to return. 0 for no limit
     * @return int Number of items in the feed
     */
    public function get_item_quantity(int $max = 0)
    {
        $qty = count($this->get_items());
        if ($max === 0) {
            return $qty;
        }

        return min($qty, $max);
    }

    /**
     * Get a single item from the feed
     *
     * This is better suited for {@link http://php.net/for for()} loops, whereas
     * {@see get_items()} is better suited for
     * {@link http://php.net/foreach foreach()} loops.
     *
     * @see get_item_quantity()
     * @since Beta 2
     * @param int $key The item that you want to return. Remember that arrays begin with 0, not 1
     * @return Item|null
     */
    public function get_item(int $key = 0)
    {
        $items = $this->get_items();
        if (isset($items[$key])) {
            return $items[$key];
        }

        return null;
    }

    /**
     * Get all items from the feed
     *
     * This is better suited for {@link http://php.net/for for()} loops, whereas
     * {@see get_items()} is better suited for
     * {@link http://php.net/foreach foreach()} loops.
     *
     * @see get_item_quantity
     * @since Beta 2
     * @param int $start Index to start at
     * @param int $end Number of items to return. 0 for all items after `$start`
     * @return Item[] List of {@see Item} objects
     */
    public function get_items(int $start = 0, int $end = 0)
    {
        if (!isset($this->data['items'])) {
            if (!empty($this->multifeed_objects)) {
                $this->data['items'] = SimplePie::merge_items($this->multifeed_objects, $start, $end, $this->item_limit);
                if (empty($this->data['items'])) {
                    return [];
                }
                return $this->data['items'];
            }
            $this->data['items'] = [];
            if ($items = $this->get_feed_tags(self::NAMESPACE_ATOM_10, 'entry')) {
                $keys = array_keys($items);
                foreach ($keys as $key) {
                    $this->data['items'][] = $this->make_item($items[$key]);
                }
            }
            if ($items = $this->get_feed_tags(self::NAMESPACE_ATOM_03, 'entry')) {
                $keys = array_keys($items);
                foreach ($keys as $key) {
                    $this->data['items'][] = $this->make_item($items[$key]);
                }
            }
            if ($items = $this->get_feed_tags(self::NAMESPACE_RSS_10, 'item')) {
                $keys = array_keys($items);
                foreach ($keys as $key) {
                    $this->data['items'][] = $this->make_item($items[$key]);
                }
            }
            if ($items = $this->get_feed_tags(self::NAMESPACE_RSS_090, 'item')) {
                $keys = array_keys($items);
                foreach ($keys as $key) {
                    $this->data['items'][] = $this->make_item($items[$key]);
                }
            }
            if ($items = $this->get_channel_tags(self::NAMESPACE_RSS_20, 'item')) {
                $keys = array_keys($items);
                foreach ($keys as $key) {
                    $this->data['items'][] = $this->make_item($items[$key]);
                }
            }
        }

        if (empty($this->data['items'])) {
            return [];
        }

        if ($this->order_by_date) {
            if (!isset($this->data['ordered_items'])) {
                $this->data['ordered_items'] = $this->data['items'];
                usort($this->data['ordered_items'], [get_class($this), 'sort_items']);
            }
            $items = $this->data['ordered_items'];
        } else {
            $items = $this->data['items'];
        }
        // Slice the data as desired
        if ($end === 0) {
            return array_slice($items, $start);
        }

        return array_slice($items, $start, $end);
    }

    /**
     * Set the favicon handler
     *
     * @deprecated Use your own favicon handling instead
     * @param string|false $page
     * @return bool
     */
    public function set_favicon_handler($page = false, string $qs = 'i')
    {
        trigger_error('Favicon handling has been removed since SimplePie 1.3, please use your own handling', \E_USER_DEPRECATED);
        return false;
    }

    /**
     * Get the favicon for the current feed
     *
     * @deprecated Use your own favicon handling instead
     * @return string|bool
     */
    public function get_favicon()
    {
        trigger_error('Favicon handling has been removed since SimplePie 1.3, please use your own handling', \E_USER_DEPRECATED);

        if (($url = $this->get_link()) !== null) {
            return 'https://www.google.com/s2/favicons?domain=' . urlencode($url);
        }

        return false;
    }

    /**
     * Magic method handler
     *
     * @param string $method Method name
     * @param array<mixed> $args Arguments to the method
     * @return mixed
     */
    public function __call(string $method, array $args)
    {
        if (strpos($method, 'subscribe_') === 0) {
            trigger_error('subscribe_*() has been deprecated since SimplePie 1.3, implement the callback yourself', \E_USER_DEPRECATED);
            return '';
        }
        if ($method === 'enable_xml_dump') {
            trigger_error('enable_xml_dump() has been deprecated since SimplePie 1.3, use get_raw_data() instead', \E_USER_DEPRECATED);
            return false;
        }

        $class = get_class($this);
        $trace = debug_backtrace();
        $file = $trace[0]['file'] ?? '';
        $line = $trace[0]['line'] ?? '';
        throw new SimplePieException("Call to undefined method $class::$method() in $file on line $line");
    }

    /**
     * Item factory
     *
     * @param array<string, mixed> $data
     */
    private function make_item(array $data): Item
    {
        $item = $this->registry->create(Item::class, [$this, $data]);
        $item->set_sanitize($this->sanitize);

        return $item;
    }

    /**
     * Sorting callback for items
     *
     * @access private
     * @param Item $a
     * @param Item $b
     * @return -1|0|1
     */
    public static function sort_items(Item $a, Item $b)
    {
        $a_date = $a->get_date('U');
        $b_date = $b->get_date('U');
        if ($a_date && $b_date) {
            return $a_date > $b_date ? -1 : 1;
        }
        // Sort items without dates to the top.
        if ($a_date) {
            return 1;
        }
        if ($b_date) {
            return -1;
        }
        return 0;
    }

    /**
     * Merge items from several feeds into one
     *
     * If you're merging multiple feeds together, they need to all have dates
     * for the items or else SimplePie will refuse to sort them.
     *
     * @link http://simplepie.org/wiki/tutorial/sort_multiple_feeds_by_time_and_date#if_feeds_require_separate_per-feed_settings
     * @param array<SimplePie> $urls List of SimplePie feed objects to merge
     * @param int $start Starting item
     * @param int $end Number of items to return
     * @param int $limit Maximum number of items per feed
     * @return array<Item>
     */
    public static function merge_items(array $urls, int $start = 0, int $end = 0, int $limit = 0)
    {
        if (count($urls) > 0) {
            $items = [];
            foreach ($urls as $arg) {
                if ($arg instanceof SimplePie) {
                    $items = array_merge($items, $arg->get_items(0, $limit));

                    // @phpstan-ignore-next-line Enforce PHPDoc type.
                } else {
                    trigger_error('Arguments must be SimplePie objects', E_USER_WARNING);
                }
            }

            usort($items, [get_class($urls[0]), 'sort_items']);

            if ($end === 0) {
                return array_slice($items, $start);
            }

            return array_slice($items, $start, $end);
        }

        trigger_error('Cannot merge zero SimplePie objects', E_USER_WARNING);
        return [];
    }

    /**
     * Store PubSubHubbub links as headers
     *
     * There is no way to find PuSH links in the body of a microformats feed,
     * so they are added to the headers when found, to be used later by get_links.
     */
    private function store_links(Response $file, ?string $hub, ?string $self): Response
    {
        $linkHeaderLine = $file->get_header_line('link');
        $linkHeader = $file->get_header('link');

        if ($hub && !preg_match('/rel=hub/', $linkHeaderLine)) {
            $linkHeader[] = '<'.$hub.'>; rel=hub';
        }

        if ($self && !preg_match('/rel=self/', $linkHeaderLine)) {
            $linkHeader[] = '<'.$self.'>; rel=self';
        }

        if (count($linkHeader) > 0) {
            $file = $file->with_header('link', $linkHeader);
        }

        return $file;
    }

    /**
     * Get a DataCache
     *
     * @param string $feed_url Only needed for BC, can be removed in SimplePie 2.0.0
     *
     * @return DataCache
     */
    private function get_cache(string $feed_url = ''): DataCache
    {
        if ($this->cache === null) {
            // @trigger_error(sprintf('Not providing as PSR-16 cache implementation is deprecated since SimplePie 1.8.0, please use "SimplePie\SimplePie::set_cache()".'), \E_USER_DEPRECATED);
            $cache = $this->registry->call(Cache::class, 'get_handler', [
                $this->cache_location,
                $this->get_cache_filename($feed_url),
                Base::TYPE_FEED
            ]);

            return new BaseDataCache($cache);
        }

        return $this->cache;
    }

    /**
     * Get a HTTP client
     */
    private function get_http_client(): Client
    {
        if ($this->http_client === null) {
            $this->http_client = new FileClient(
                $this->get_registry(),
                [
                    'timeout' => $this->timeout,
                    'redirects' => 5,
                    'useragent' => $this->useragent,
                    'force_fsockopen' => $this->force_fsockopen,
                    'curl_options' => $this->curl_options,
                ]
            );
            $this->http_client_injected = true;
        }

        return $this->http_client;
    }
}

class_alias('SimplePie\SimplePie', 'SimplePie');

}

namespace SimplePie {

/**
 * Manages all author-related data
 *
 * Used by {@see Item::get_author()} and {@see SimplePie::get_authors()}
 *
 * This class can be overloaded with {@see SimplePie::set_author_class()}
 */
class Author
{
    /**
     * Author's name
     *
     * @var ?string
     * @see get_name()
     */
    public $name;

    /**
     * Author's link
     *
     * @var ?string
     * @see get_link()
     */
    public $link;

    /**
     * Author's email address
     *
     * @var ?string
     * @see get_email()
     */
    public $email;

    /**
     * Constructor, used to input the data
     */
    public function __construct(
        ?string $name = null,
        ?string $link = null,
        ?string $email = null
    ) {
        $this->name = $name;
        $this->link = $link;
        $this->email = $email;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Author's name
     *
     * @return string|null
     */
    public function get_name()
    {
        if ($this->name !== null) {
            return $this->name;
        }

        return null;
    }

    /**
     * Author's link
     *
     * @return string|null
     */
    public function get_link()
    {
        if ($this->link !== null) {
            return $this->link;
        }

        return null;
    }

    /**
     * Author's email address
     *
     * @return string|null
     */
    public function get_email()
    {
        if ($this->email !== null) {
            return $this->email;
        }

        return null;
    }
}

class_alias('SimplePie\Author', 'SimplePie_Author');

}

namespace SimplePie {

use SimplePie\Cache\Base;

/**
 * Used to create cache objects
 *
 * This class can be overloaded with {@see SimplePie::set_cache_class()},
 * although the preferred way is to create your own handler
 * via {@see register()}
 *
 * @deprecated since SimplePie 1.8.0, use "SimplePie\SimplePie::set_cache()" instead
 */
class Cache
{
    /**
     * Cache handler classes
     *
     * These receive 3 parameters to their constructor, as documented in
     * {@see register()}
     * @var array<string, class-string<Base>>
     */
    protected static $handlers = [
        'mysql'     => Cache\MySQL::class,
        'memcache'  => Cache\Memcache::class,
        'memcached' => Cache\Memcached::class,
        'redis'     => Cache\Redis::class,
    ];

    /**
     * Don't call the constructor. Please.
     */
    private function __construct()
    {
    }

    /**
     * Create a new SimplePie\Cache object
     *
     * @param string $location URL location (scheme is used to determine handler)
     * @param string $filename Unique identifier for cache object
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $extension 'spi' or 'spc'
     * @return Base Type of object depends on scheme of `$location`
     */
    public static function get_handler(string $location, string $filename, $extension)
    {
        $type = explode(':', $location, 2);
        $type = $type[0];
        if (!empty(self::$handlers[$type])) {
            $class = self::$handlers[$type];
            return new $class($location, $filename, $extension);
        }

        return new \SimplePie\Cache\File($location, $filename, $extension);
    }

    /**
     * Create a new SimplePie\Cache object
     *
     * @deprecated since SimplePie 1.3.1, use {@see get_handler()} instead
     * @param string $location
     * @param string $filename
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $extension
     * @return Base
     */
    public function create(string $location, string $filename, $extension)
    {
        trigger_error('Cache::create() has been replaced with Cache::get_handler() since SimplePie 1.3.1, use the registry system instead.', \E_USER_DEPRECATED);

        return self::get_handler($location, $filename, $extension);
    }

    /**
     * Register a handler
     *
     * @param string $type DSN type to register for
     * @param class-string<Base> $class Name of handler class. Must implement Base
     * @return void
     */
    public static function register(string $type, $class)
    {
        self::$handlers[$type] = $class;
    }

    /**
     * Parse a URL into an array
     *
     * @param string $url
     * @return array<string, mixed>
     */
    public static function parse_URL(string $url)
    {
        $parsedUrl = parse_url($url);

        if ($parsedUrl === false) {
            return [];
        }

        $params = array_merge($parsedUrl, ['extras' => []]);
        if (isset($params['query'])) {
            parse_str($params['query'], $params['extras']);
        }
        return $params;
    }
}

class_alias('SimplePie\Cache', 'SimplePie_Cache');

}

namespace SimplePie\Cache {

/**
 * Base for cache objects
 *
 * Classes to be used with {@see \SimplePie\Cache::register()} are expected
 * to implement this interface.
 *
 * @deprecated since SimplePie 1.8.0, use "Psr\SimpleCache\CacheInterface" instead
 */
interface Base
{
    /**
     * Feed cache type
     *
     * @var string
     */
    public const TYPE_FEED = 'spc';

    /**
     * Image cache type
     *
     * @var string
     */
    public const TYPE_IMAGE = 'spi';

    /**
     * Create a new cache object
     *
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $type Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $type);

    /**
     * Save data to the cache
     *
     * @param array<mixed>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data);

    /**
     * Retrieve the data saved to the cache
     *
     * @return array<mixed> Data for SimplePie::$data
     */
    public function load();

    /**
     * Retrieve the last modified time for the cache
     *
     * @return int Timestamp
     */
    public function mtime();

    /**
     * Set the last modified time to the current time
     *
     * @return bool Success status
     */
    public function touch();

    /**
     * Remove the cache
     *
     * @return bool Success status
     */
    public function unlink();
}

class_alias('SimplePie\Cache\Base', 'SimplePie_Cache_Base');

}

namespace SimplePie\Cache {

use InvalidArgumentException;

/**
 * Adapter for deprecated \SimplePie\Cache\Base implementations
 *
 * @internal
 */
final class BaseDataCache implements DataCache
{
    /**
     * @var Base
     */
    private $cache;

    public function __construct(Base $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Fetches a value from the cache.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::get()
     * <code>
     * public function get(string $key, mixed $default = null): mixed;
     * </code>
     *
     * @param string $key     The unique key of this item in the cache.
     * @param mixed  $default Default value to return if the key does not exist.
     *
     * @return array|mixed The value of the item from the cache, or $default in case of cache miss.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function get_data(string $key, $default = null)
    {
        $data = $this->cache->load();

        if (!is_array($data)) {
            return $default;
        }

        // ignore data if internal cache expiration time is not set
        if (!array_key_exists('__cache_expiration_time', $data)) {
            return $default;
        }

        // ignore data if internal cache expiration time is expired
        if ($data['__cache_expiration_time'] < time()) {
            return $default;
        }

        // remove internal cache expiration time
        unset($data['__cache_expiration_time']);

        return $data;
    }

    /**
     * Persists data in the cache, uniquely referenced by a key with an optional expiration TTL time.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::set()
     * <code>
     * public function set(string $key, mixed $value, null|int|\DateInterval $ttl = null): bool;
     * </code>
     *
     * @param string   $key   The key of the item to store.
     * @param array<mixed> $value The value of the item to store, must be serializable.
     * @param null|int $ttl   Optional. The TTL value of this item. If no value is sent and
     *                                      the driver supports TTL then the library may set a default value
     *                                      for it or let the driver take care of that.
     *
     * @return bool True on success and false on failure.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function set_data(string $key, array $value, ?int $ttl = null): bool
    {
        if ($ttl === null) {
            $ttl = 3600;
        }

        // place internal cache expiration time
        $value['__cache_expiration_time'] = time() + $ttl;

        return $this->cache->save($value);
    }

    /**
     * Delete an item from the cache by its unique key.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::delete()
     * <code>
     * public function delete(string $key): bool;
     * </code>
     *
     * @param string $key The unique cache key of the item to delete.
     *
     * @return bool True if the item was successfully removed. False if there was an error.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function delete_data(string $key): bool
    {
        return $this->cache->unlink();
    }
}

}

namespace SimplePie\Cache {

/**
 * Creating a cache filename with callables
 */
final class CallableNameFilter implements NameFilter
{
    /**
     * @var callable(string): string
     */
    private $callable;

    /**
     * @param callable(string): string $callable
     */
    public function __construct(callable $callable)
    {
        $this->callable = $callable;
    }

    /**
     * Method to create cache filename with.
     *
     * The returning name MUST follow the rules for keys in PSR-16.
     *
     * @link https://www.php-fig.org/psr/psr-16/
     *
     * The returning name MUST be a string of at least one character
     * that uniquely identifies a cached item, MUST only contain the
     * characters A-Z, a-z, 0-9, _, and . in any order in UTF-8 encoding
     * and MUST not longer then 64 characters. The following characters
     * are reserved for future extensions and MUST NOT be used: {}()/\@:
     *
     * A provided implementing library MAY support additional characters
     * and encodings or longer lengths, but MUST support at least that
     * minimum.
     *
     * @param string $name The name for the cache will be most likely an url with query string
     *
     * @return string the new cache name
     */
    public function filter(string $name): string
    {
        return call_user_func($this->callable, $name);
    }
}

}

namespace SimplePie\Cache {

use SimplePie\Item;

/**
 * Base class for database-based caches
 *
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
abstract class DB implements Base
{
    /**
     * Helper for database conversion
     *
     * Converts a given {@see SimplePie} object into data to be stored
     *
     * @param \SimplePie\SimplePie $data
     * @return array{string, array<string, Item>} First item is the serialized data for storage, second item is the unique ID for this item
     */
    protected static function prepare_simplepie_object_for_cache(\SimplePie\SimplePie $data)
    {
        $items = $data->get_items();
        $items_by_id = [];

        if (!empty($items)) {
            foreach ($items as $item) {
                $items_by_id[$item->get_id()] = $item;
            }

            if (count($items_by_id) !== count($items)) {
                $items_by_id = [];
                foreach ($items as $item) {
                    $items_by_id[$item->get_id(true)] = $item;
                }
            }

            if (isset($data->data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['feed'][0])) {
                $channel = &$data->data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['feed'][0];
            } elseif (isset($data->data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['feed'][0])) {
                $channel = &$data->data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['feed'][0];
            } elseif (isset($data->data['child'][\SimplePie\SimplePie::NAMESPACE_RDF]['RDF'][0])) {
                $channel = &$data->data['child'][\SimplePie\SimplePie::NAMESPACE_RDF]['RDF'][0];
            } elseif (isset($data->data['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['rss'][0]['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['channel'][0])) {
                $channel = &$data->data['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['rss'][0]['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['channel'][0];
            } else {
                $channel = null;
            }

            if ($channel !== null) {
                if (isset($channel['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['entry'])) {
                    unset($channel['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['entry']);
                }
                if (isset($channel['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['entry'])) {
                    unset($channel['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['entry']);
                }
                if (isset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_10]['item'])) {
                    unset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_10]['item']);
                }
                if (isset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_090]['item'])) {
                    unset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_090]['item']);
                }
                if (isset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['item'])) {
                    unset($channel['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['item']);
                }
            }
            if (isset($data->data['items'])) {
                unset($data->data['items']);
            }
            if (isset($data->data['ordered_items'])) {
                unset($data->data['ordered_items']);
            }
        }
        return [serialize($data->data), $items_by_id];
    }
}

class_alias('SimplePie\Cache\DB', 'SimplePie_Cache_DB');

}

namespace SimplePie\Cache {

use InvalidArgumentException;

/**
 * Subset of PSR-16 Cache client for caching data arrays
 *
 * Only get(), set() and delete() methods are used,
 * but not has(), getMultiple(), setMultiple() or deleteMultiple().
 *
 * The methods names must be different, but should be compatible to the
 * methods of \Psr\SimpleCache\CacheInterface.
 *
 * @internal
 */
interface DataCache
{
    /**
     * Fetches a value from the cache.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::get()
     * <code>
     * public function get(string $key, mixed $default = null): mixed;
     * </code>
     *
     * @param string   $key     The unique key of this item in the cache.
     * @param mixed    $default Default value to return if the key does not exist.
     *
     * @return array|mixed The value of the item from the cache, or $default in case of cache miss.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function get_data(string $key, $default = null);

    /**
     * Persists data in the cache, uniquely referenced by a key with an optional expiration TTL time.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::set()
     * <code>
     * public function set(string $key, mixed $value, null|int|\DateInterval $ttl = null): bool;
     * </code>
     *
     * @param string   $key   The key of the item to store.
     * @param array<mixed> $value The value of the item to store, must be serializable.
     * @param null|int $ttl   Optional. The TTL value of this item. If no value is sent and
     *                                      the driver supports TTL then the library may set a default value
     *                                      for it or let the driver take care of that.
     *
     * @return bool True on success and false on failure.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function set_data(string $key, array $value, ?int $ttl = null): bool;

    /**
     * Delete an item from the cache by its unique key.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::delete()
     * <code>
     * public function delete(string $key): bool;
     * </code>
     *
     * @param string $key The unique cache key of the item to delete.
     *
     * @return bool True if the item was successfully removed. False if there was an error.
     *
     * @throws InvalidArgumentException
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function delete_data(string $key): bool;
}

}

namespace SimplePie\Cache {

/**
 * Caches data to the filesystem
 *
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
class File implements Base
{
    /**
     * Location string
     *
     * @see SimplePie::$cache_location
     * @var string
     */
    protected $location;

    /**
     * Filename
     *
     * @var string
     */
    protected $filename;

    /**
     * File extension
     *
     * @var string
     */
    protected $extension;

    /**
     * File path
     *
     * @var string
     */
    protected $name;

    /**
     * Create a new cache object
     *
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $type Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $type)
    {
        $this->location = $location;
        $this->filename = $name;
        $this->extension = $type;
        $this->name = "$this->location/$this->filename.$this->extension";
    }

    /**
     * Save data to the cache
     *
     * @param array<mixed>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data)
    {
        if (file_exists($this->name) && is_writable($this->name) || file_exists($this->location) && is_writable($this->location)) {
            if ($data instanceof \SimplePie\SimplePie) {
                $data = $data->data;
            }

            $data = serialize($data);
            return (bool) file_put_contents($this->name, $data);
        }
        return false;
    }

    /**
     * Retrieve the data saved to the cache
     *
     * @return array<mixed>|false Data for SimplePie::$data
     */
    public function load()
    {
        if (file_exists($this->name) && is_readable($this->name)) {
            return unserialize((string) file_get_contents($this->name));
        }
        return false;
    }

    /**
     * Retrieve the last modified time for the cache
     *
     * @return int|false Timestamp
     */
    public function mtime()
    {
        return @filemtime($this->name);
    }

    /**
     * Set the last modified time to the current time
     *
     * @return bool Success status
     */
    public function touch()
    {
        return @touch($this->name);
    }

    /**
     * Remove the cache
     *
     * @return bool Success status
     */
    public function unlink()
    {
        if (file_exists($this->name)) {
            return unlink($this->name);
        }
        return false;
    }
}

class_alias('SimplePie\Cache\File', 'SimplePie_Cache_File');

}

namespace SimplePie\Cache {

use Memcache as NativeMemcache;

/**
 * Caches data to memcache
 *
 * Registered for URLs with the "memcache" protocol
 *
 * For example, `memcache://localhost:11211/?timeout=3600&prefix=sp_` will
 * connect to memcache on `localhost` on port 11211. All tables will be
 * prefixed with `sp_` and data will expire after 3600 seconds
 *
 * @uses Memcache
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
class Memcache implements Base
{
    /**
     * Memcache instance
     *
     * @var NativeMemcache
     */
    protected $cache;

    /**
     * Options
     *
     * @var array<string, mixed>
     */
    protected $options;

    /**
     * Cache name
     *
     * @var string
     */
    protected $name;

    /**
     * Create a new cache object
     *
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $type Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $type)
    {
        $this->options = [
            'host' => '127.0.0.1',
            'port' => 11211,
            'extras' => [
                'timeout' => 3600, // one hour
                'prefix' => 'simplepie_',
            ],
        ];
        $this->options = array_replace_recursive($this->options, \SimplePie\Cache::parse_URL($location));

        $this->name = $this->options['extras']['prefix'] . md5("$name:$type");

        $this->cache = new NativeMemcache();
        $this->cache->addServer($this->options['host'], (int) $this->options['port']);
    }

    /**
     * Save data to the cache
     *
     * @param array<mixed>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data)
    {
        if ($data instanceof \SimplePie\SimplePie) {
            $data = $data->data;
        }
        return $this->cache->set($this->name, serialize($data), MEMCACHE_COMPRESSED, (int) $this->options['extras']['timeout']);
    }

    /**
     * Retrieve the data saved to the cache
     *
     * @return array<mixed>|false Data for SimplePie::$data
     */
    public function load()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            return unserialize($data);
        }
        return false;
    }

    /**
     * Retrieve the last modified time for the cache
     *
     * @return int|false Timestamp
     */
    public function mtime()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            // essentially ignore the mtime because Memcache expires on its own
            return time();
        }

        return false;
    }

    /**
     * Set the last modified time to the current time
     *
     * @return bool Success status
     */
    public function touch()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            return $this->cache->set($this->name, $data, MEMCACHE_COMPRESSED, (int) $this->options['extras']['timeout']);
        }

        return false;
    }

    /**
     * Remove the cache
     *
     * @return bool Success status
     */
    public function unlink()
    {
        return $this->cache->delete($this->name, 0);
    }
}

class_alias('SimplePie\Cache\Memcache', 'SimplePie_Cache_Memcache');

}

namespace SimplePie\Cache {

use Memcached as NativeMemcached;

/**
 * Caches data to memcached
 *
 * Registered for URLs with the "memcached" protocol
 *
 * For example, `memcached://localhost:11211/?timeout=3600&prefix=sp_` will
 * connect to memcached on `localhost` on port 11211. All tables will be
 * prefixed with `sp_` and data will expire after 3600 seconds
 *
 * @uses       Memcached
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
class Memcached implements Base
{
    /**
     * NativeMemcached instance
     * @var NativeMemcached
     */
    protected $cache;

    /**
     * Options
     * @var array<string, mixed>
     */
    protected $options;

    /**
     * Cache name
     * @var string
     */
    protected $name;

    /**
     * Create a new cache object
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $type Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $type)
    {
        $this->options = [
            'host'   => '127.0.0.1',
            'port'   => 11211,
            'extras' => [
                'timeout' => 3600, // one hour
                'prefix'  => 'simplepie_',
            ],
        ];
        $this->options = array_replace_recursive($this->options, \SimplePie\Cache::parse_URL($location));

        $this->name = $this->options['extras']['prefix'] . md5("$name:$type");

        $this->cache = new NativeMemcached();
        $this->cache->addServer($this->options['host'], (int)$this->options['port']);
    }

    /**
     * Save data to the cache
     * @param array<mixed>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data)
    {
        if ($data instanceof \SimplePie\SimplePie) {
            $data = $data->data;
        }

        return $this->setData(serialize($data));
    }

    /**
     * Retrieve the data saved to the cache
     * @return array<mixed>|false Data for SimplePie::$data
     */
    public function load()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            return unserialize($data);
        }
        return false;
    }

    /**
     * Retrieve the last modified time for the cache
     * @return int Timestamp
     */
    public function mtime()
    {
        $data = $this->cache->get($this->name . '_mtime');
        return (int) $data;
    }

    /**
     * Set the last modified time to the current time
     * @return bool Success status
     */
    public function touch()
    {
        $data = $this->cache->get($this->name);
        return $this->setData($data);
    }

    /**
     * Remove the cache
     * @return bool Success status
     */
    public function unlink()
    {
        return $this->cache->delete($this->name, 0);
    }

    /**
     * Set the last modified time and data to NativeMemcached
     * @param string|false $data
     * @return bool Success status
     */
    private function setData($data): bool
    {
        if ($data !== false) {
            $this->cache->set($this->name . '_mtime', time(), (int)$this->options['extras']['timeout']);
            return $this->cache->set($this->name, $data, (int)$this->options['extras']['timeout']);
        }

        return false;
    }
}

class_alias('SimplePie\Cache\Memcached', 'SimplePie_Cache_Memcached');

}

namespace SimplePie\Cache {

/**
 * Caches data to a MySQL database
 *
 * Registered for URLs with the "mysql" protocol
 *
 * For example, `mysql://root:password@localhost:3306/mydb?prefix=sp_` will
 * connect to the `mydb` database on `localhost` on port 3306, with the user
 * `root` and the password `password`. All tables will be prefixed with `sp_`
 *
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
class MySQL extends DB
{
    /**
     * PDO instance
     *
     * @var \PDO|null
     */
    protected $mysql;

    /**
     * Options
     *
     * @var array<string, mixed>
     */
    protected $options;

    /**
     * Cache ID
     *
     * @var string
     */
    protected $id;

    /**
     * Create a new cache object
     *
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE $type Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $type)
    {
        $this->options = [
            'user' => null,
            'pass' => null,
            'host' => '127.0.0.1',
            'port' => '3306',
            'path' => '',
            'extras' => [
                'prefix' => '',
                'cache_purge_time' => 2592000
            ],
        ];

        $this->options = array_replace_recursive($this->options, \SimplePie\Cache::parse_URL($location));

        // Path is prefixed with a "/"
        $this->options['dbname'] = substr($this->options['path'], 1);

        try {
            $this->mysql = new \PDO("mysql:dbname={$this->options['dbname']};host={$this->options['host']};port={$this->options['port']}", $this->options['user'], $this->options['pass'], [\PDO::MYSQL_ATTR_INIT_COMMAND => 'SET NAMES utf8']);
        } catch (\PDOException $e) {
            $this->mysql = null;
            return;
        }

        $this->id = $name . $type;

        if (!$query = $this->mysql->query('SHOW TABLES')) {
            $this->mysql = null;
            return;
        }

        $db = [];
        while ($row = $query->fetchColumn()) {
            $db[] = $row;
        }

        if (!in_array($this->options['extras']['prefix'] . 'cache_data', $db)) {
            $query = $this->mysql->exec('CREATE TABLE `' . $this->options['extras']['prefix'] . 'cache_data` (`id` TEXT CHARACTER SET utf8 NOT NULL, `items` SMALLINT NOT NULL DEFAULT 0, `data` BLOB NOT NULL, `mtime` INT UNSIGNED NOT NULL, UNIQUE (`id`(125)))');
            if ($query === false) {
                trigger_error("Can't create " . $this->options['extras']['prefix'] . "cache_data table, check permissions", \E_USER_WARNING);
                $this->mysql = null;
                return;
            }
        }

        if (!in_array($this->options['extras']['prefix'] . 'items', $db)) {
            $query = $this->mysql->exec('CREATE TABLE `' . $this->options['extras']['prefix'] . 'items` (`feed_id` TEXT CHARACTER SET utf8 NOT NULL, `id` TEXT CHARACTER SET utf8 NOT NULL, `data` MEDIUMBLOB NOT NULL, `posted` INT UNSIGNED NOT NULL, INDEX `feed_id` (`feed_id`(125)))');
            if ($query === false) {
                trigger_error("Can't create " . $this->options['extras']['prefix'] . "items table, check permissions", \E_USER_WARNING);
                $this->mysql = null;
                return;
            }
        }
    }

    /**
     * Save data to the cache
     *
     * @param array<string>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data)
    {
        if ($this->mysql === null) {
            return false;
        }

        $query = $this->mysql->prepare('DELETE i, cd FROM `' . $this->options['extras']['prefix'] . 'cache_data` cd, ' .
            '`' . $this->options['extras']['prefix'] . 'items` i ' .
            'WHERE cd.id = i.feed_id ' .
            'AND cd.mtime < (unix_timestamp() - :purge_time)');
        $query->bindValue(':purge_time', $this->options['extras']['cache_purge_time']);

        if (!$query->execute()) {
            return false;
        }

        if ($data instanceof \SimplePie\SimplePie) {
            $data = clone $data;

            $prepared = self::prepare_simplepie_object_for_cache($data);

            $query = $this->mysql->prepare('SELECT COUNT(*) FROM `' . $this->options['extras']['prefix'] . 'cache_data` WHERE `id` = :feed');
            $query->bindValue(':feed', $this->id);
            if ($query->execute()) {
                if ($query->fetchColumn() > 0) {
                    $items = count($prepared[1]);
                    if ($items) {
                        $sql = 'UPDATE `' . $this->options['extras']['prefix'] . 'cache_data` SET `items` = :items, `data` = :data, `mtime` = :time WHERE `id` = :feed';
                        $query = $this->mysql->prepare($sql);
                        $query->bindValue(':items', $items);
                    } else {
                        $sql = 'UPDATE `' . $this->options['extras']['prefix'] . 'cache_data` SET `data` = :data, `mtime` = :time WHERE `id` = :feed';
                        $query = $this->mysql->prepare($sql);
                    }

                    $query->bindValue(':data', $prepared[0]);
                    $query->bindValue(':time', time());
                    $query->bindValue(':feed', $this->id);
                    if (!$query->execute()) {
                        return false;
                    }
                } else {
                    $query = $this->mysql->prepare('INSERT INTO `' . $this->options['extras']['prefix'] . 'cache_data` (`id`, `items`, `data`, `mtime`) VALUES(:feed, :count, :data, :time)');
                    $query->bindValue(':feed', $this->id);
                    $query->bindValue(':count', count($prepared[1]));
                    $query->bindValue(':data', $prepared[0]);
                    $query->bindValue(':time', time());
                    if (!$query->execute()) {
                        return false;
                    }
                }

                $ids = array_keys($prepared[1]);
                if (!empty($ids)) {
                    foreach ($ids as $id) {
                        $database_ids[] = $this->mysql->quote($id);
                    }

                    $query = $this->mysql->prepare('SELECT `id` FROM `' . $this->options['extras']['prefix'] . 'items` WHERE `id` = ' . implode(' OR `id` = ', $database_ids) . ' AND `feed_id` = :feed');
                    $query->bindValue(':feed', $this->id);

                    if ($query->execute()) {
                        $existing_ids = [];
                        while ($row = $query->fetchColumn()) {
                            $existing_ids[] = $row;
                        }

                        $new_ids = array_diff($ids, $existing_ids);

                        foreach ($new_ids as $new_id) {
                            if (!($date = $prepared[1][$new_id]->get_date('U'))) {
                                $date = time();
                            }

                            $query = $this->mysql->prepare('INSERT INTO `' . $this->options['extras']['prefix'] . 'items` (`feed_id`, `id`, `data`, `posted`) VALUES(:feed, :id, :data, :date)');
                            $query->bindValue(':feed', $this->id);
                            $query->bindValue(':id', $new_id);
                            $query->bindValue(':data', serialize($prepared[1][$new_id]->data));
                            $query->bindValue(':date', $date);
                            if (!$query->execute()) {
                                return false;
                            }
                        }
                        return true;
                    }
                } else {
                    return true;
                }
            }
        } else {
            $query = $this->mysql->prepare('SELECT `id` FROM `' . $this->options['extras']['prefix'] . 'cache_data` WHERE `id` = :feed');
            $query->bindValue(':feed', $this->id);
            if ($query->execute()) {
                if ($query->rowCount() > 0) {
                    $query = $this->mysql->prepare('UPDATE `' . $this->options['extras']['prefix'] . 'cache_data` SET `items` = 0, `data` = :data, `mtime` = :time WHERE `id` = :feed');
                    $query->bindValue(':data', serialize($data));
                    $query->bindValue(':time', time());
                    $query->bindValue(':feed', $this->id);
                    if ($query->execute()) {
                        return true;
                    }
                } else {
                    $query = $this->mysql->prepare('INSERT INTO `' . $this->options['extras']['prefix'] . 'cache_data` (`id`, `items`, `data`, `mtime`) VALUES(:id, 0, :data, :time)');
                    $query->bindValue(':id', $this->id);
                    $query->bindValue(':data', serialize($data));
                    $query->bindValue(':time', time());
                    if ($query->execute()) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Retrieve the data saved to the cache
     *
     * @return array<string>|false Data for SimplePie::$data
     */
    public function load()
    {
        if ($this->mysql === null) {
            return false;
        }

        $query = $this->mysql->prepare('SELECT `items`, `data` FROM `' . $this->options['extras']['prefix'] . 'cache_data` WHERE `id` = :id');
        $query->bindValue(':id', $this->id);
        if ($query->execute() && ($row = $query->fetch())) {
            $data = unserialize($row[1]);

            if (isset($this->options['items'][0])) {
                $items = (int) $this->options['items'][0];
            } else {
                $items = (int) $row[0];
            }

            if ($items !== 0) {
                if (isset($data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['feed'][0])) {
                    $feed = &$data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['feed'][0];
                } elseif (isset($data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['feed'][0])) {
                    $feed = &$data['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['feed'][0];
                } elseif (isset($data['child'][\SimplePie\SimplePie::NAMESPACE_RDF]['RDF'][0])) {
                    $feed = &$data['child'][\SimplePie\SimplePie::NAMESPACE_RDF]['RDF'][0];
                } elseif (isset($data['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['rss'][0])) {
                    $feed = &$data['child'][\SimplePie\SimplePie::NAMESPACE_RSS_20]['rss'][0];
                } else {
                    $feed = null;
                }

                if ($feed !== null) {
                    $sql = 'SELECT `data` FROM `' . $this->options['extras']['prefix'] . 'items` WHERE `feed_id` = :feed ORDER BY `posted` DESC';
                    if ($items > 0) {
                        $sql .= ' LIMIT ' . $items;
                    }

                    $query = $this->mysql->prepare($sql);
                    $query->bindValue(':feed', $this->id);
                    if ($query->execute()) {
                        while ($row = $query->fetchColumn()) {
                            $feed['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['entry'][] = unserialize((string) $row);
                        }
                    } else {
                        return false;
                    }
                }
            }
            return $data;
        }
        return false;
    }

    /**
     * Retrieve the last modified time for the cache
     *
     * @return int|false Timestamp
     */
    public function mtime()
    {
        if ($this->mysql === null) {
            return false;
        }

        $query = $this->mysql->prepare('SELECT `mtime` FROM `' . $this->options['extras']['prefix'] . 'cache_data` WHERE `id` = :id');
        $query->bindValue(':id', $this->id);
        if ($query->execute() && ($time = $query->fetchColumn())) {
            return (int) $time;
        }

        return false;
    }

    /**
     * Set the last modified time to the current time
     *
     * @return bool Success status
     */
    public function touch()
    {
        if ($this->mysql === null) {
            return false;
        }

        $query = $this->mysql->prepare('UPDATE `' . $this->options['extras']['prefix'] . 'cache_data` SET `mtime` = :time WHERE `id` = :id');
        $query->bindValue(':time', time());
        $query->bindValue(':id', $this->id);

        return $query->execute() && $query->rowCount() > 0;
    }

    /**
     * Remove the cache
     *
     * @return bool Success status
     */
    public function unlink()
    {
        if ($this->mysql === null) {
            return false;
        }

        $query = $this->mysql->prepare('DELETE FROM `' . $this->options['extras']['prefix'] . 'cache_data` WHERE `id` = :id');
        $query->bindValue(':id', $this->id);
        $query2 = $this->mysql->prepare('DELETE FROM `' . $this->options['extras']['prefix'] . 'items` WHERE `feed_id` = :id');
        $query2->bindValue(':id', $this->id);

        return $query->execute() && $query2->execute();
    }
}

class_alias('SimplePie\Cache\MySQL', 'SimplePie_Cache_MySQL');

}

namespace SimplePie\Cache {

/**
 * Interface for creating a cache filename
 */
interface NameFilter
{
    /**
     * Method to create cache filename with.
     *
     * The returning name MUST follow the rules for keys in PSR-16.
     *
     * @link https://www.php-fig.org/psr/psr-16/
     *
     * The returning name MUST be a string of at least one character
     * that uniquely identifies a cached item, MUST only contain the
     * characters A-Z, a-z, 0-9, _, and . in any order in UTF-8 encoding
     * and MUST not longer then 64 characters. The following characters
     * are reserved for future extensions and MUST NOT be used: {}()/\@:
     *
     * A provided implementing library MAY support additional characters
     * and encodings or longer lengths, but MUST support at least that
     * minimum.
     *
     * @param string $name The name for the cache will be most likely an url with query string
     *
     * @return string the new cache name
     */
    public function filter(string $name): string;
}

}

namespace SimplePie\Cache {

use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;
use Throwable;

/**
 * Caches data into a PSR-16 cache implementation
 *
 * @internal
 */
final class Psr16 implements DataCache
{
    /**
     * PSR-16 cache implementation
     *
     * @var CacheInterface
     */
    private $cache;

    /**
     * PSR-16 cache implementation
     *
     * @param CacheInterface $cache
     */
    public function __construct(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Fetches a value from the cache.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::get()
     * <code>
     * public function get(string $key, mixed $default = null): mixed;
     * </code>
     *
     * @param string $key     The unique key of this item in the cache.
     * @param mixed  $default Default value to return if the key does not exist.
     *
     * @return array|mixed The value of the item from the cache, or $default in case of cache miss.
     *
     * @throws InvalidArgumentException&Throwable
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function get_data(string $key, $default = null)
    {
        $data = $this->cache->get($key, $default);

        if (!is_array($data) || $data === $default) {
            return $default;
        }

        return $data;
    }

    /**
     * Persists data in the cache, uniquely referenced by a key with an optional expiration TTL time.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::set()
     * <code>
     * public function set(string $key, mixed $value, null|int|\DateInterval $ttl = null): bool;
     * </code>
     *
     * @param string   $key   The key of the item to store.
     * @param array<mixed> $value The value of the item to store, must be serializable.
     * @param null|int $ttl   Optional. The TTL value of this item. If no value is sent and
     *                                      the driver supports TTL then the library may set a default value
     *                                      for it or let the driver take care of that.
     *
     * @return bool True on success and false on failure.
     *
     * @throws InvalidArgumentException&Throwable
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function set_data(string $key, array $value, ?int $ttl = null): bool
    {
        return $this->cache->set($key, $value, $ttl);
    }

    /**
     * Delete an item from the cache by its unique key.
     *
     * Equivalent to \Psr\SimpleCache\CacheInterface::delete()
     * <code>
     * public function delete(string $key): bool;
     * </code>
     *
     * @param string $key The unique cache key of the item to delete.
     *
     * @return bool True if the item was successfully removed. False if there was an error.
     *
     * @throws InvalidArgumentException&Throwable
     *   MUST be thrown if the $key string is not a legal value.
     */
    public function delete_data(string $key): bool
    {
        return $this->cache->delete($key);
    }
}

}

namespace SimplePie\Cache {

use Redis as NativeRedis;

/**
 * Caches data to redis
 *
 * Registered for URLs with the "redis" protocol
 *
 * For example, `redis://localhost:6379/?timeout=3600&prefix=sp_&dbIndex=0` will
 * connect to redis on `localhost` on port 6379. All tables will be
 * prefixed with `simple_primary-` and data will expire after 3600 seconds
 *
 * @uses Redis
 * @deprecated since SimplePie 1.8.0, use implementation of "Psr\SimpleCache\CacheInterface" instead
 */
class Redis implements Base
{
    /**
     * Redis instance
     *
     * @var NativeRedis
     */
    protected $cache;

    /**
     * Options
     *
     * @var array<string, mixed>
     */
    protected $options;

    /**
     * Cache name
     *
     * @var string
     */
    protected $name;

    /**
     * Create a new cache object
     *
     * @param string $location Location string (from SimplePie::$cache_location)
     * @param string $name Unique ID for the cache
     * @param Base::TYPE_FEED|Base::TYPE_IMAGE|array<string, mixed>|null $options Either TYPE_FEED for SimplePie data, or TYPE_IMAGE for image data
     */
    public function __construct(string $location, string $name, $options = null)
    {
        //$this->cache = \flow\simple\cache\Redis::getRedisClientInstance();
        $parsed = \SimplePie\Cache::parse_URL($location);
        $redis = new NativeRedis();
        $redis->connect($parsed['host'], $parsed['port']);
        if (isset($parsed['pass'])) {
            $redis->auth($parsed['pass']);
        }
        if (isset($parsed['path'])) {
            $redis->select((int)substr($parsed['path'], 1));
        }
        $this->cache = $redis;

        if (!is_null($options) && is_array($options)) {
            $this->options = $options;
        } else {
            $this->options = [
                'prefix' => 'rss:simple_primary:',
                'expire' => 0,
            ];
        }

        $this->name = $this->options['prefix'] . $name;
    }

    /**
     * @param NativeRedis $cache
     * @return void
     */
    public function setRedisClient(NativeRedis $cache)
    {
        $this->cache = $cache;
    }

    /**
     * Save data to the cache
     *
     * @param array<mixed>|\SimplePie\SimplePie $data Data to store in the cache. If passed a SimplePie object, only cache the $data property
     * @return bool Successfulness
     */
    public function save($data)
    {
        if ($data instanceof \SimplePie\SimplePie) {
            $data = $data->data;
        }
        $response = $this->cache->set($this->name, serialize($data));
        if ($this->options['expire']) {
            $this->cache->expire($this->name, $this->options['expire']);
        }

        return $response;
    }

    /**
     * Retrieve the data saved to the cache
     *
     * @return array<mixed>|false Data for SimplePie::$data
     */
    public function load()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            return unserialize($data);
        }
        return false;
    }

    /**
     * Retrieve the last modified time for the cache
     *
     * @return int|false Timestamp
     */
    public function mtime()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            return time();
        }

        return false;
    }

    /**
     * Set the last modified time to the current time
     *
     * @return bool Success status
     */
    public function touch()
    {
        $data = $this->cache->get($this->name);

        if ($data !== false) {
            $return = $this->cache->set($this->name, $data);
            if ($this->options['expire']) {
                return $this->cache->expire($this->name, $this->options['expire']);
            }
            return $return;
        }

        return false;
    }

    /**
     * Remove the cache
     *
     * @return bool Success status
     */
    public function unlink()
    {
        return $this->cache->set($this->name, null);
    }
}

class_alias('SimplePie\Cache\Redis', 'SimplePie_Cache_Redis');

}

namespace SimplePie {

/**
 * Handles `<media:text>` captions as defined in Media RSS.
 *
 * Used by {@see \SimplePie\Enclosure::get_caption()} and {@see \SimplePie\Enclosure::get_captions()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_caption_class()}
 */
class Caption
{
    /**
     * Content type
     *
     * @var ?string
     * @see get_type()
     */
    public $type;

    /**
     * Language
     *
     * @var ?string
     * @see get_language()
     */
    public $lang;

    /**
     * Start time
     *
     * @var ?string
     * @see get_starttime()
     */
    public $startTime;

    /**
     * End time
     *
     * @var ?string
     * @see get_endtime()
     */
    public $endTime;

    /**
     * Caption text
     *
     * @var ?string
     * @see get_text()
     */
    public $text;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     */
    public function __construct(
        ?string $type = null,
        ?string $lang = null,
        ?string $startTime = null,
        ?string $endTime = null,
        ?string $text = null
    ) {
        $this->type = $type;
        $this->lang = $lang;
        $this->startTime = $startTime;
        $this->endTime = $endTime;
        $this->text = $text;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the end time
     *
     * @return string|null Time in the format 'hh:mm:ss.SSS'
     */
    public function get_endtime()
    {
        if ($this->endTime !== null) {
            return $this->endTime;
        }

        return null;
    }

    /**
     * Get the language
     *
     * @link http://tools.ietf.org/html/rfc3066
     * @return string|null Language code as per RFC 3066
     */
    public function get_language()
    {
        if ($this->lang !== null) {
            return $this->lang;
        }

        return null;
    }

    /**
     * Get the start time
     *
     * @return string|null Time in the format 'hh:mm:ss.SSS'
     */
    public function get_starttime()
    {
        if ($this->startTime !== null) {
            return $this->startTime;
        }

        return null;
    }

    /**
     * Get the text of the caption
     *
     * @return string|null
     */
    public function get_text()
    {
        if ($this->text !== null) {
            return $this->text;
        }

        return null;
    }

    /**
     * Get the content type (not MIME type)
     *
     * @return string|null Either 'text' or 'html'
     */
    public function get_type()
    {
        if ($this->type !== null) {
            return $this->type;
        }

        return null;
    }
}

class_alias('SimplePie\Caption', 'SimplePie_Caption');

}

namespace SimplePie {

/**
 * Manages all category-related data
 *
 * Used by {@see \SimplePie\Item::get_category()} and {@see \SimplePie\Item::get_categories()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_category_class()}
 */
class Category
{
    /**
     * Category identifier
     *
     * @var string|null
     * @see get_term
     */
    public $term;

    /**
     * Categorization scheme identifier
     *
     * @var string|null
     * @see get_scheme()
     */
    public $scheme;

    /**
     * Human readable label
     *
     * @var string|null
     * @see get_label()
     */
    public $label;

    /**
     * Category type
     *
     * category for <category>
     * subject for <dc:subject>
     *
     * @var string|null
     * @see get_type()
     */
    public $type;

    /**
     * Constructor, used to input the data
     *
     * @param string|null $term
     * @param string|null $scheme
     * @param string|null $label
     * @param string|null $type
     */
    public function __construct(?string $term = null, ?string $scheme = null, ?string $label = null, ?string $type = null)
    {
        $this->term = $term;
        $this->scheme = $scheme;
        $this->label = $label;
        $this->type = $type;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the category identifier
     *
     * @return string|null
     */
    public function get_term()
    {
        return $this->term;
    }

    /**
     * Get the categorization scheme identifier
     *
     * @return string|null
     */
    public function get_scheme()
    {
        return $this->scheme;
    }

    /**
     * Get the human readable label
     *
     * @param bool $strict
     * @return string|null
     */
    public function get_label(bool $strict = false)
    {
        if ($this->label === null && $strict !== true) {
            return $this->get_term();
        }
        return $this->label;
    }

    /**
     * Get the category type
     *
     * @return string|null
     */
    public function get_type()
    {
        return $this->type;
    }
}

class_alias('SimplePie\Category', 'SimplePie_Category');

}

namespace SimplePie\Content\Type {

use InvalidArgumentException;
use SimplePie\File;
use SimplePie\HTTP\Response;

/**
 * Content-type sniffing
 *
 * Based on the rules in http://tools.ietf.org/html/draft-abarth-mime-sniff-06
 *
 * This is used since we can't always trust Content-Type headers, and is based
 * upon the HTML5 parsing rules.
 *
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_content_type_sniffer_class()}
 */
class Sniffer
{
    /**
     * File object
     *
     * @var File|Response
     */
    public $file;

    /**
     * Create an instance of the class with the input file
     *
     * @param File|Response $file Input file
     */
    public function __construct(/* File */ $file)
    {
        if (!is_object($file) || !$file instanceof Response) {
            // For BC we're asking for `File`, but internally we accept every `Response` implementation
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($file) must be of type %s',
                __METHOD__,
                File::class
            ), 1);
        }

        $this->file = $file;
    }

    /**
     * Get the Content-Type of the specified file
     *
     * @return string Actual Content-Type
     */
    public function get_type()
    {
        $content_type = $this->file->has_header('content-type') ? $this->file->get_header_line('content-type') : null;
        $content_encoding = $this->file->has_header('content-encoding') ? $this->file->get_header_line('content-encoding') : null;
        if ($content_type !== null) {
            if ($content_encoding === null
                && ($content_type === 'text/plain'
                    || $content_type === 'text/plain; charset=ISO-8859-1'
                    || $content_type === 'text/plain; charset=iso-8859-1'
                    || $content_type === 'text/plain; charset=UTF-8')) {
                return $this->text_or_binary();
            }

            if (($pos = strpos($content_type, ';')) !== false) {
                $official = substr($content_type, 0, $pos);
            } else {
                $official = $content_type;
            }
            $official = trim(strtolower($official));

            if ($official === 'unknown/unknown'
                || $official === 'application/unknown') {
                return $this->unknown();
            } elseif (substr($official, -4) === '+xml'
                || $official === 'text/xml'
                || $official === 'application/xml') {
                return $official;
            } elseif (substr($official, 0, 6) === 'image/') {
                if ($return = $this->image()) {
                    return $return;
                }

                return $official;
            } elseif ($official === 'text/html') {
                return $this->feed_or_html();
            }

            return $official;
        }

        return $this->unknown();
    }

    /**
     * Sniff text or binary
     *
     * @return string Actual Content-Type
     */
    public function text_or_binary()
    {
        $body = $this->file->get_body_content();

        if (substr($body, 0, 2) === "\xFE\xFF"
            || substr($body, 0, 2) === "\xFF\xFE"
            || substr($body, 0, 4) === "\x00\x00\xFE\xFF"
            || substr($body, 0, 3) === "\xEF\xBB\xBF") {
            return 'text/plain';
        } elseif (preg_match('/[\x00-\x08\x0E-\x1A\x1C-\x1F]/', $body)) {
            return 'application/octet-stream';
        }

        return 'text/plain';
    }

    /**
     * Sniff unknown
     *
     * @return string Actual Content-Type
     */
    public function unknown()
    {
        $body = $this->file->get_body_content();

        $ws = strspn($body, "\x09\x0A\x0B\x0C\x0D\x20");
        if (strtolower(substr($body, $ws, 14)) === '<!doctype html'
            || strtolower(substr($body, $ws, 5)) === '<html'
            || strtolower(substr($body, $ws, 7)) === '<script') {
            return 'text/html';
        } elseif (substr($body, 0, 5) === '%PDF-') {
            return 'application/pdf';
        } elseif (substr($body, 0, 11) === '%!PS-Adobe-') {
            return 'application/postscript';
        } elseif (substr($body, 0, 6) === 'GIF87a'
            || substr($body, 0, 6) === 'GIF89a') {
            return 'image/gif';
        } elseif (substr($body, 0, 8) === "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A") {
            return 'image/png';
        } elseif (substr($body, 0, 3) === "\xFF\xD8\xFF") {
            return 'image/jpeg';
        } elseif (substr($body, 0, 2) === "\x42\x4D") {
            return 'image/bmp';
        } elseif (substr($body, 0, 4) === "\x00\x00\x01\x00") {
            return 'image/vnd.microsoft.icon';
        }

        return $this->text_or_binary();
    }

    /**
     * Sniff images
     *
     * @return string|false Actual Content-Type
     */
    public function image()
    {
        $body = $this->file->get_body_content();

        if (substr($body, 0, 6) === 'GIF87a'
            || substr($body, 0, 6) === 'GIF89a') {
            return 'image/gif';
        } elseif (substr($body, 0, 8) === "\x89\x50\x4E\x47\x0D\x0A\x1A\x0A") {
            return 'image/png';
        } elseif (substr($body, 0, 3) === "\xFF\xD8\xFF") {
            return 'image/jpeg';
        } elseif (substr($body, 0, 2) === "\x42\x4D") {
            return 'image/bmp';
        } elseif (substr($body, 0, 4) === "\x00\x00\x01\x00") {
            return 'image/vnd.microsoft.icon';
        }

        return false;
    }

    /**
     * Sniff HTML
     *
     * @return string Actual Content-Type
     */
    public function feed_or_html()
    {
        $body = $this->file->get_body_content();

        $len = strlen($body);
        $pos = strspn($body, "\x09\x0A\x0D\x20\xEF\xBB\xBF");

        while ($pos < $len) {
            switch ($body[$pos]) {
                case "\x09":
                case "\x0A":
                case "\x0D":
                case "\x20":
                    $pos += strspn($body, "\x09\x0A\x0D\x20", $pos);
                    continue 2;

                case '<':
                    $pos++;
                    break;

                default:
                    return 'text/html';
            }

            if (substr($body, $pos, 3) === '!--') {
                $pos += 3;
                if ($pos < $len && ($pos = strpos($body, '-->', $pos)) !== false) {
                    $pos += 3;
                } else {
                    return 'text/html';
                }
            } elseif (substr($body, $pos, 1) === '!') {
                if ($pos < $len && ($pos = strpos($body, '>', $pos)) !== false) {
                    $pos++;
                } else {
                    return 'text/html';
                }
            } elseif (substr($body, $pos, 1) === '?') {
                if ($pos < $len && ($pos = strpos($body, '?>', $pos)) !== false) {
                    $pos += 2;
                } else {
                    return 'text/html';
                }
            } elseif (substr($body, $pos, 3) === 'rss'
                || substr($body, $pos, 7) === 'rdf:RDF') {
                return 'application/rss+xml';
            } elseif (substr($body, $pos, 4) === 'feed') {
                return 'application/atom+xml';
            } else {
                return 'text/html';
            }
        }

        return 'text/html';
    }
}

class_alias('SimplePie\Content\Type\Sniffer', 'SimplePie_Content_Type_Sniffer');

}

namespace SimplePie {

/**
 * Manages `<media:copyright>` copyright tags as defined in Media RSS
 *
 * Used by {@see \SimplePie\Enclosure::get_copyright()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_copyright_class()}
 */
class Copyright
{
    /**
     * Copyright URL
     *
     * @var ?string
     * @see get_url()
     */
    public $url;

    /**
     * Attribution
     *
     * @var ?string
     * @see get_attribution()
     */
    public $label;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     */
    public function __construct(
        ?string $url = null,
        ?string $label = null
    ) {
        $this->url = $url;
        $this->label = $label;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the copyright URL
     *
     * @return string|null URL to copyright information
     */
    public function get_url()
    {
        if ($this->url !== null) {
            return $this->url;
        }

        return null;
    }

    /**
     * Get the attribution text
     *
     * @return string|null
     */
    public function get_attribution()
    {
        if ($this->label !== null) {
            return $this->label;
        }

        return null;
    }
}

class_alias('SimplePie\Copyright', 'SimplePie_Copyright');

}

namespace SimplePie {

/**
 * Handles `<media:credit>` as defined in Media RSS
 *
 * Used by {@see \SimplePie\Enclosure::get_credit()} and {@see \SimplePie\Enclosure::get_credits()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_credit_class()}
 */
class Credit
{
    /**
     * Credited role
     *
     * @var ?string
     * @see get_role()
     */
    public $role;

    /**
     * Organizational scheme
     *
     * @var ?string
     * @see get_scheme()
     */
    public $scheme;

    /**
     * Credited name
     *
     * @var ?string
     * @see get_name()
     */
    public $name;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     */
    public function __construct(
        ?string $role = null,
        ?string $scheme = null,
        ?string $name = null
    ) {
        $this->role = $role;
        $this->scheme = $scheme;
        $this->name = $name;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the role of the person receiving credit
     *
     * @return string|null
     */
    public function get_role()
    {
        if ($this->role !== null) {
            return $this->role;
        }

        return null;
    }

    /**
     * Get the organizational scheme
     *
     * @return string|null
     */
    public function get_scheme()
    {
        if ($this->scheme !== null) {
            return $this->scheme;
        }

        return null;
    }

    /**
     * Get the credited person/entity's name
     *
     * @return string|null
     */
    public function get_name()
    {
        if ($this->name !== null) {
            return $this->name;
        }

        return null;
    }
}

class_alias('SimplePie\Credit', 'SimplePie_Credit');

}

namespace SimplePie {

/**
 * Handles everything related to enclosures (including Media RSS and iTunes RSS)
 *
 * Used by {@see \SimplePie\Item::get_enclosure()} and {@see \SimplePie\Item::get_enclosures()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_enclosure_class()}
 */
class Enclosure
{
    /**
     * @var ?string
     * @see get_bitrate()
     */
    public $bitrate;

    /**
     * @var Caption[]|null
     * @see get_captions()
     */
    public $captions;

    /**
     * @var Category[]|null
     * @see get_categories()
     */
    public $categories;

    /**
     * @var ?int
     * @see get_channels()
     */
    public $channels;

    /**
     * @var ?Copyright
     * @see get_copyright()
     */
    public $copyright;

    /**
     * @var Credit[]|null
     * @see get_credits()
     */
    public $credits;

    /**
     * @var ?string
     * @see get_description()
     */
    public $description;

    /**
     * @var ?int
     * @see get_duration()
     */
    public $duration;

    /**
     * @var ?string
     * @see get_expression()
     */
    public $expression;

    /**
     * @var ?string
     * @see get_framerate()
     */
    public $framerate;

    /**
     * @var ?string
     * @see get_handler()
     */
    public $handler;

    /**
     * @var string[]|null
     * @see get_hashes()
     */
    public $hashes;

    /**
     * @var ?string
     * @see get_height()
     */
    public $height;

    /**
     * @deprecated
     * @var null
     */
    public $javascript;

    /**
     * @var string[]|null
     * @see get_keywords()
     */
    public $keywords;

    /**
     * @var ?string
     * @see get_language()
     */
    public $lang;

    /**
     * @var ?int
     * @see get_length()
     */
    public $length;

    /**
     * @var ?string
     * @see get_link()
     */
    public $link;

    /**
     * @var ?string
     * @see get_medium()
     */
    public $medium;

    /**
     * @var ?string
     * @see get_player()
     */
    public $player;

    /**
     * @var Rating[]|null
     * @see get_ratings()
     */
    public $ratings;

    /**
     * @var ?Restriction[]
     * @see get_restrictions()
     */
    public $restrictions;

    /**
     * @var ?string
     * @see get_sampling_rate()
     */
    public $samplingrate;

    /**
     * @var string[]|null
     * @see get_thumbnails()
     */
    public $thumbnails;

    /**
     * @var ?string
     * @see get_title()
     */
    public $title;

    /**
     * @var ?string
     * @see get_type()
     */
    public $type;

    /**
     * @var ?string
     * @see get_width()
     */
    public $width;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     *
     * @uses idn_to_ascii If available, this will convert an IDN
     *
     * @param null $javascript
     * @param Caption[]|null $captions
     * @param Category[]|null $categories
     * @param Credit[]|null $credits
     * @param string[]|null $hashes
     * @param string[]|null $keywords
     * @param Rating[]|null $ratings
     * @param Restriction[]|null $restrictions
     * @param string[]|null $thumbnails
     */
    public function __construct(
        ?string $link = null,
        ?string $type = null,
        ?int $length = null,
        $javascript = null,
        ?string $bitrate = null,
        ?array $captions = null,
        ?array $categories = null,
        ?int $channels = null,
        ?Copyright $copyright = null,
        ?array $credits = null,
        ?string $description = null,
        ?int $duration = null,
        ?string $expression = null,
        ?string $framerate = null,
        ?array $hashes = null,
        ?string $height = null,
        ?array $keywords = null,
        ?string $lang = null,
        ?string $medium = null,
        ?string $player = null,
        ?array $ratings = null,
        ?array $restrictions = null,
        ?string $samplingrate = null,
        ?array $thumbnails = null,
        ?string $title = null,
        ?string $width = null
    ) {
        $this->bitrate = $bitrate;
        $this->captions = $captions;
        $this->categories = $categories;
        $this->channels = $channels;
        $this->copyright = $copyright;
        $this->credits = $credits;
        $this->description = $description;
        $this->duration = $duration;
        $this->expression = $expression;
        $this->framerate = $framerate;
        $this->hashes = $hashes;
        $this->height = $height;
        $this->keywords = $keywords;
        $this->lang = $lang;
        $this->length = $length;
        $this->link = $link;
        $this->medium = $medium;
        $this->player = $player;
        $this->ratings = $ratings;
        $this->restrictions = $restrictions;
        $this->samplingrate = $samplingrate;
        $this->thumbnails = $thumbnails;
        $this->title = $title;
        $this->type = $type;
        $this->width = $width;

        if (function_exists('idn_to_ascii')) {
            $parsed = \SimplePie\Misc::parse_url($link ?? '');
            if ($parsed['authority'] !== '' && !ctype_print($parsed['authority'])) {
                $authority = (string) \idn_to_ascii($parsed['authority'], \IDNA_NONTRANSITIONAL_TO_ASCII, \INTL_IDNA_VARIANT_UTS46);
                $this->link = \SimplePie\Misc::compress_parse_url($parsed['scheme'], $authority, $parsed['path'], $parsed['query'], $parsed['fragment']);
            }
        }
        $this->handler = $this->get_handler(); // Needs to load last
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the bitrate
     *
     * @return string|null
     */
    public function get_bitrate()
    {
        if ($this->bitrate !== null) {
            return $this->bitrate;
        }

        return null;
    }

    /**
     * Get a single caption
     *
     * @param int $key
     * @return \SimplePie\Caption|null
     */
    public function get_caption(int $key = 0)
    {
        $captions = $this->get_captions();
        if (isset($captions[$key])) {
            return $captions[$key];
        }

        return null;
    }

    /**
     * Get all captions
     *
     * @return Caption[]|null
     */
    public function get_captions()
    {
        if ($this->captions !== null) {
            return $this->captions;
        }

        return null;
    }

    /**
     * Get a single category
     *
     * @param int $key
     * @return \SimplePie\Category|null
     */
    public function get_category(int $key = 0)
    {
        $categories = $this->get_categories();
        if (isset($categories[$key])) {
            return $categories[$key];
        }

        return null;
    }

    /**
     * Get all categories
     *
     * @return \SimplePie\Category[]|null
     */
    public function get_categories()
    {
        if ($this->categories !== null) {
            return $this->categories;
        }

        return null;
    }

    /**
     * Get the number of audio channels
     *
     * @return int|null
     */
    public function get_channels()
    {
        if ($this->channels !== null) {
            return $this->channels;
        }

        return null;
    }

    /**
     * Get the copyright information
     *
     * @return \SimplePie\Copyright|null
     */
    public function get_copyright()
    {
        if ($this->copyright !== null) {
            return $this->copyright;
        }

        return null;
    }

    /**
     * Get a single credit
     *
     * @param int $key
     * @return \SimplePie\Credit|null
     */
    public function get_credit(int $key = 0)
    {
        $credits = $this->get_credits();
        if (isset($credits[$key])) {
            return $credits[$key];
        }

        return null;
    }

    /**
     * Get all credits
     *
     * @return Credit[]|null
     */
    public function get_credits()
    {
        if ($this->credits !== null) {
            return $this->credits;
        }

        return null;
    }

    /**
     * Get the description of the enclosure
     *
     * @return string|null
     */
    public function get_description()
    {
        if ($this->description !== null) {
            return $this->description;
        }

        return null;
    }

    /**
     * Get the duration of the enclosure
     *
     * @param bool $convert Convert seconds into hh:mm:ss
     * @return string|int|null 'hh:mm:ss' string if `$convert` was specified, otherwise integer (or null if none found)
     */
    public function get_duration(bool $convert = false)
    {
        if ($this->duration !== null) {
            if ($convert) {
                $time = \SimplePie\Misc::time_hms($this->duration);
                return $time;
            }

            return $this->duration;
        }

        return null;
    }

    /**
     * Get the expression
     *
     * @return string Probably one of 'sample', 'full', 'nonstop', 'clip'. Defaults to 'full'
     */
    public function get_expression()
    {
        if ($this->expression !== null) {
            return $this->expression;
        }

        return 'full';
    }

    /**
     * Get the file extension
     *
     * @return string|null
     */
    public function get_extension()
    {
        if ($this->link !== null) {
            $url = \SimplePie\Misc::parse_url($this->link);
            if ($url['path'] !== '') {
                return pathinfo($url['path'], PATHINFO_EXTENSION);
            }
        }
        return null;
    }

    /**
     * Get the framerate (in frames-per-second)
     *
     * @return string|null
     */
    public function get_framerate()
    {
        if ($this->framerate !== null) {
            return $this->framerate;
        }

        return null;
    }

    /**
     * Get the preferred handler
     *
     * @return string|null One of 'flash', 'fmedia', 'quicktime', 'wmedia', 'mp3'
     */
    public function get_handler()
    {
        return $this->get_real_type(true);
    }

    /**
     * Get a single hash
     *
     * @link http://www.rssboard.org/media-rss#media-hash
     * @param int $key
     * @return string|null Hash as per `media:hash`, prefixed with "$algo:"
     */
    public function get_hash(int $key = 0)
    {
        $hashes = $this->get_hashes();
        if (isset($hashes[$key])) {
            return $hashes[$key];
        }

        return null;
    }

    /**
     * Get all credits
     *
     * @return string[]|null Array of strings, see {@see get_hash()}
     */
    public function get_hashes()
    {
        if ($this->hashes !== null) {
            return $this->hashes;
        }

        return null;
    }

    /**
     * Get the height
     *
     * @return string|null
     */
    public function get_height()
    {
        if ($this->height !== null) {
            return $this->height;
        }

        return null;
    }

    /**
     * Get the language
     *
     * @link http://tools.ietf.org/html/rfc3066
     * @return string|null Language code as per RFC 3066
     */
    public function get_language()
    {
        if ($this->lang !== null) {
            return $this->lang;
        }

        return null;
    }

    /**
     * Get a single keyword
     *
     * @param int $key
     * @return string|null
     */
    public function get_keyword(int $key = 0)
    {
        $keywords = $this->get_keywords();
        if (isset($keywords[$key])) {
            return $keywords[$key];
        }

        return null;
    }

    /**
     * Get all keywords
     *
     * @return string[]|null
     */
    public function get_keywords()
    {
        if ($this->keywords !== null) {
            return $this->keywords;
        }

        return null;
    }

    /**
     * Get length
     *
     * @return ?int Length in bytes
     */
    public function get_length()
    {
        if ($this->length !== null) {
            return $this->length;
        }

        return null;
    }

    /**
     * Get the URL
     *
     * @return string|null
     */
    public function get_link()
    {
        if ($this->link !== null) {
            return $this->link;
        }

        return null;
    }

    /**
     * Get the medium
     *
     * @link http://www.rssboard.org/media-rss#media-content
     * @return string|null Should be one of 'image', 'audio', 'video', 'document', 'executable'
     */
    public function get_medium()
    {
        if ($this->medium !== null) {
            return $this->medium;
        }

        return null;
    }

    /**
     * Get the player URL
     *
     * Typically the same as {@see get_permalink()}
     * @return string|null Player URL
     */
    public function get_player()
    {
        if ($this->player !== null) {
            return $this->player;
        }

        return null;
    }

    /**
     * Get a single rating
     *
     * @param int $key
     * @return \SimplePie\Rating|null
     */
    public function get_rating(int $key = 0)
    {
        $ratings = $this->get_ratings();
        if (isset($ratings[$key])) {
            return $ratings[$key];
        }

        return null;
    }

    /**
     * Get all ratings
     *
     * @return Rating[]|null
     */
    public function get_ratings()
    {
        if ($this->ratings !== null) {
            return $this->ratings;
        }

        return null;
    }

    /**
     * Get a single restriction
     *
     * @param int $key
     * @return \SimplePie\Restriction|null
     */
    public function get_restriction(int $key = 0)
    {
        $restrictions = $this->get_restrictions();
        if (isset($restrictions[$key])) {
            return $restrictions[$key];
        }

        return null;
    }

    /**
     * Get all restrictions
     *
     * @return Restriction[]|null
     */
    public function get_restrictions()
    {
        if ($this->restrictions !== null) {
            return $this->restrictions;
        }

        return null;
    }

    /**
     * Get the sampling rate (in kHz)
     *
     * @return string|null
     */
    public function get_sampling_rate()
    {
        if ($this->samplingrate !== null) {
            return $this->samplingrate;
        }

        return null;
    }

    /**
     * Get the file size (in MiB)
     *
     * @return float|null File size in mebibytes (1048 bytes)
     */
    public function get_size()
    {
        $length = $this->get_length();
        if ($length !== null) {
            return round($length / 1048576, 2);
        }

        return null;
    }

    /**
     * Get a single thumbnail
     *
     * @param int $key
     * @return string|null Thumbnail URL
     */
    public function get_thumbnail(int $key = 0)
    {
        $thumbnails = $this->get_thumbnails();
        if (isset($thumbnails[$key])) {
            return $thumbnails[$key];
        }

        return null;
    }

    /**
     * Get all thumbnails
     *
     * @return string[]|null Array of thumbnail URLs
     */
    public function get_thumbnails()
    {
        if ($this->thumbnails !== null) {
            return $this->thumbnails;
        }

        return null;
    }

    /**
     * Get the title
     *
     * @return string|null
     */
    public function get_title()
    {
        if ($this->title !== null) {
            return $this->title;
        }

        return null;
    }

    /**
     * Get mimetype of the enclosure
     *
     * @see get_real_type()
     * @return string|null MIME type
     */
    public function get_type()
    {
        if ($this->type !== null) {
            return $this->type;
        }

        return null;
    }

    /**
     * Get the width
     *
     * @return string|null
     */
    public function get_width()
    {
        if ($this->width !== null) {
            return $this->width;
        }

        return null;
    }

    /**
     * Embed the enclosure using `<embed>`
     *
     * @deprecated Use the second parameter to {@see embed} instead
     *
     * @param array<string, mixed>|string $options See first parameter to {@see embed}
     * @return string HTML string to output
     */
    public function native_embed($options = '')
    {
        return $this->embed($options, true);
    }

    /**
     * Embed the enclosure using Javascript
     *
     * `$options` is an array or comma-separated key:value string, with the
     * following properties:
     *
     * - `alt` (string): Alternate content for when an end-user does not have
     *    the appropriate handler installed or when a file type is
     *    unsupported. Can be any text or HTML. Defaults to blank.
     * - `altclass` (string): If a file type is unsupported, the end-user will
     *    see the alt text (above) linked directly to the content. That link
     *    will have this value as its class name. Defaults to blank.
     * - `audio` (string): This is an image that should be used as a
     *    placeholder for audio files before they're loaded (QuickTime-only).
     *    Can be any relative or absolute URL. Defaults to blank.
     * - `bgcolor` (string): The background color for the media, if not
     *    already transparent. Defaults to `#ffffff`.
     * - `height` (integer): The height of the embedded media. Accepts any
     *    numeric pixel value (such as `360`) or `auto`. Defaults to `auto`,
     *    and it is recommended that you use this default.
     * - `loop` (boolean): Do you want the media to loop when it's done?
     *    Defaults to `false`.
     * - `mediaplayer` (string): The location of the included
     *    `mediaplayer.swf` file. This allows for the playback of Flash Video
     *    (`.flv`) files, and is the default handler for non-Odeo MP3's.
     *    Defaults to blank.
     * - `video` (string): This is an image that should be used as a
     *    placeholder for video files before they're loaded (QuickTime-only).
     *    Can be any relative or absolute URL. Defaults to blank.
     * - `width` (integer): The width of the embedded media. Accepts any
     *    numeric pixel value (such as `480`) or `auto`. Defaults to `auto`,
     *    and it is recommended that you use this default.
     * - `widescreen` (boolean): Is the enclosure widescreen or standard?
     *    This applies only to video enclosures, and will automatically resize
     *    the content appropriately.  Defaults to `false`, implying 4:3 mode.
     *
     * Note: Non-widescreen (4:3) mode with `width` and `height` set to `auto`
     * will default to 480x360 video resolution.  Widescreen (16:9) mode with
     * `width` and `height` set to `auto` will default to 480x270 video resolution.
     *
     * @todo If the dimensions for media:content are defined, use them when width/height are set to 'auto'.
     * @param array<string, mixed>|string $options Comma-separated key:value list, or array
     * @param bool $native Use `<embed>`
     * @return string HTML string to output
     */
    public function embed($options = '', bool $native = false)
    {
        // Set up defaults
        $audio = '';
        $video = '';
        $alt = '';
        $altclass = '';
        $loop = 'false';
        $width = 'auto';
        $height = 'auto';
        $bgcolor = '#ffffff';
        $mediaplayer = '';
        $widescreen = false;
        $handler = $this->get_handler();
        $type = $this->get_real_type();
        $placeholder = '';

        // Process options and reassign values as necessary
        if (is_array($options)) {
            extract($options);
        } else {
            $options = explode(',', $options);
            foreach ($options as $option) {
                $opt = explode(':', $option, 2);
                if (isset($opt[0], $opt[1])) {
                    $opt[0] = trim($opt[0]);
                    $opt[1] = trim($opt[1]);
                    switch ($opt[0]) {
                        case 'audio':
                            $audio = $opt[1];
                            break;

                        case 'video':
                            $video = $opt[1];
                            break;

                        case 'alt':
                            $alt = $opt[1];
                            break;

                        case 'altclass':
                            $altclass = $opt[1];
                            break;

                        case 'loop':
                            $loop = $opt[1];
                            break;

                        case 'width':
                            $width = $opt[1];
                            break;

                        case 'height':
                            $height = $opt[1];
                            break;

                        case 'bgcolor':
                            $bgcolor = $opt[1];
                            break;

                        case 'mediaplayer':
                            $mediaplayer = $opt[1];
                            break;

                        case 'widescreen':
                            $widescreen = $opt[1];
                            break;
                    }
                }
            }
        }

        $mime = explode('/', (string) $type, 2);
        $mime = $mime[0];

        // Process values for 'auto'
        if ($width === 'auto') {
            if ($mime === 'video') {
                if ($height === 'auto') {
                    $width = 480;
                } elseif ($widescreen) {
                    $width = round((intval($height) / 9) * 16);
                } else {
                    $width = round((intval($height) / 3) * 4);
                }
            } else {
                $width = '100%';
            }
        }

        if ($height === 'auto') {
            if ($mime === 'audio') {
                $height = 0;
            } elseif ($mime === 'video') {
                if ($width === 'auto') {
                    if ($widescreen) {
                        $height = 270;
                    } else {
                        $height = 360;
                    }
                } elseif ($widescreen) {
                    $height = round((intval($width) / 16) * 9);
                } else {
                    $height = round((intval($width) / 4) * 3);
                }
            } else {
                $height = 376;
            }
        } elseif ($mime === 'audio') {
            $height = 0;
        }

        // Set proper placeholder value
        if ($mime === 'audio') {
            $placeholder = $audio;
        } elseif ($mime === 'video') {
            $placeholder = $video;
        }

        $embed = '';

        // Flash
        if ($handler === 'flash') {
            if ($native) {
                $embed .= "<embed src=\"" . $this->get_link() . "\" pluginspage=\"http://adobe.com/go/getflashplayer\" type=\"$type\" quality=\"high\" width=\"$width\" height=\"$height\" bgcolor=\"$bgcolor\" loop=\"$loop\"></embed>";
            } else {
                $embed .= "<script type='text/javascript'>embed_flash('$bgcolor', '$width', '$height', '" . $this->get_link() . "', '$loop', '$type');</script>";
            }
        }

        // Flash Media Player file types.
        // Preferred handler for MP3 file types.
        elseif ($handler === 'fmedia' || ($handler === 'mp3' && $mediaplayer !== '')) {
            if (is_numeric($height)) {
                $height += 20;
            }

            if ($native) {
                $embed .= "<embed src=\"$mediaplayer\" pluginspage=\"http://adobe.com/go/getflashplayer\" type=\"application/x-shockwave-flash\" quality=\"high\" width=\"$width\" height=\"$height\" wmode=\"transparent\" flashvars=\"file=" . rawurlencode($this->get_link().'?file_extension=.'.$this->get_extension()) . "&autostart=false&repeat=$loop&showdigits=true&showfsbutton=false\"></embed>";
            } else {
                $embed .= "<script type='text/javascript'>embed_flv('$width', '$height', '" . rawurlencode($this->get_link().'?file_extension=.'.$this->get_extension()) . "', '$placeholder', '$loop', '$mediaplayer');</script>";
            }
        }

        // QuickTime 7 file types.  Need to test with QuickTime 6.
        // Only handle MP3's if the Flash Media Player is not present.
        elseif ($handler === 'quicktime' || ($handler === 'mp3' && $mediaplayer === '')) {
            if (is_numeric($height)) {
                $height += 16;
            }

            if ($native) {
                if ($placeholder !== '') {
                    $embed .= "<embed type=\"$type\" style=\"cursor:hand; cursor:pointer;\" href=\"" . $this->get_link() . "\" src=\"$placeholder\" width=\"$width\" height=\"$height\" autoplay=\"false\" target=\"myself\" controller=\"false\" loop=\"$loop\" scale=\"aspect\" bgcolor=\"$bgcolor\" pluginspage=\"http://apple.com/quicktime/download/\"></embed>";
                } else {
                    $embed .= "<embed type=\"$type\" style=\"cursor:hand; cursor:pointer;\" src=\"" . $this->get_link() . "\" width=\"$width\" height=\"$height\" autoplay=\"false\" target=\"myself\" controller=\"true\" loop=\"$loop\" scale=\"aspect\" bgcolor=\"$bgcolor\" pluginspage=\"http://apple.com/quicktime/download/\"></embed>";
                }
            } else {
                $embed .= "<script type='text/javascript'>embed_quicktime('$type', '$bgcolor', '$width', '$height', '" . $this->get_link() . "', '$placeholder', '$loop');</script>";
            }
        }

        // Windows Media
        elseif ($handler === 'wmedia') {
            if (is_numeric($height)) {
                $height += 45;
            }

            if ($native) {
                $embed .= "<embed type=\"application/x-mplayer2\" src=\"" . $this->get_link() . "\" autosize=\"1\" width=\"$width\" height=\"$height\" showcontrols=\"1\" showstatusbar=\"0\" showdisplay=\"0\" autostart=\"0\"></embed>";
            } else {
                $embed .= "<script type='text/javascript'>embed_wmedia('$width', '$height', '" . $this->get_link() . "');</script>";
            }
        }

        // Everything else
        else {
            $embed .= '<a href="' . $this->get_link() . '" class="' . $altclass . '">' . $alt . '</a>';
        }

        return $embed;
    }

    /**
     * Get the real media type
     *
     * Often, feeds lie to us, necessitating a bit of deeper inspection. This
     * converts types to their canonical representations based on the file
     * extension
     *
     * @see get_type()
     * @param bool $find_handler Internal use only, use {@see get_handler()} instead
     * @return string|null MIME type
     */
    public function get_real_type(bool $find_handler = false)
    {
        // Mime-types by handler.
        $types_flash = ['application/x-shockwave-flash', 'application/futuresplash']; // Flash
        $types_fmedia = ['video/flv', 'video/x-flv','flv-application/octet-stream']; // Flash Media Player
        $types_quicktime = ['audio/3gpp', 'audio/3gpp2', 'audio/aac', 'audio/x-aac', 'audio/aiff', 'audio/x-aiff', 'audio/mid', 'audio/midi', 'audio/x-midi', 'audio/mp4', 'audio/m4a', 'audio/x-m4a', 'audio/wav', 'audio/x-wav', 'video/3gpp', 'video/3gpp2', 'video/m4v', 'video/x-m4v', 'video/mp4', 'video/mpeg', 'video/x-mpeg', 'video/quicktime', 'video/sd-video']; // QuickTime
        $types_wmedia = ['application/asx', 'application/x-mplayer2', 'audio/x-ms-wma', 'audio/x-ms-wax', 'video/x-ms-asf-plugin', 'video/x-ms-asf', 'video/x-ms-wm', 'video/x-ms-wmv', 'video/x-ms-wvx']; // Windows Media
        $types_mp3 = ['audio/mp3', 'audio/x-mp3', 'audio/mpeg', 'audio/x-mpeg']; // MP3

        $type = $this->get_type();
        if ($type !== null) {
            $type = strtolower($type);
        }

        // If we encounter an unsupported mime-type, check the file extension and guess intelligently.
        if (!in_array($type, array_merge($types_flash, $types_fmedia, $types_quicktime, $types_wmedia, $types_mp3))) {
            $extension = $this->get_extension();
            if ($extension === null) {
                return null;
            }

            switch (strtolower($extension)) {
                // Audio mime-types
                case 'aac':
                case 'adts':
                    $type = 'audio/acc';
                    break;

                case 'aif':
                case 'aifc':
                case 'aiff':
                case 'cdda':
                    $type = 'audio/aiff';
                    break;

                case 'bwf':
                    $type = 'audio/wav';
                    break;

                case 'kar':
                case 'mid':
                case 'midi':
                case 'smf':
                    $type = 'audio/midi';
                    break;

                case 'm4a':
                    $type = 'audio/x-m4a';
                    break;

                case 'mp3':
                case 'swa':
                    $type = 'audio/mp3';
                    break;

                case 'wav':
                    $type = 'audio/wav';
                    break;

                case 'wax':
                    $type = 'audio/x-ms-wax';
                    break;

                case 'wma':
                    $type = 'audio/x-ms-wma';
                    break;

                case '3gp':
                case '3gpp':
                    // Video mime-types
                    $type = 'video/3gpp';
                    break;

                case '3g2':
                case '3gp2':
                    $type = 'video/3gpp2';
                    break;

                case 'asf':
                    $type = 'video/x-ms-asf';
                    break;

                case 'flv':
                    $type = 'video/x-flv';
                    break;

                case 'm1a':
                case 'm1s':
                case 'm1v':
                case 'm15':
                case 'm75':
                case 'mp2':
                case 'mpa':
                case 'mpeg':
                case 'mpg':
                case 'mpm':
                case 'mpv':
                    $type = 'video/mpeg';
                    break;

                case 'm4v':
                    $type = 'video/x-m4v';
                    break;

                case 'mov':
                case 'qt':
                    $type = 'video/quicktime';
                    break;

                case 'mp4':
                case 'mpg4':
                    $type = 'video/mp4';
                    break;

                case 'sdv':
                    $type = 'video/sd-video';
                    break;

                case 'wm':
                    $type = 'video/x-ms-wm';
                    break;

                case 'wmv':
                    $type = 'video/x-ms-wmv';
                    break;

                case 'wvx':
                    $type = 'video/x-ms-wvx';
                    break;

                case 'spl':
                    // Flash mime-types
                    $type = 'application/futuresplash';
                    break;

                case 'swf':
                    $type = 'application/x-shockwave-flash';
                    break;
            }
        }

        if ($find_handler) {
            if (in_array($type, $types_flash)) {
                return 'flash';
            } elseif (in_array($type, $types_fmedia)) {
                return 'fmedia';
            } elseif (in_array($type, $types_quicktime)) {
                return 'quicktime';
            } elseif (in_array($type, $types_wmedia)) {
                return 'wmedia';
            } elseif (in_array($type, $types_mp3)) {
                return 'mp3';
            }

            return null;
        }

        return $type;
    }
}

class_alias('SimplePie\Enclosure', 'SimplePie_Enclosure');

}

namespace SimplePie {

use Exception as NativeException;

/**
 * General SimplePie exception class
 */
class Exception extends NativeException
{
}

class_alias('SimplePie\Exception', 'SimplePie_Exception');

}

namespace SimplePie {

use SimplePie\HTTP\Response;

/**
 * Used for fetching remote files and reading local files
 *
 * Supports HTTP 1.0 via cURL or fsockopen, with spotty HTTP 1.1 support
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_file_class()}
 *
 * @todo Move to properly supporting RFC2616 (HTTP/1.1)
 */
class File implements Response
{
    /**
     * @var string The final URL after following all redirects
     * @deprecated Use `get_final_requested_uri()` method.
     */
    public $url;

    /**
     * @var ?string User agent to use in requests
     * @deprecated Set the user agent in constructor.
     */
    public $useragent;

    /** @var bool */
    public $success = true;

    /** @var array<string, non-empty-array<string>> Canonical representation of headers */
    private $parsed_headers = [];
    /** @var array<string, string> Last known value of $headers property (used to detect external modification) */
    private $last_headers = [];
    /**
     * @var array<string, string> Headers as string for BC
     * @deprecated Use `get_headers()` method.
     */
    public $headers = [];

    /**
     * @var ?string Body of the HTTP response
     * @deprecated Use `get_body_content()` method.
     */
    public $body;

    /**
     * @var int Status code of the HTTP response
     * @deprecated Use `get_status_code()` method.
     */
    public $status_code = 0;

    /** @var non-negative-int Number of redirect that were already performed during this request sequence. */
    public $redirects = 0;

    /** @var ?string */
    public $error;

    /**
     * @var int-mask-of<SimplePie::FILE_SOURCE_*> Bit mask representing the method used to fetch the file and whether it is a local file or remote file obtained over HTTP.
     * @deprecated Backend is implementation detail which you should not care about; to see if the file was retrieved over HTTP, check if `get_final_requested_uri()` with `Misc::is_remote_uri()`.
     */
    public $method = \SimplePie\SimplePie::FILE_SOURCE_NONE;

    /**
     * @var string The permanent URL or the resource (first URL after the prefix of (only) permanent redirects)
     * @deprecated Use `get_permanent_uri()` method.
     */
    public $permanent_url;
    /** @var bool Whether the permanent URL is still writeable (prefix of permanent redirects has not ended) */
    private $permanentUrlMutable = true;

    /**
     * @param string $url
     * @param int $timeout
     * @param int $redirects
     * @param ?array<string, string> $headers
     * @param ?string $useragent
     * @param bool $force_fsockopen
     * @param array<int, mixed> $curl_options
     */
    public function __construct(string $url, int $timeout = 10, int $redirects = 5, ?array $headers = null, ?string $useragent = null, bool $force_fsockopen = false, array $curl_options = [])
    {
        if (function_exists('idn_to_ascii')) {
            $parsed = \SimplePie\Misc::parse_url($url);
            if ($parsed['authority'] !== '' && !ctype_print($parsed['authority'])) {
                $authority = (string) \idn_to_ascii($parsed['authority'], \IDNA_NONTRANSITIONAL_TO_ASCII, \INTL_IDNA_VARIANT_UTS46);
                $url = \SimplePie\Misc::compress_parse_url($parsed['scheme'], $authority, $parsed['path'], $parsed['query'], null);
            }
        }
        $this->url = $url;
        if ($this->permanentUrlMutable) {
            $this->permanent_url = $url;
        }
        $this->useragent = $useragent;
        if (preg_match('/^http(s)?:\/\//i', $url)) {
            if ($useragent === null) {
                $useragent = (string) ini_get('user_agent');
                $this->useragent = $useragent;
            }
            if (!is_array($headers)) {
                $headers = [];
            }
            if (!$force_fsockopen && function_exists('curl_exec')) {
                $this->method = \SimplePie\SimplePie::FILE_SOURCE_REMOTE | \SimplePie\SimplePie::FILE_SOURCE_CURL;
                $fp = curl_init();
                $headers2 = [];
                foreach ($headers as $key => $value) {
                    $headers2[] = "$key: $value";
                }
                if (isset($curl_options[CURLOPT_HTTPHEADER])) {
                    if (is_array($curl_options[CURLOPT_HTTPHEADER])) {
                        $headers2 = array_merge($headers2, $curl_options[CURLOPT_HTTPHEADER]);
                    }
                    unset($curl_options[CURLOPT_HTTPHEADER]);
                }
                if (version_compare(\SimplePie\Misc::get_curl_version(), '7.10.5', '>=')) {
                    curl_setopt($fp, CURLOPT_ENCODING, '');
                }
                curl_setopt($fp, CURLOPT_URL, $url);
                curl_setopt($fp, CURLOPT_HEADER, 1);
                curl_setopt($fp, CURLOPT_RETURNTRANSFER, 1);
                curl_setopt($fp, CURLOPT_FAILONERROR, 1);
                curl_setopt($fp, CURLOPT_TIMEOUT, $timeout);
                curl_setopt($fp, CURLOPT_CONNECTTIMEOUT, $timeout);
                curl_setopt($fp, CURLOPT_REFERER, \SimplePie\Misc::url_remove_credentials($url));
                curl_setopt($fp, CURLOPT_USERAGENT, $useragent);
                curl_setopt($fp, CURLOPT_HTTPHEADER, $headers2);
                foreach ($curl_options as $curl_param => $curl_value) {
                    curl_setopt($fp, $curl_param, $curl_value);
                }

                $responseHeaders = curl_exec($fp);
                if (curl_errno($fp) === CURLE_WRITE_ERROR || curl_errno($fp) === CURLE_BAD_CONTENT_ENCODING) {
                    curl_setopt($fp, CURLOPT_ENCODING, 'none');
                    $responseHeaders = curl_exec($fp);
                }
                $this->status_code = curl_getinfo($fp, CURLINFO_HTTP_CODE);
                if (curl_errno($fp)) {
                    $this->error = 'cURL error ' . curl_errno($fp) . ': ' . curl_error($fp);
                    $this->success = false;
                } else {
                    // Use the updated url provided by curl_getinfo after any redirects.
                    if ($info = curl_getinfo($fp)) {
                        $this->url = $info['url'];
                    }
                    // For PHPStan: We already checked that error did not occur.
                    assert(is_array($info) && $info['redirect_count'] >= 0);
                    if (\PHP_VERSION_ID < 80000) {
                        curl_close($fp);
                    }
                    $responseHeaders = \SimplePie\HTTP\Parser::prepareHeaders((string) $responseHeaders, $info['redirect_count'] + 1);
                    $parser = new \SimplePie\HTTP\Parser($responseHeaders, true);
                    if ($parser->parse()) {
                        $this->set_headers($parser->headers);
                        $this->body = $parser->body;
                        $this->status_code = $parser->status_code;
                        if ((in_array($this->status_code, [300, 301, 302, 303, 307]) || $this->status_code > 307 && $this->status_code < 400) && ($locationHeader = $this->get_header_line('location')) !== '' && $this->redirects < $redirects) {
                            $this->redirects++;
                            $location = \SimplePie\Misc::absolutize_url($locationHeader, $url);
                            if ($location === false) {
                                $this->error = "Invalid redirect location, trying to base “{$locationHeader}” onto “{$url}”";
                                $this->success = false;
                                return;
                            }
                            $this->permanentUrlMutable = $this->permanentUrlMutable && ($this->status_code == 301 || $this->status_code == 308);
                            $this->__construct($location, $timeout, $redirects, $headers, $useragent, $force_fsockopen, $curl_options);
                            return;
                        }
                    }
                }
            } else {
                $this->method = \SimplePie\SimplePie::FILE_SOURCE_REMOTE | \SimplePie\SimplePie::FILE_SOURCE_FSOCKOPEN;
                if (($url_parts = parse_url($url)) === false) {
                    throw new \InvalidArgumentException('Malformed URL: ' . $url);
                }
                if (!isset($url_parts['host'])) {
                    throw new \InvalidArgumentException('Missing hostname: ' . $url);
                }
                $socket_host = $url_parts['host'];
                if (isset($url_parts['scheme']) && strtolower($url_parts['scheme']) === 'https') {
                    $socket_host = 'ssl://' . $socket_host;
                    $url_parts['port'] = 443;
                }
                if (!isset($url_parts['port'])) {
                    $url_parts['port'] = 80;
                }
                $fp = @fsockopen($socket_host, $url_parts['port'], $errno, $errstr, $timeout);
                if (!$fp) {
                    $this->error = 'fsockopen error: ' . $errstr;
                    $this->success = false;
                } else {
                    stream_set_timeout($fp, $timeout);
                    if (isset($url_parts['path'])) {
                        if (isset($url_parts['query'])) {
                            $get = "$url_parts[path]?$url_parts[query]";
                        } else {
                            $get = $url_parts['path'];
                        }
                    } else {
                        $get = '/';
                    }
                    $out = "GET $get HTTP/1.1\r\n";
                    $out .= "Host: $url_parts[host]\r\n";
                    $out .= "User-Agent: $useragent\r\n";
                    if (extension_loaded('zlib')) {
                        $out .= "Accept-Encoding: x-gzip,gzip,deflate\r\n";
                    }

                    if (isset($url_parts['user']) && isset($url_parts['pass'])) {
                        $out .= "Authorization: Basic " . base64_encode("$url_parts[user]:$url_parts[pass]") . "\r\n";
                    }
                    foreach ($headers as $key => $value) {
                        $out .= "$key: $value\r\n";
                    }
                    $out .= "Connection: Close\r\n\r\n";
                    fwrite($fp, $out);

                    $info = stream_get_meta_data($fp);

                    $responseHeaders = '';
                    while (!$info['eof'] && !$info['timed_out']) {
                        $responseHeaders .= fread($fp, 1160);
                        $info = stream_get_meta_data($fp);
                    }
                    if (!$info['timed_out']) {
                        $parser = new \SimplePie\HTTP\Parser($responseHeaders, true);
                        if ($parser->parse()) {
                            $this->set_headers($parser->headers);
                            $this->body = $parser->body;
                            $this->status_code = $parser->status_code;
                            if ((in_array($this->status_code, [300, 301, 302, 303, 307]) || $this->status_code > 307 && $this->status_code < 400) && ($locationHeader = $this->get_header_line('location')) !== '' && $this->redirects < $redirects) {
                                $this->redirects++;
                                $location = \SimplePie\Misc::absolutize_url($locationHeader, $url);
                                $this->permanentUrlMutable = $this->permanentUrlMutable && ($this->status_code == 301 || $this->status_code == 308);
                                if ($location === false) {
                                    $this->error = "Invalid redirect location, trying to base “{$locationHeader}” onto “{$url}”";
                                    $this->success = false;
                                    return;
                                }
                                $this->__construct($location, $timeout, $redirects, $headers, $useragent, $force_fsockopen, $curl_options);
                                return;
                            }
                            if (($contentEncodingHeader = $this->get_header_line('content-encoding')) !== '') {
                                // Hey, we act dumb elsewhere, so let's do that here too
                                switch (strtolower(trim($contentEncodingHeader, "\x09\x0A\x0D\x20"))) {
                                    case 'gzip':
                                    case 'x-gzip':
                                        if (($decompressed = gzdecode($this->body)) === false) {
                                            $this->error = 'Unable to decode HTTP "gzip" stream';
                                            $this->success = false;
                                        } else {
                                            $this->body = $decompressed;
                                        }
                                        break;

                                    case 'deflate':
                                        if (($decompressed = gzinflate($this->body)) !== false) {
                                            $this->body = $decompressed;
                                        } elseif (($decompressed = gzuncompress($this->body)) !== false) {
                                            $this->body = $decompressed;
                                        } elseif (($decompressed = gzdecode($this->body)) !== false) {
                                            $this->body = $decompressed;
                                        } else {
                                            $this->error = 'Unable to decode HTTP "deflate" stream';
                                            $this->success = false;
                                        }
                                        break;

                                    default:
                                        $this->error = 'Unknown content coding';
                                        $this->success = false;
                                }
                            }
                        }
                    } else {
                        $this->error = 'fsocket timed out';
                        $this->success = false;
                    }
                    fclose($fp);
                }
            }
        } else {
            $this->method = \SimplePie\SimplePie::FILE_SOURCE_LOCAL | \SimplePie\SimplePie::FILE_SOURCE_FILE_GET_CONTENTS;
            if (empty($url) || !is_readable($url) ||  false === $filebody = file_get_contents($url)) {
                $this->body = '';
                $this->error = sprintf('file "%s" is not readable', $url);
                $this->success = false;
            } else {
                $this->body = $filebody;
                $this->status_code = 200;
            }
        }
        if ($this->success) {
            assert($this->body !== null); // For PHPStan
            // Leading whitespace may cause XML parsing errors (XML declaration cannot be preceded by anything other than BOM) so we trim it.
            // Note that unlike built-in `trim` function’s default settings, we do not trim `\x00` to avoid breaking characters in UTF-16 or UTF-32 encoded strings.
            // We also only do that when the whitespace is followed by `<`, so that we do not break e.g. UTF-16LE encoded whitespace like `\n\x00` in half.
            $this->body = preg_replace('/^[ \n\r\t\v]+</', '<', $this->body);
        }
    }

    public function get_permanent_uri(): string
    {
        return (string) $this->permanent_url;
    }

    public function get_final_requested_uri(): string
    {
        return (string) $this->url;
    }

    public function get_status_code(): int
    {
        return (int) $this->status_code;
    }

    public function get_headers(): array
    {
        $this->maybe_update_headers();
        return $this->parsed_headers;
    }

    public function has_header(string $name): bool
    {
        $this->maybe_update_headers();
        return $this->get_header($name) !== [];
    }

    public function get_header(string $name): array
    {
        $this->maybe_update_headers();
        return $this->parsed_headers[strtolower($name)] ?? [];
    }

    public function with_header(string $name, $value)
    {
        $this->maybe_update_headers();
        $new = clone $this;

        $newHeader = [
            strtolower($name) => (array) $value,
        ];
        $new->set_headers($newHeader + $this->get_headers());

        return $new;
    }

    public function get_header_line(string $name): string
    {
        $this->maybe_update_headers();
        return implode(', ', $this->get_header($name));
    }

    public function get_body_content(): string
    {
        return (string) $this->body;
    }

    /**
     * Check if the $headers property was changed and update the internal state accordingly.
     */
    private function maybe_update_headers(): void
    {
        if ($this->headers !== $this->last_headers) {
            $this->parsed_headers = array_map(
                function (string $header_line): array {
                    if (strpos($header_line, ',') === false) {
                        return [$header_line];
                    } else {
                        return array_map('trim', explode(',', $header_line));
                    }
                },
                $this->headers
            );
        }
        $this->last_headers = $this->headers;
    }

    /**
     * Sets headers internally.
     *
     * @param array<string, non-empty-array<string>> $headers
     */
    private function set_headers(array $headers): void
    {
        $this->parsed_headers = $headers;
        $this->headers = self::flatten_headers($headers);
        $this->last_headers = $this->headers;
    }

    /**
     * Converts PSR-7 compatible headers into a legacy format.
     *
     * @param array<string, non-empty-array<string>> $headers
     *
     * @return array<string, string>
     */
    private function flatten_headers(array $headers): array
    {
        return array_map(function (array $values): string {
            return implode(',', $values);
        }, $headers);
    }

    /**
     * Create a File instance from another Response
     *
     * For BC reasons in some places there MUST be a `File` instance
     * instead of a `Response` implementation
     *
     * @see Locator::__construct()
     * @internal
     */
    final public static function fromResponse(Response $response): self
    {
        $headers = [];

        foreach ($response->get_headers() as $name => $header) {
            $headers[$name] = implode(', ', $header);
        }

        /** @var File */
        $file = (new \ReflectionClass(File::class))->newInstanceWithoutConstructor();

        $file->url = $response->get_final_requested_uri();
        $file->useragent = null;
        $file->headers = $headers;
        $file->body = $response->get_body_content();
        $file->status_code = $response->get_status_code();
        $file->permanent_url = $response->get_permanent_uri();

        return $file;
    }
}

class_alias('SimplePie\File', 'SimplePie_File');

}

namespace SimplePie {

/**
 * Decode 'gzip' encoded HTTP data
 *
 * @link http://www.gzip.org/format.txt
 * @link https://www.php.net/manual/en/function.gzdecode.php
 * @deprecated since SimplePie 1.9.0, use `gzdecode` function instead.
 */
class Gzdecode
{
    /**
     * Compressed data
     *
     * @access private
     * @var string
     * @see gzdecode::$data
     */
    public $compressed_data;

    /**
     * Size of compressed data
     *
     * @access private
     * @var int
     */
    public $compressed_size;

    /**
     * Minimum size of a valid gzip string
     *
     * @access private
     * @var int
     */
    public $min_compressed_size = 18;

    /**
     * Current position of pointer
     *
     * @access private
     * @var int
     */
    public $position = 0;

    /**
     * Flags (FLG)
     *
     * @access private
     * @var int
     */
    public $flags;

    /**
     * Uncompressed data
     *
     * @access public
     * @see gzdecode::$compressed_data
     * @var string
     */
    public $data;

    /**
     * Modified time
     *
     * @access public
     * @var int
     */
    public $MTIME;

    /**
     * Extra Flags
     *
     * @access public
     * @var int
     */
    public $XFL;

    /**
     * Operating System
     *
     * @access public
     * @var int
     */
    public $OS;

    /**
     * Subfield ID 1
     *
     * @access public
     * @see gzdecode::$extra_field
     * @see gzdecode::$SI2
     * @var string
     */
    public $SI1;

    /**
     * Subfield ID 2
     *
     * @access public
     * @see gzdecode::$extra_field
     * @see gzdecode::$SI1
     * @var string
     */
    public $SI2;

    /**
     * Extra field content
     *
     * @access public
     * @see gzdecode::$SI1
     * @see gzdecode::$SI2
     * @var string
     */
    public $extra_field;

    /**
     * Original filename
     *
     * @access public
     * @var string
     */
    public $filename;

    /**
     * Human readable comment
     *
     * @access public
     * @var string
     */
    public $comment;

    /**
     * Don't allow anything to be set
     *
     * @param string $name
     * @param mixed $value
     */
    public function __set(string $name, $value)
    {
        throw new Exception("Cannot write property $name");
    }

    /**
     * Set the compressed string and related properties
     *
     * @param string $data
     */
    public function __construct(string $data)
    {
        $this->compressed_data = $data;
        $this->compressed_size = strlen($data);
    }

    /**
     * Decode the GZIP stream
     *
     * @return bool Successfulness
     */
    public function parse()
    {
        if ($this->compressed_size >= $this->min_compressed_size) {
            $len = 0;

            // Check ID1, ID2, and CM
            if (substr($this->compressed_data, 0, 3) !== "\x1F\x8B\x08") {
                return false;
            }

            // Get the FLG (FLaGs)
            $this->flags = ord($this->compressed_data[3]);

            // FLG bits above (1 << 4) are reserved
            if ($this->flags > 0x1F) {
                return false;
            }

            // Advance the pointer after the above
            $this->position += 4;

            // MTIME
            $mtime = substr($this->compressed_data, $this->position, 4);
            // Reverse the string if we're on a big-endian arch because l is the only signed long and is machine endianness
            if (current((array) unpack('S', "\x00\x01")) === 1) {
                $mtime = strrev($mtime);
            }
            $this->MTIME = current((array) unpack('l', $mtime));
            $this->position += 4;

            // Get the XFL (eXtra FLags)
            $this->XFL = ord($this->compressed_data[$this->position++]);

            // Get the OS (Operating System)
            $this->OS = ord($this->compressed_data[$this->position++]);

            // Parse the FEXTRA
            if ($this->flags & 4) {
                // Read subfield IDs
                $this->SI1 = $this->compressed_data[$this->position++];
                $this->SI2 = $this->compressed_data[$this->position++];

                // SI2 set to zero is reserved for future use
                if ($this->SI2 === "\x00") {
                    return false;
                }

                // Get the length of the extra field
                $len = current((array) unpack('v', substr($this->compressed_data, $this->position, 2)));
                $this->position += 2;

                // Check the length of the string is still valid
                $this->min_compressed_size += $len + 4;
                if ($this->compressed_size >= $this->min_compressed_size) {
                    // Set the extra field to the given data
                    $this->extra_field = substr($this->compressed_data, $this->position, $len);
                    $this->position += $len;
                } else {
                    return false;
                }
            }

            // Parse the FNAME
            if ($this->flags & 8) {
                // Get the length of the filename
                $len = strcspn($this->compressed_data, "\x00", $this->position);

                // Check the length of the string is still valid
                $this->min_compressed_size += $len + 1;
                if ($this->compressed_size >= $this->min_compressed_size) {
                    // Set the original filename to the given string
                    $this->filename = substr($this->compressed_data, $this->position, $len);
                    $this->position += $len + 1;
                } else {
                    return false;
                }
            }

            // Parse the FCOMMENT
            if ($this->flags & 16) {
                // Get the length of the comment
                $len = strcspn($this->compressed_data, "\x00", $this->position);

                // Check the length of the string is still valid
                $this->min_compressed_size += $len + 1;
                if ($this->compressed_size >= $this->min_compressed_size) {
                    // Set the original comment to the given string
                    $this->comment = substr($this->compressed_data, $this->position, $len);
                    $this->position += $len + 1;
                } else {
                    return false;
                }
            }

            // Parse the FHCRC
            if ($this->flags & 2) {
                // Check the length of the string is still valid
                $this->min_compressed_size += $len + 2;
                if ($this->compressed_size >= $this->min_compressed_size) {
                    // Read the CRC
                    $crc = current((array) unpack('v', substr($this->compressed_data, $this->position, 2)));

                    // Check the CRC matches
                    if ((crc32(substr($this->compressed_data, 0, $this->position)) & 0xFFFF) === $crc) {
                        $this->position += 2;
                    } else {
                        return false;
                    }
                } else {
                    return false;
                }
            }

            // Decompress the actual data
            if (($data = gzinflate(substr($this->compressed_data, $this->position, -8))) === false) {
                return false;
            }

            $this->data = $data;
            $this->position = $this->compressed_size - 8;

            // Check CRC of data
            $crc = current((array) unpack('V', substr($this->compressed_data, $this->position, 4)));
            $this->position += 4;
            /*if (extension_loaded('hash') && sprintf('%u', current(unpack('V', hash('crc32b', $this->data)))) !== sprintf('%u', $crc))
            {
                return false;
            }*/

            // Check ISIZE of data
            $isize = current((array) unpack('V', substr($this->compressed_data, $this->position, 4)));
            $this->position += 4;
            if (sprintf('%u', strlen($this->data) & 0xFFFFFFFF) !== sprintf('%u', $isize)) {
                return false;
            }

            // Wow, against all odds, we've actually got a valid gzip string
            return true;
        }

        return false;
    }
}

class_alias('SimplePie\Gzdecode', 'SimplePie_gzdecode');

}

namespace SimplePie\HTTP {

/**
 * HTTP Client interface
 *
 * @internal
 */
interface Client
{
    public const METHOD_GET = 'GET';

    /**
     * send a request and return the response
     *
     * @param Client::METHOD_* $method
     * @param array<string, string> $headers
     *
     * @throws ClientException if anything goes wrong requesting the data
     */
    public function request(string $method, string $url, array $headers = []): Response;
}

}

namespace SimplePie\HTTP {

use SimplePie\Exception as SimplePieException;

/**
 * Client exception class
 *
 * @internal
 */
final class ClientException extends SimplePieException
{
}

}

namespace SimplePie\HTTP {

use InvalidArgumentException;
use SimplePie\File;
use SimplePie\Misc;
use SimplePie\Registry;
use Throwable;

/**
 * HTTP Client based on \SimplePie\File
 *
 * @internal
 */
final class FileClient implements Client
{
    /** @var Registry */
    private $registry;

    /** @var array{timeout?: int, redirects?: int, useragent?: string, force_fsockopen?: bool, curl_options?: array<mixed>} */
    private $options;

    /**
     * @param array{timeout?: int, redirects?: int, useragent?: string, force_fsockopen?: bool, curl_options?: array<mixed>} $options
     */
    public function __construct(Registry $registry, array $options = [])
    {
        $this->registry = $registry;
        $this->options = $options;
    }

    /**
     * send a request and return the response
     *
     * @param Client::METHOD_* $method
     * @param array<string, string> $headers
     *
     * @throws ClientException if anything goes wrong requesting the data
     */
    public function request(string $method, string $url, array $headers = []): Response
    {
        // @phpstan-ignore-next-line Enforce PHPDoc type.
        if ($method !== self::METHOD_GET) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($method) only supports method "%s".',
                __METHOD__,
                self::METHOD_GET
            ), 1);
        }

        try {
            $file = $this->registry->create(File::class, [
                $url,
                $this->options['timeout'] ?? 10,
                $this->options['redirects'] ?? 5,
                $headers,
                $this->options['useragent'] ?? Misc::get_default_useragent(),
                $this->options['force_fsockopen'] ?? false,
                $this->options['curl_options'] ?? []
            ]);
        } catch (Throwable $th) {
            throw new ClientException($th->getMessage(), $th->getCode(), $th);
        }

        if ($file->error !== null && $file->get_status_code() === 0) {
            throw new ClientException($file->error);
        }

        return $file;
    }
}

}

namespace SimplePie\HTTP {

/**
 * HTTP Response Parser
 * @template Psr7Compatible of bool
 */
class Parser
{
    /**
     * HTTP Version
     *
     * @var float
     */
    public $http_version = 0.0;

    /**
     * Status code
     *
     * @var int
     */
    public $status_code = 0;

    /**
     * Reason phrase
     *
     * @var string
     */
    public $reason = '';

    /**
     * @var Psr7Compatible whether headers are compatible with PSR-7 format.
     */
    private $psr7Compatible;

    /**
     * Key/value pairs of the headers
     *
     * @var (Psr7Compatible is true ? array<string, non-empty-array<string>> : array<string, string>)
     */
    public $headers = [];

    /**
     * Body of the response
     *
     * @var string
     */
    public $body = '';

    private const STATE_HTTP_VERSION = 'http_version';

    private const STATE_STATUS = 'status';

    private const STATE_REASON = 'reason';

    private const STATE_NEW_LINE = 'new_line';

    private const STATE_BODY = 'body';

    private const STATE_NAME = 'name';

    private const STATE_VALUE = 'value';

    private const STATE_VALUE_CHAR = 'value_char';

    private const STATE_QUOTE = 'quote';

    private const STATE_QUOTE_ESCAPED = 'quote_escaped';

    private const STATE_QUOTE_CHAR = 'quote_char';

    private const STATE_CHUNKED = 'chunked';

    private const STATE_EMIT = 'emit';

    private const STATE_ERROR = false;

    /**
     * Current state of the state machine
     *
     * @var self::STATE_*
     */
    protected $state = self::STATE_HTTP_VERSION;

    /**
     * Input data
     *
     * @var string
     */
    protected $data = '';

    /**
     * Input data length (to avoid calling strlen() everytime this is needed)
     *
     * @var int
     */
    protected $data_length = 0;

    /**
     * Current position of the pointer
     *
     * @var int
     */
    protected $position = 0;

    /**
     * Name of the header currently being parsed
     *
     * @var string
     */
    protected $name = '';

    /**
     * Value of the header currently being parsed
     *
     * @var string
     */
    protected $value = '';

    /**
     * Create an instance of the class with the input data
     *
     * @param string $data Input data
     * @param Psr7Compatible $psr7Compatible Whether the data types are in format compatible with PSR-7.
     */
    public function __construct(string $data, bool $psr7Compatible = false)
    {
        $this->data = $data;
        $this->data_length = strlen($this->data);
        $this->psr7Compatible = $psr7Compatible;
    }

    /**
     * Parse the input data
     *
     * @return bool true on success, false on failure
     */
    public function parse()
    {
        while ($this->state && $this->state !== self::STATE_EMIT && $this->has_data()) {
            $state = $this->state;
            $this->$state();
        }
        $this->data = '';
        if ($this->state === self::STATE_EMIT || $this->state === self::STATE_BODY) {
            return true;
        }

        // Reset the parser state.
        $this->http_version = 0.0;
        $this->status_code = 0;
        $this->reason = '';
        $this->headers = [];
        $this->body = '';
        return false;
    }

    /**
     * Check whether there is data beyond the pointer
     *
     * @return bool true if there is further data, false if not
     */
    protected function has_data()
    {
        return (bool) ($this->position < $this->data_length);
    }

    /**
     * See if the next character is LWS
     *
     * @return bool true if the next character is LWS, false if not
     */
    protected function is_linear_whitespace()
    {
        return (bool) ($this->data[$this->position] === "\x09"
            || $this->data[$this->position] === "\x20"
            || ($this->data[$this->position] === "\x0A"
                && isset($this->data[$this->position + 1])
                && ($this->data[$this->position + 1] === "\x09" || $this->data[$this->position + 1] === "\x20")));
    }

    /**
     * Parse the HTTP version
     * @return void
     */
    protected function http_version()
    {
        if (strpos($this->data, "\x0A") !== false && strtoupper(substr($this->data, 0, 5)) === 'HTTP/') {
            $len = strspn($this->data, '0123456789.', 5);
            $http_version = substr($this->data, 5, $len);
            $this->position += 5 + $len;
            if (substr_count($http_version, '.') <= 1) {
                $this->http_version = (float) $http_version;
                $this->position += strspn($this->data, "\x09\x20", $this->position);
                $this->state = self::STATE_STATUS;
            } else {
                $this->state = self::STATE_ERROR;
            }
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    /**
     * Parse the status code
     * @return void
     */
    protected function status()
    {
        if ($len = strspn($this->data, '0123456789', $this->position)) {
            $this->status_code = (int) substr($this->data, $this->position, $len);
            $this->position += $len;
            $this->state = self::STATE_REASON;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    /**
     * Parse the reason phrase
     * @return void
     */
    protected function reason()
    {
        $len = strcspn($this->data, "\x0A", $this->position);
        $this->reason = trim(substr($this->data, $this->position, $len), "\x09\x0D\x20");
        $this->position += $len + 1;
        $this->state = self::STATE_NEW_LINE;
    }

    private function add_header(string $name, string $value): void
    {
        if ($this->psr7Compatible) {
            // For PHPStan: should be enforced by template parameter but PHPStan is not smart enough.
            /** @var array<string, non-empty-array<string>> */
            $headers = &$this->headers;
            $headers[$name][] = $value;
        } else {
            // For PHPStan: should be enforced by template parameter but PHPStan is not smart enough.
            /** @var array<string, string>) */
            $headers = &$this->headers;
            $headers[$name] .= ', ' . $value;
        }
    }

    private function replace_header(string $name, string $value): void
    {
        if ($this->psr7Compatible) {
            // For PHPStan: should be enforced by template parameter but PHPStan is not smart enough.
            /** @var array<string, non-empty-array<string>> */
            $headers = &$this->headers;
            $headers[$name] = [$value];
        } else {
            // For PHPStan: should be enforced by template parameter but PHPStan is not smart enough.
            /** @var array<string, string>) */
            $headers = &$this->headers;
            $headers[$name] = $value;
        }
    }

    /**
     * Deal with a new line, shifting data around as needed
     * @return void
     */
    protected function new_line()
    {
        $this->value = trim($this->value, "\x0D\x20");
        if ($this->name !== '' && $this->value !== '') {
            $this->name = strtolower($this->name);
            // We should only use the last Content-Type header. c.f. issue #1
            if (isset($this->headers[$this->name]) && $this->name !== 'content-type') {
                $this->add_header($this->name, $this->value);
            } else {
                $this->replace_header($this->name, $this->value);
            }
        }
        $this->name = '';
        $this->value = '';
        if (substr($this->data[$this->position], 0, 2) === "\x0D\x0A") {
            $this->position += 2;
            $this->state = self::STATE_BODY;
        } elseif ($this->data[$this->position] === "\x0A") {
            $this->position++;
            $this->state = self::STATE_BODY;
        } else {
            $this->state = self::STATE_NAME;
        }
    }

    /**
     * Parse a header name
     * @return void
     */
    protected function name()
    {
        $len = strcspn($this->data, "\x0A:", $this->position);
        if (isset($this->data[$this->position + $len])) {
            if ($this->data[$this->position + $len] === "\x0A") {
                $this->position += $len;
                $this->state = self::STATE_NEW_LINE;
            } else {
                $this->name = substr($this->data, $this->position, $len);
                $this->position += $len + 1;
                $this->state = self::STATE_VALUE;
            }
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    /**
     * Parse LWS, replacing consecutive LWS characters with a single space
     * @return void
     */
    protected function linear_whitespace()
    {
        do {
            if (substr($this->data, $this->position, 2) === "\x0D\x0A") {
                $this->position += 2;
            } elseif ($this->data[$this->position] === "\x0A") {
                $this->position++;
            }
            $this->position += strspn($this->data, "\x09\x20", $this->position);
        } while ($this->has_data() && $this->is_linear_whitespace());
        $this->value .= "\x20";
    }

    /**
     * See what state to move to while within non-quoted header values
     * @return void
     */
    protected function value()
    {
        if ($this->is_linear_whitespace()) {
            $this->linear_whitespace();
        } else {
            switch ($this->data[$this->position]) {
                case '"':
                    // Workaround for ETags: we have to include the quotes as
                    // part of the tag.
                    if (strtolower($this->name) === 'etag') {
                        $this->value .= '"';
                        $this->position++;
                        $this->state = self::STATE_VALUE_CHAR;
                        break;
                    }
                    $this->position++;
                    $this->state = self::STATE_QUOTE;
                    break;

                case "\x0A":
                    $this->position++;
                    $this->state = self::STATE_NEW_LINE;
                    break;

                default:
                    $this->state = self::STATE_VALUE_CHAR;
                    break;
            }
        }
    }

    /**
     * Parse a header value while outside quotes
     * @return void
     */
    protected function value_char()
    {
        $len = strcspn($this->data, "\x09\x20\x0A\"", $this->position);
        $this->value .= substr($this->data, $this->position, $len);
        $this->position += $len;
        $this->state = self::STATE_VALUE;
    }

    /**
     * See what state to move to while within quoted header values
     * @return void
     */
    protected function quote()
    {
        if ($this->is_linear_whitespace()) {
            $this->linear_whitespace();
        } else {
            switch ($this->data[$this->position]) {
                case '"':
                    $this->position++;
                    $this->state = self::STATE_VALUE;
                    break;

                case "\x0A":
                    $this->position++;
                    $this->state = self::STATE_NEW_LINE;
                    break;

                case '\\':
                    $this->position++;
                    $this->state = self::STATE_QUOTE_ESCAPED;
                    break;

                default:
                    $this->state = self::STATE_QUOTE_CHAR;
                    break;
            }
        }
    }

    /**
     * Parse a header value while within quotes
     * @return void
     */
    protected function quote_char()
    {
        $len = strcspn($this->data, "\x09\x20\x0A\"\\", $this->position);
        $this->value .= substr($this->data, $this->position, $len);
        $this->position += $len;
        $this->state = self::STATE_VALUE;
    }

    /**
     * Parse an escaped character within quotes
     * @return void
     */
    protected function quote_escaped()
    {
        $this->value .= $this->data[$this->position];
        $this->position++;
        $this->state = self::STATE_QUOTE;
    }

    /**
     * Parse the body
     * @return void
     */
    protected function body()
    {
        $this->body = substr($this->data, $this->position);
        if (!empty($this->headers['transfer-encoding'])) {
            unset($this->headers['transfer-encoding']);
            $this->state = self::STATE_CHUNKED;
        } else {
            $this->state = self::STATE_EMIT;
        }
    }

    /**
     * Parsed a "Transfer-Encoding: chunked" body
     * @return void
     */
    protected function chunked()
    {
        if (!preg_match('/^([0-9a-f]+)[^\r\n]*\r\n/i', trim($this->body))) {
            $this->state = self::STATE_EMIT;
            return;
        }

        $decoded = '';
        $encoded = $this->body;

        while (true) {
            $is_chunked = (bool) preg_match('/^([0-9a-f]+)[^\r\n]*\r\n/i', $encoded, $matches);
            if (!$is_chunked) {
                // Looks like it's not chunked after all
                $this->state = self::STATE_EMIT;
                return;
            }

            $length = hexdec(trim($matches[1]));
            // For PHPStan: this will only be float when larger than PHP_INT_MAX.
            // But even on 32-bit systems, it would mean 2GiB chunk, which sounds unlikely.
            \assert(\is_int($length), "Length needs to be shorter than PHP_INT_MAX");
            if ($length === 0) {
                // Ignore trailer headers
                $this->state = self::STATE_EMIT;
                $this->body = $decoded;
                return;
            }

            $chunk_length = strlen($matches[0]);
            $decoded .= substr($encoded, $chunk_length, $length);
            $encoded = substr($encoded, $chunk_length + $length + 2);

            // BC for PHP < 8.0: substr() can return bool instead of string
            $encoded = ($encoded === false) ? '' : $encoded;

            if (trim($encoded) === '0' || empty($encoded)) {
                $this->state = self::STATE_EMIT;
                $this->body = $decoded;
                return;
            }
        }
    }

    /**
     * Prepare headers (take care of proxies headers)
     *
     * @param string  $headers Raw headers
     * @param non-negative-int $count Redirection count. Default to 1.
     *
     * @return string
     */
    public static function prepareHeaders(string $headers, int $count = 1)
    {
        $data = explode("\r\n\r\n", $headers, $count);
        $data = array_pop($data);
        if (false !== stripos($data, "HTTP/1.0 200 Connection established\r\n")) {
            $exploded = explode("\r\n\r\n", $data, 2);
            $data = end($exploded);
        }
        if (false !== stripos($data, "HTTP/1.1 200 Connection established\r\n")) {
            $exploded = explode("\r\n\r\n", $data, 2);
            $data = end($exploded);
        }
        return $data;
    }
}

class_alias('SimplePie\HTTP\Parser', 'SimplePie_HTTP_Parser');

}

namespace SimplePie\HTTP {

use Psr\Http\Message\ResponseInterface;

/**
 * HTTP Response based on a PSR-7 response
 *
 * This interface must be interoperable with Psr\Http\Message\ResponseInterface
 * @see https://www.php-fig.org/psr/psr-7/#33-psrhttpmessageresponseinterface
 *
 * @internal
 */
final class Psr7Response implements Response
{
    /**
     * @var ResponseInterface
     */
    private $response;

    /**
     * @var string
     */
    private $permanent_url;

    /**
     * @var string
     */
    private $requested_url;

    public function __construct(ResponseInterface $response, string $permanent_url, string $requested_url)
    {
        $this->response = $response;
        $this->permanent_url = $permanent_url;
        $this->requested_url = $requested_url;
    }

    public function get_permanent_uri(): string
    {
        return $this->permanent_url;
    }

    public function get_final_requested_uri(): string
    {
        return $this->requested_url;
    }

    public function get_status_code(): int
    {
        return $this->response->getStatusCode();
    }

    public function get_headers(): array
    {
        // The filtering is probably redundant but let’s make PHPStan happy.
        return array_filter($this->response->getHeaders(), function (array $header): bool {
            return count($header) >= 1;
        });
    }

    public function has_header(string $name): bool
    {
        return $this->response->hasHeader($name);
    }

    public function with_header(string $name, $value)
    {
        return new self($this->response->withHeader($name, $value), $this->permanent_url, $this->requested_url);
    }

    public function get_header(string $name): array
    {
        return $this->response->getHeader($name);
    }

    public function get_header_line(string $name): string
    {
        return $this->response->getHeaderLine($name);
    }

    public function get_body_content(): string
    {
        return $this->response->getBody()->__toString();
    }
}

}

namespace SimplePie\HTTP {

use InvalidArgumentException;
use Psr\Http\Client\ClientExceptionInterface;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\UriFactoryInterface;
use Throwable;

/**
 * HTTP Client based on PSR-18 and PSR-17 implementations
 *
 * @internal
 */
final class Psr18Client implements Client
{
    /**
     * @var ClientInterface
     */
    private $httpClient;

    /**
     * @var RequestFactoryInterface
     */
    private $requestFactory;

    /**
     * @var UriFactoryInterface
     */
    private $uriFactory;

    /**
     * @var int
     */
    private $allowedRedirects = 5;

    public function __construct(ClientInterface $httpClient, RequestFactoryInterface $requestFactory, UriFactoryInterface $uriFactory)
    {
        $this->httpClient = $httpClient;
        $this->requestFactory = $requestFactory;
        $this->uriFactory = $uriFactory;
    }

    public function getHttpClient(): ClientInterface
    {
        return $this->httpClient;
    }

    public function getRequestFactory(): RequestFactoryInterface
    {
        return $this->requestFactory;
    }

    public function getUriFactory(): UriFactoryInterface
    {
        return $this->uriFactory;
    }

    /**
     * send a request and return the response
     *
     * @param Client::METHOD_* $method
     * @param string $url
     * @param array<string,string|string[]> $headers
     *
     * @throws ClientException if anything goes wrong requesting the data
     */
    public function request(string $method, string $url, array $headers = []): Response
    {
        if ($method !== self::METHOD_GET) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($method) only supports method "%s".',
                __METHOD__,
                self::METHOD_GET
            ), 1);
        }

        if (preg_match('/^http(s)?:\/\//i', $url)) {
            return $this->requestUrl($method, $url, $headers);
        }

        return $this->requestLocalFile($url);
    }

    /**
     * @param array<string,string|string[]> $headers
     */
    private function requestUrl(string $method, string $url, array $headers): Response
    {
        $permanentUrl = $url;
        $requestedUrl = $url;
        $remainingRedirects = $this->allowedRedirects;

        $request = $this->requestFactory->createRequest(
            $method,
            $this->uriFactory->createUri($requestedUrl)
        );

        foreach ($headers as $name => $value) {
            $request = $request->withHeader($name, $value);
        }

        do {
            $followRedirect = false;

            try {
                $response = $this->httpClient->sendRequest($request);
            } catch (ClientExceptionInterface $th) {
                throw new ClientException($th->getMessage(), $th->getCode(), $th);
            }

            $statusCode = $response->getStatusCode();

            // If we have a redirect
            if (in_array($statusCode, [300, 301, 302, 303, 307]) && $response->hasHeader('Location')) {
                // Prevent infinity redirect loops
                if ($remainingRedirects <= 0) {
                    break;
                }

                $remainingRedirects--;
                $followRedirect = true;

                $requestedUrl = $response->getHeaderLine('Location');

                if ($statusCode === 301) {
                    $permanentUrl = $requestedUrl;
                }

                $request = $request->withUri($this->uriFactory->createUri($requestedUrl));
            }
        } while ($followRedirect);

        return new Psr7Response($response, $permanentUrl, $requestedUrl);
    }

    private function requestLocalFile(string $path): Response
    {
        if (!is_readable($path)) {
            throw new ClientException(sprintf('file "%s" is not readable', $path));
        }

        try {
            $raw = file_get_contents($path);
        } catch (Throwable $th) {
            throw new ClientException($th->getMessage(), $th->getCode(), $th);
        }

        if ($raw === false) {
            throw new ClientException('file_get_contents() could not read the file', 1);
        }

        return new RawTextResponse($raw, $path);
    }
}

}

namespace SimplePie\HTTP {

/**
 * HTTP Response for rax text
 *
 * This interface must be interoperable with Psr\Http\Message\ResponseInterface
 * @see https://www.php-fig.org/psr/psr-7/#33-psrhttpmessageresponseinterface
 *
 * @internal
 */
final class RawTextResponse implements Response
{
    /**
     * @var string
     */
    private $raw_text;

    /**
     * @var string
     */
    private $permanent_url;

    /**
     * @var array<non-empty-array<string>>
     */
    private $headers = [];

    /**
     * @var string
     */
    private $requested_url;

    public function __construct(string $raw_text, string $filepath)
    {
        $this->raw_text = $raw_text;
        $this->permanent_url = $filepath;
        $this->requested_url = $filepath;
    }

    public function get_permanent_uri(): string
    {
        return $this->permanent_url;
    }

    public function get_final_requested_uri(): string
    {
        return $this->requested_url;
    }

    public function get_status_code(): int
    {
        return 200;
    }

    public function get_headers(): array
    {
        return $this->headers;
    }

    public function has_header(string $name): bool
    {
        return isset($this->headers[strtolower($name)]);
    }

    public function get_header(string $name): array
    {
        return isset($this->headers[strtolower($name)]) ? $this->headers[$name] : [];
    }

    public function with_header(string $name, $value)
    {
        $new = clone $this;

        $newHeader = [
            strtolower($name) => (array) $value,
        ];
        $new->headers = $newHeader + $this->headers;

        return $new;
    }

    public function get_header_line(string $name): string
    {
        return isset($this->headers[strtolower($name)]) ? implode(", ", $this->headers[$name]) : '';
    }

    public function get_body_content(): string
    {
        return $this->raw_text;
    }
}

}

namespace SimplePie\HTTP {

/**
 * HTTP Response interface
 *
 * This interface must be interoperable with Psr\Http\Message\ResponseInterface
 * @see https://www.php-fig.org/psr/psr-7/#33-psrhttpmessageresponseinterface
 *
 * @internal
 */
interface Response
{
    /**
     * Return the string representation of the permanent URI of the requested resource
     * (the first location after a prefix of (only) permanent redirects).
     *
     * Depending on which components of the URI are present, the resulting
     * string is either a full URI or relative reference according to RFC 3986,
     * Section 4.1. The method concatenates the various components of the URI,
     * using the appropriate delimiters:
     *
     * - If a scheme is present, it MUST be suffixed by ":".
     * - If an authority is present, it MUST be prefixed by "//".
     * - The path can be concatenated without delimiters. But there are two
     *   cases where the path has to be adjusted to make the URI reference
     *   valid as PHP does not allow to throw an exception in __toString():
     *     - If the path is rootless and an authority is present, the path MUST
     *       be prefixed by "/".
     *     - If the path is starting with more than one "/" and no authority is
     *       present, the starting slashes MUST be reduced to one.
     * - If a query is present, it MUST be prefixed by "?".
     * - If a fragment is present, it MUST be prefixed by "#".
     *
     * @see http://tools.ietf.org/html/rfc3986#section-4.1
     */
    public function get_permanent_uri(): string;

    /**
     * Return the string representation of the final requested URL after following all redirects.
     *
     * Depending on which components of the URI are present, the resulting
     * string is either a full URI or relative reference according to RFC 3986,
     * Section 4.1. The method concatenates the various components of the URI,
     * using the appropriate delimiters:
     *
     * - If a scheme is present, it MUST be suffixed by ":".
     * - If an authority is present, it MUST be prefixed by "//".
     * - The path can be concatenated without delimiters. But there are two
     *   cases where the path has to be adjusted to make the URI reference
     *   valid as PHP does not allow to throw an exception in __toString():
     *     - If the path is rootless and an authority is present, the path MUST
     *       be prefixed by "/".
     *     - If the path is starting with more than one "/" and no authority is
     *       present, the starting slashes MUST be reduced to one.
     * - If a query is present, it MUST be prefixed by "?".
     * - If a fragment is present, it MUST be prefixed by "#".
     *
     * @see http://tools.ietf.org/html/rfc3986#section-4.1
     */
    public function get_final_requested_uri(): string;

    /**
     * Gets the response status code.
     *
     * The status code is a 3-digit integer result code of the server's attempt
     * to understand and satisfy the request.
     *
     * @return int Status code.
     */
    public function get_status_code(): int;

    /**
     * Retrieves all message header values.
     *
     * The keys represent the header name as it will be sent over the wire, and
     * each value is an array of strings associated with the header.
     *
     *     // Represent the headers as a string
     *     foreach ($message->get_headers() as $name => $values) {
     *         echo $name . ': ' . implode(', ', $values);
     *     }
     *
     *     // Emit headers iteratively:
     *     foreach ($message->get_headers() as $name => $values) {
     *         foreach ($values as $value) {
     *             header(sprintf('%s: %s', $name, $value), false);
     *         }
     *     }
     *
     * @return array<non-empty-array<string>> Returns an associative array of the message's headers.
     *     Each key MUST be a header name, and each value MUST be an array of
     *     strings for that header.
     */
    public function get_headers(): array;

    /**
     * Checks if a header exists by the given case-insensitive name.
     *
     * @param string $name Case-insensitive header field name.
     * @return bool Returns true if any header names match the given header
     *     name using a case-insensitive string comparison. Returns false if
     *     no matching header name is found in the message.
     */
    public function has_header(string $name): bool;

    /**
     * Retrieves a message header value by the given case-insensitive name.
     *
     * This method returns an array of all the header values of the given
     * case-insensitive header name.
     *
     * If the header does not appear in the message, this method MUST return an
     * empty array.
     *
     * @param string $name Case-insensitive header field name.
     * @return string[] An array of string values as provided for the given
     *    header. If the header does not appear in the message, this method MUST
     *    return an empty array.
     */
    public function get_header(string $name): array;

    /**
     * Return an instance with the provided value replacing the specified header.
     *
     * This method MUST be implemented in such a way as to retain the
     * immutability of the message, and MUST return an instance that has the
     * new and/or updated header and value.
     *
     * @param string $name Case-insensitive header field name.
     * @param string|non-empty-array<string> $value Header value(s).
     * @return static
     * @throws \InvalidArgumentException for invalid header names or values.
     */
    public function with_header(string $name, $value);

    /**
     * Retrieves a comma-separated string of the values for a single header.
     *
     * This method returns all of the header values of the given
     * case-insensitive header name as a string concatenated together using
     * a comma.
     *
     * NOTE: Not all header values may be appropriately represented using
     * comma concatenation. For such headers, use getHeader() instead
     * and supply your own delimiter when concatenating.
     *
     * If the header does not appear in the message, this method MUST return
     * an empty string.
     *
     * @param string $name Case-insensitive header field name.
     * @return string A string of values as provided for the given header
     *    concatenated together using a comma. If the header does not appear in
     *    the message, this method MUST return an empty string.
     */
    public function get_header_line(string $name): string;

    /**
     * get the body as string
     *
     * @return string
     */
    public function get_body_content(): string;
}

}

namespace SimplePie {

/**
 * IRI parser/serialiser/normaliser
 *
 * @property ?string $scheme
 * @property ?string $userinfo
 * @property ?string $host
 * @property ?int $port
 * @property-write int|string|null $port
 * @property ?string $authority
 * @property string $path
 * @property ?string $query
 * @property ?string $fragment
 */
class IRI
{
    /**
     * Scheme
     *
     * @var ?string
     */
    protected $scheme = null;

    /**
     * User Information
     *
     * @var ?string
     */
    protected $iuserinfo = null;

    /**
     * ihost
     *
     * @var ?string
     */
    protected $ihost = null;

    /**
     * Port
     *
     * @var ?int
     */
    protected $port = null;

    /**
     * ipath
     *
     * @var string
     */
    protected $ipath = '';

    /**
     * iquery
     *
     * @var ?string
     */
    protected $iquery = null;

    /**
     * ifragment
     *
     * @var ?string
     */
    protected $ifragment = null;

    /**
     * Normalization database
     *
     * Each key is the scheme, each value is an array with each key as the IRI
     * part and value as the default value for that part.
     *
     * @var array<string, array<string, mixed>>
     */
    protected $normalization = [
        'acap' => [
            'port' => 674
        ],
        'dict' => [
            'port' => 2628
        ],
        'file' => [
            'ihost' => 'localhost'
        ],
        'http' => [
            'port' => 80,
            'ipath' => '/'
        ],
        'https' => [
            'port' => 443,
            'ipath' => '/'
        ],
    ];

    /**
     * Return the entire IRI when you try and read the object as a string
     *
     * @return string
     */
    public function __toString()
    {
        return (string) $this->get_iri();
    }

    /**
     * Overload __set() to provide access via properties
     *
     * @param string $name Property name
     * @param mixed $value Property value
     * @return void
     */
    public function __set(string $name, $value)
    {
        $callable = [$this, 'set_' . $name];
        if (is_callable($callable)) {
            call_user_func($callable, $value);
        } elseif (
            $name === 'iauthority'
            || $name === 'iuserinfo'
            || $name === 'ihost'
            || $name === 'ipath'
            || $name === 'iquery'
            || $name === 'ifragment'
        ) {
            call_user_func([$this, 'set_' . substr($name, 1)], $value);
        }
    }

    /**
     * Overload __get() to provide access via properties
     *
     * @param string $name Property name
     * @return mixed
     */
    public function __get(string $name)
    {
        // isset() returns false for null, we don't want to do that
        // Also why we use array_key_exists below instead of isset()
        $props = get_object_vars($this);

        if (
            $name === 'iri' ||
            $name === 'uri' ||
            $name === 'iauthority' ||
            $name === 'authority'
        ) {
            $return = $this->{"get_$name"}();
        } elseif (array_key_exists($name, $props)) {
            $return = $this->$name;
        }
        // host -> ihost
        elseif (array_key_exists($prop = 'i' . $name, $props)) {
            $name = $prop;
            $return = $this->$prop;
        }
        // ischeme -> scheme
        elseif (($prop = substr($name, 1)) && array_key_exists($prop, $props)) {
            $name = $prop;
            $return = $this->$prop;
        } else {
            trigger_error('Undefined property: ' . get_class($this) . '::' . $name, E_USER_NOTICE);
            $return = null;
        }

        if ($return === null && isset($this->normalization[$this->scheme][$name])) {
            return $this->normalization[$this->scheme][$name];
        }

        return $return;
    }

    /**
     * Overload __isset() to provide access via properties
     *
     * @param string $name Property name
     * @return bool
     */
    public function __isset(string $name)
    {
        return method_exists($this, 'get_' . $name) || isset($this->$name);
    }

    /**
     * Overload __unset() to provide access via properties
     *
     * @param string $name Property name
     * @return void
     */
    public function __unset(string $name)
    {
        $callable = [$this, 'set_' . $name];
        if (is_callable($callable)) {
            call_user_func($callable, '');
        }
    }

    /**
     * Create a new IRI object, from a specified string
     *
     * @param string|null $iri
     */
    public function __construct(?string $iri = null)
    {
        $this->set_iri($iri);
    }

    /**
     * Clean up
     * @return void
     */
    public function __destruct()
    {
        $this->set_iri(null, true);
        $this->set_path(null, true);
        $this->set_authority(null, true);
    }

    /**
     * Create a new IRI object by resolving a relative IRI
     *
     * Returns false if $base is not absolute, otherwise an IRI.
     *
     * @param IRI|string $base (Absolute) Base IRI
     * @param IRI|string $relative Relative IRI
     * @return IRI|false
     */
    public static function absolutize($base, $relative)
    {
        if (!($relative instanceof IRI)) {
            $relative = new IRI($relative);
        }
        if (!$relative->is_valid()) {
            return false;
        } elseif ($relative->scheme !== null) {
            return clone $relative;
        } else {
            if (!($base instanceof IRI)) {
                $base = new IRI($base);
            }
            if ($base->scheme !== null && $base->is_valid()) {
                if ($relative->get_iri() !== '') {
                    if ($relative->iuserinfo !== null || $relative->ihost !== null || $relative->port !== null) {
                        $target = clone $relative;
                        $target->scheme = $base->scheme;
                    } else {
                        $target = new IRI();
                        $target->scheme = $base->scheme;
                        $target->iuserinfo = $base->iuserinfo;
                        $target->ihost = $base->ihost;
                        $target->port = $base->port;
                        if ($relative->ipath !== '') {
                            if ($relative->ipath[0] === '/') {
                                $target->ipath = $relative->ipath;
                            } elseif (($base->iuserinfo !== null || $base->ihost !== null || $base->port !== null) && $base->ipath === '') {
                                $target->ipath = '/' . $relative->ipath;
                            } elseif (($last_segment = strrpos($base->ipath, '/')) !== false) {
                                $target->ipath = substr($base->ipath, 0, $last_segment + 1) . $relative->ipath;
                            } else {
                                $target->ipath = $relative->ipath;
                            }
                            $target->ipath = $target->remove_dot_segments($target->ipath);
                            $target->iquery = $relative->iquery;
                        } else {
                            $target->ipath = $base->ipath;
                            if ($relative->iquery !== null) {
                                $target->iquery = $relative->iquery;
                            } elseif ($base->iquery !== null) {
                                $target->iquery = $base->iquery;
                            }
                        }
                        $target->ifragment = $relative->ifragment;
                    }
                } else {
                    $target = clone $base;
                    $target->ifragment = null;
                }
                $target->scheme_normalization();
                return $target;
            }

            return false;
        }
    }

    /**
     * Parse an IRI into scheme/authority/path/query/fragment segments
     *
     * @param string $iri
     * @return array{
     *   scheme: string|null,
     *   authority: string|null,
     *   path: string,
     *   query: string|null,
     *   fragment: string|null,
     * }|false
     */
    protected function parse_iri(string $iri)
    {
        $iri = trim($iri, "\x20\x09\x0A\x0C\x0D");
        if (preg_match('/^(?:(?P<scheme>[^:\/?#]+):)?(:?\/\/(?P<authority>[^\/?#]*))?(?P<path>[^?#]*)(?:\?(?P<query>[^#]*))?(?:#(?P<fragment>.*))?$/', $iri, $match, \PREG_UNMATCHED_AS_NULL)) {
            // TODO: Remove once we require PHP ≥ 7.4.
            $match['query'] = $match['query'] ?? null;
            $match['fragment'] = $match['fragment'] ?? null;
            return $match;
        }

        // This can occur when a paragraph is accidentally parsed as a URI
        return false;
    }

    /**
     * Remove dot segments from a path
     *
     * @param string $input
     * @return string
     */
    protected function remove_dot_segments(string $input)
    {
        $output = '';
        while (strpos($input, './') !== false || strpos($input, '/.') !== false || $input === '.' || $input === '..') {
            // A: If the input buffer begins with a prefix of "../" or "./", then remove that prefix from the input buffer; otherwise,
            if (strpos($input, '../') === 0) {
                $input = substr($input, 3);
            } elseif (strpos($input, './') === 0) {
                $input = substr($input, 2);
            }
            // B: if the input buffer begins with a prefix of "/./" or "/.", where "." is a complete path segment, then replace that prefix with "/" in the input buffer; otherwise,
            elseif (strpos($input, '/./') === 0) {
                $input = substr($input, 2);
            } elseif ($input === '/.') {
                $input = '/';
            }
            // C: if the input buffer begins with a prefix of "/../" or "/..", where ".." is a complete path segment, then replace that prefix with "/" in the input buffer and remove the last segment and its preceding "/" (if any) from the output buffer; otherwise,
            elseif (strpos($input, '/../') === 0) {
                $input = substr($input, 3);
                $output = substr_replace($output, '', intval(strrpos($output, '/')));
            } elseif ($input === '/..') {
                $input = '/';
                $output = substr_replace($output, '', intval(strrpos($output, '/')));
            }
            // D: if the input buffer consists only of "." or "..", then remove that from the input buffer; otherwise,
            elseif ($input === '.' || $input === '..') {
                $input = '';
            }
            // E: move the first path segment in the input buffer to the end of the output buffer, including the initial "/" character (if any) and any subsequent characters up to, but not including, the next "/" character or the end of the input buffer
            elseif (($pos = strpos($input, '/', 1)) !== false) {
                $output .= substr($input, 0, $pos);
                $input = substr_replace($input, '', 0, $pos);
            } else {
                $output .= $input;
                $input = '';
            }
        }
        return $output . $input;
    }

    /**
     * Replace invalid character with percent encoding
     *
     * @param string $string Input string
     * @param string $extra_chars Valid characters not in iunreserved or
     *                            iprivate (this is ASCII-only)
     * @param bool $iprivate Allow iprivate
     * @return string
     */
    protected function replace_invalid_with_pct_encoding(string $string, string $extra_chars, bool $iprivate = false)
    {
        // Normalize as many pct-encoded sections as possible
        $string = preg_replace_callback('/(?:%[A-Fa-f0-9]{2})+/', [$this, 'remove_iunreserved_percent_encoded'], $string);
        \assert(\is_string($string), "For PHPStan: Should not occur, the regex is valid");

        // Replace invalid percent characters
        $string = preg_replace('/%(?![A-Fa-f0-9]{2})/', '%25', $string);
        \assert(\is_string($string), "For PHPStan: Should not occur, the regex is valid");

        // Add unreserved and % to $extra_chars (the latter is safe because all
        // pct-encoded sections are now valid).
        $extra_chars .= 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-._~%';

        // Now replace any bytes that aren't allowed with their pct-encoded versions
        $position = 0;
        $strlen = strlen($string);
        while (($position += strspn($string, $extra_chars, $position)) < $strlen) {
            $value = ord($string[$position]);
            $character = 0;

            // Start position
            $start = $position;

            // By default we are valid
            $valid = true;

            // No one byte sequences are valid due to the while.
            // Two byte sequence:
            if (($value & 0xE0) === 0xC0) {
                $character = ($value & 0x1F) << 6;
                $length = 2;
                $remaining = 1;
            }
            // Three byte sequence:
            elseif (($value & 0xF0) === 0xE0) {
                $character = ($value & 0x0F) << 12;
                $length = 3;
                $remaining = 2;
            }
            // Four byte sequence:
            elseif (($value & 0xF8) === 0xF0) {
                $character = ($value & 0x07) << 18;
                $length = 4;
                $remaining = 3;
            }
            // Invalid byte:
            else {
                $valid = false;
                $length = 1;
                $remaining = 0;
            }

            if ($remaining) {
                if ($position + $length <= $strlen) {
                    for ($position++; $remaining; $position++) {
                        $value = ord($string[$position]);

                        // Check that the byte is valid, then add it to the character:
                        if (($value & 0xC0) === 0x80) {
                            $character |= ($value & 0x3F) << (--$remaining * 6);
                        }
                        // If it is invalid, count the sequence as invalid and reprocess the current byte:
                        else {
                            $valid = false;
                            $position--;
                            break;
                        }
                    }
                } else {
                    $position = $strlen - 1;
                    $valid = false;
                }
            }

            // Percent encode anything invalid or not in ucschar
            if (
                // Invalid sequences
                !$valid
                // Non-shortest form sequences are invalid
                || $length > 1 && $character <= 0x7F
                || $length > 2 && $character <= 0x7FF
                || $length > 3 && $character <= 0xFFFF
                // Outside of range of ucschar codepoints
                // Noncharacters
                || ($character & 0xFFFE) === 0xFFFE
                || $character >= 0xFDD0 && $character <= 0xFDEF
                || (
                    // Everything else not in ucschar
                    $character > 0xD7FF && $character < 0xF900
                    || $character < 0xA0
                    || $character > 0xEFFFD
                )
                && (
                    // Everything not in iprivate, if it applies
                    !$iprivate
                    || $character < 0xE000
                    || $character > 0x10FFFD
                )
            ) {
                // If we were a character, pretend we weren't, but rather an error.
                if ($valid) {
                    $position--;
                }

                for ($j = $start; $j <= $position; $j++) {
                    $string = substr_replace($string, sprintf('%%%02X', ord($string[$j])), $j, 1);
                    $j += 2;
                    $position += 2;
                    $strlen += 2;
                }
            }
        }

        return $string;
    }

    /**
     * Callback function for preg_replace_callback.
     *
     * Removes sequences of percent encoded bytes that represent UTF-8
     * encoded characters in iunreserved
     *
     * @param array{string} $match PCRE match, a capture group #0 consisting of a sequence of valid percent-encoded bytes
     * @return string Replacement
     */
    protected function remove_iunreserved_percent_encoded(array $match)
    {
        // As we just have valid percent encoded sequences we can just explode
        // and ignore the first member of the returned array (an empty string).
        $bytes = explode('%', $match[0]);

        // Initialize the new string (this is what will be returned) and that
        // there are no bytes remaining in the current sequence (unsurprising
        // at the first byte!).
        $string = '';
        $remaining = 0;

        // these variables will be initialized in the loop but PHPStan is not able to detect it currently
        $start = 0;
        $character = 0;
        $length = 0;
        $valid = true;

        // Loop over each and every byte, and set $value to its value
        for ($i = 1, $len = count($bytes); $i < $len; $i++) {
            $value = hexdec($bytes[$i]);

            // If we're the first byte of sequence:
            if (!$remaining) {
                // Start position
                $start = $i;

                // By default we are valid
                $valid = true;

                // One byte sequence:
                if ($value <= 0x7F) {
                    $character = $value;
                    $length = 1;
                }
                // Two byte sequence:
                elseif (($value & 0xE0) === 0xC0) {
                    $character = ($value & 0x1F) << 6;
                    $length = 2;
                    $remaining = 1;
                }
                // Three byte sequence:
                elseif (($value & 0xF0) === 0xE0) {
                    $character = ($value & 0x0F) << 12;
                    $length = 3;
                    $remaining = 2;
                }
                // Four byte sequence:
                elseif (($value & 0xF8) === 0xF0) {
                    $character = ($value & 0x07) << 18;
                    $length = 4;
                    $remaining = 3;
                }
                // Invalid byte:
                else {
                    $valid = false;
                    $remaining = 0;
                }
            }
            // Continuation byte:
            else {
                // Check that the byte is valid, then add it to the character:
                if (($value & 0xC0) === 0x80) {
                    $remaining--;
                    $character |= ($value & 0x3F) << ($remaining * 6);
                }
                // If it is invalid, count the sequence as invalid and reprocess the current byte as the start of a sequence:
                else {
                    $valid = false;
                    $remaining = 0;
                    $i--;
                }
            }

            // If we've reached the end of the current byte sequence, append it to Unicode::$data
            if (!$remaining) {
                // Percent encode anything invalid or not in iunreserved
                if (
                    // Invalid sequences
                    !$valid
                    // Non-shortest form sequences are invalid
                    || $length > 1 && $character <= 0x7F
                    || $length > 2 && $character <= 0x7FF
                    || $length > 3 && $character <= 0xFFFF
                    // Outside of range of iunreserved codepoints
                    || $character < 0x2D
                    || $character > 0xEFFFD
                    // Noncharacters
                    || ($character & 0xFFFE) === 0xFFFE
                    || $character >= 0xFDD0 && $character <= 0xFDEF
                    // Everything else not in iunreserved (this is all BMP)
                    || $character === 0x2F
                    || $character > 0x39 && $character < 0x41
                    || $character > 0x5A && $character < 0x61
                    || $character > 0x7A && $character < 0x7E
                    || $character > 0x7E && $character < 0xA0
                    || $character > 0xD7FF && $character < 0xF900
                ) {
                    for ($j = $start; $j <= $i; $j++) {
                        $string .= '%' . strtoupper($bytes[$j]);
                    }
                } else {
                    for ($j = $start; $j <= $i; $j++) {
                        // Cast for PHPStan, this will always be a number between 0 and 0xFF so hexdec will return int.
                        $string .= chr((int) hexdec($bytes[$j]));
                    }
                }
            }
        }

        // If we have any bytes left over they are invalid (i.e., we are
        // mid-way through a multi-byte sequence)
        if ($remaining) {
            for ($j = $start; $j < $len; $j++) {
                $string .= '%' . strtoupper($bytes[$j]);
            }
        }

        return $string;
    }

    /**
     * @return void
     */
    protected function scheme_normalization()
    {
        if (isset($this->normalization[$this->scheme]['iuserinfo']) && $this->iuserinfo === $this->normalization[$this->scheme]['iuserinfo']) {
            $this->iuserinfo = null;
        }
        if (isset($this->normalization[$this->scheme]['ihost']) && $this->ihost === $this->normalization[$this->scheme]['ihost']) {
            $this->ihost = null;
        }
        if (isset($this->normalization[$this->scheme]['port']) && $this->port === $this->normalization[$this->scheme]['port']) {
            $this->port = null;
        }
        if (isset($this->normalization[$this->scheme]['ipath']) && $this->ipath === $this->normalization[$this->scheme]['ipath']) {
            $this->ipath = '';
        }
        if (isset($this->normalization[$this->scheme]['iquery']) && $this->iquery === $this->normalization[$this->scheme]['iquery']) {
            $this->iquery = null;
        }
        if (isset($this->normalization[$this->scheme]['ifragment']) && $this->ifragment === $this->normalization[$this->scheme]['ifragment']) {
            $this->ifragment = null;
        }
    }

    /**
     * Check if the object represents a valid IRI. This needs to be done on each
     * call as some things change depending on another part of the IRI.
     *
     * @return bool
     */
    public function is_valid()
    {
        if ($this->ipath === '') {
            return true;
        }

        $isauthority = $this->iuserinfo !== null || $this->ihost !== null ||
            $this->port !== null;
        if ($isauthority && $this->ipath[0] === '/') {
            return true;
        }

        if (!$isauthority && (substr($this->ipath, 0, 2) === '//')) {
            return false;
        }

        // Relative urls cannot have a colon in the first path segment (and the
        // slashes themselves are not included so skip the first character).
        if (!$this->scheme && !$isauthority &&
            strpos($this->ipath, ':') !== false &&
            strpos($this->ipath, '/', 1) !== false &&
            strpos($this->ipath, ':') < strpos($this->ipath, '/', 1)) {
            return false;
        }

        return true;
    }

    /**
     * Set the entire IRI. Returns true on success, false on failure (if there
     * are any invalid characters).
     *
     * @param string|null $iri
     * @return bool
     */
    public function set_iri(?string $iri, bool $clear_cache = false)
    {
        static $cache;
        if ($clear_cache) {
            $cache = null;
            return false;
        }
        if (!$cache) {
            $cache = [];
        }

        if ($iri === null) {
            return true;
        } elseif (isset($cache[$iri])) {
            [
                $this->scheme,
                $this->iuserinfo,
                $this->ihost,
                $this->port,
                $this->ipath,
                $this->iquery,
                $this->ifragment,
                $return
            ] = $cache[$iri];

            return $return;
        }

        $parsed = $this->parse_iri((string) $iri);
        if (!$parsed) {
            return false;
        }

        $return = $this->set_scheme($parsed['scheme'])
            && $this->set_authority($parsed['authority'])
            && $this->set_path($parsed['path'])
            && $this->set_query($parsed['query'])
            && $this->set_fragment($parsed['fragment']);

        $cache[$iri] = [
            $this->scheme,
            $this->iuserinfo,
            $this->ihost,
            $this->port,
            $this->ipath,
            $this->iquery,
            $this->ifragment,
            $return
        ];

        return $return;
    }

    /**
     * Set the scheme. Returns true on success, false on failure (if there are
     * any invalid characters).
     *
     * @param string|null $scheme
     * @return bool
     */
    public function set_scheme(?string $scheme)
    {
        if ($scheme === null) {
            $this->scheme = null;
        } elseif (!preg_match('/^[A-Za-z][0-9A-Za-z+\-.]*$/', $scheme)) {
            $this->scheme = null;
            return false;
        } else {
            $this->scheme = strtolower($scheme);
        }
        return true;
    }

    /**
     * Set the authority. Returns true on success, false on failure (if there are
     * any invalid characters).
     *
     * @param string|null $authority
     * @return bool
     */
    public function set_authority(?string $authority, bool $clear_cache = false)
    {
        static $cache;
        if ($clear_cache) {
            $cache = null;
            return false;
        }
        if (!$cache) {
            $cache = [];
        }

        if ($authority === null) {
            $this->iuserinfo = null;
            $this->ihost = null;
            $this->port = null;
            return true;
        } elseif (isset($cache[$authority])) {
            [
                $this->iuserinfo,
                $this->ihost,
                $this->port,
                $return
            ] = $cache[$authority];

            return $return;
        }

        $remaining = $authority;
        if (($iuserinfo_end = strrpos($remaining, '@')) !== false) {
            // Cast for PHPStan on PHP < 8.0. It does not detect that
            // the range is not flipped so substr cannot return false.
            $iuserinfo = (string) substr($remaining, 0, $iuserinfo_end);
            $remaining = substr($remaining, $iuserinfo_end + 1);
        } else {
            $iuserinfo = null;
        }
        if (($port_start = strpos($remaining, ':', intval(strpos($remaining, ']')))) !== false) {
            $port = substr($remaining, $port_start + 1);
            if ($port === false) {
                $port = null;
            }
            $remaining = substr($remaining, 0, $port_start);
        } else {
            $port = null;
        }

        $return = $this->set_userinfo($iuserinfo) &&
                  $this->set_host($remaining) &&
                  $this->set_port($port);

        $cache[$authority] = [
            $this->iuserinfo,
            $this->ihost,
            $this->port,
            $return
        ];

        return $return;
    }

    /**
     * Set the iuserinfo.
     *
     * @param string|null $iuserinfo
     * @return bool
     */
    public function set_userinfo(?string $iuserinfo)
    {
        if ($iuserinfo === null) {
            $this->iuserinfo = null;
        } else {
            $this->iuserinfo = $this->replace_invalid_with_pct_encoding($iuserinfo, '!$&\'()*+,;=:');
            $this->scheme_normalization();
        }

        return true;
    }

    /**
     * Set the ihost. Returns true on success, false on failure (if there are
     * any invalid characters).
     *
     * @param string|null $ihost
     * @return bool
     */
    public function set_host(?string $ihost)
    {
        if ($ihost === null) {
            $this->ihost = null;
            return true;
        } elseif (substr($ihost, 0, 1) === '[' && substr($ihost, -1) === ']') {
            if (\SimplePie\Net\IPv6::check_ipv6(substr($ihost, 1, -1))) {
                $this->ihost = '[' . \SimplePie\Net\IPv6::compress(substr($ihost, 1, -1)) . ']';
            } else {
                $this->ihost = null;
                return false;
            }
        } else {
            $ihost = $this->replace_invalid_with_pct_encoding($ihost, '!$&\'()*+,;=');

            // Lowercase, but ignore pct-encoded sections (as they should
            // remain uppercase). This must be done after the previous step
            // as that can add unescaped characters.
            $position = 0;
            $strlen = strlen($ihost);
            while (($position += strcspn($ihost, 'ABCDEFGHIJKLMNOPQRSTUVWXYZ%', $position)) < $strlen) {
                if ($ihost[$position] === '%') {
                    $position += 3;
                } else {
                    $ihost[$position] = strtolower($ihost[$position]);
                    $position++;
                }
            }

            $this->ihost = $ihost;
        }

        $this->scheme_normalization();

        return true;
    }

    /**
     * Set the port. Returns true on success, false on failure (if there are
     * any invalid characters).
     *
     * @param string|int|null $port
     * @return bool
     */
    public function set_port($port)
    {
        if ($port === null) {
            $this->port = null;
            return true;
        } elseif (strspn((string) $port, '0123456789') === strlen((string) $port)) {
            $this->port = (int) $port;
            $this->scheme_normalization();
            return true;
        }

        $this->port = null;
        return false;
    }

    /**
     * Set the ipath.
     *
     * @param string|null $ipath
     * @return bool
     */
    public function set_path(?string $ipath, bool $clear_cache = false)
    {
        static $cache;
        if ($clear_cache) {
            $cache = null;
            return false;
        }
        if (!$cache) {
            $cache = [];
        }

        $ipath = (string) $ipath;

        if (isset($cache[$ipath])) {
            $this->ipath = $cache[$ipath][(int) ($this->scheme !== null)];
        } else {
            $valid = $this->replace_invalid_with_pct_encoding($ipath, '!$&\'()*+,;=@:/');
            $removed = $this->remove_dot_segments($valid);

            $cache[$ipath] = [$valid, $removed];
            $this->ipath =  ($this->scheme !== null) ? $removed : $valid;
        }

        $this->scheme_normalization();
        return true;
    }

    /**
     * Set the iquery.
     *
     * @param string|null $iquery
     * @return bool
     */
    public function set_query(?string $iquery)
    {
        if ($iquery === null) {
            $this->iquery = null;
        } else {
            $this->iquery = $this->replace_invalid_with_pct_encoding($iquery, '!$&\'()*+,;=:@/?', true);
            $this->scheme_normalization();
        }
        return true;
    }

    /**
     * Set the ifragment.
     *
     * @param string|null $ifragment
     * @return bool
     */
    public function set_fragment(?string $ifragment)
    {
        if ($ifragment === null) {
            $this->ifragment = null;
        } else {
            $this->ifragment = $this->replace_invalid_with_pct_encoding($ifragment, '!$&\'()*+,;=:@/?');
            $this->scheme_normalization();
        }
        return true;
    }

    /**
     * Convert an IRI to a URI (or parts thereof)
     *
     * @param string $string
     * @return string
     */
    public function to_uri(string $string)
    {
        static $non_ascii;
        if (!$non_ascii) {
            $non_ascii = implode('', range("\x80", "\xFF"));
        }

        $position = 0;
        $strlen = strlen($string);
        while (($position += strcspn($string, $non_ascii, $position)) < $strlen) {
            $string = substr_replace($string, sprintf('%%%02X', ord($string[$position])), $position, 1);
            $position += 3;
            $strlen += 2;
        }

        return $string;
    }

    /**
     * Get the complete IRI
     *
     * @return string|false
     */
    public function get_iri()
    {
        if (!$this->is_valid()) {
            return false;
        }

        $iri = '';
        if ($this->scheme !== null) {
            $iri .= $this->scheme . ':';
        }
        if (($iauthority = $this->get_iauthority()) !== null) {
            $iri .= '//' . $iauthority;
        }
        if ($this->ipath !== '') {
            $iri .= $this->ipath;
        } elseif (!empty($this->normalization[$this->scheme]['ipath']) && $iauthority !== null && $iauthority !== '') {
            $iri .= $this->normalization[$this->scheme]['ipath'];
        }
        if ($this->iquery !== null) {
            $iri .= '?' . $this->iquery;
        }
        if ($this->ifragment !== null) {
            $iri .= '#' . $this->ifragment;
        }

        return $iri;
    }

    /**
     * Get the complete URI
     *
     * @return string
     */
    public function get_uri()
    {
        return $this->to_uri((string) $this->get_iri());
    }

    /**
     * Get the complete iauthority
     *
     * @return ?string
     */
    protected function get_iauthority()
    {
        if ($this->iuserinfo !== null || $this->ihost !== null || $this->port !== null) {
            $iauthority = '';
            if ($this->iuserinfo !== null) {
                $iauthority .= $this->iuserinfo . '@';
            }
            if ($this->ihost !== null) {
                $iauthority .= $this->ihost;
            }
            if ($this->port !== null && $this->port !== 0) {
                $iauthority .= ':' . $this->port;
            }
            return $iauthority;
        }

        return null;
    }

    /**
     * Get the complete authority
     *
     * @return ?string
     */
    protected function get_authority()
    {
        $iauthority = $this->get_iauthority();
        if (is_string($iauthority)) {
            return $this->to_uri($iauthority);
        }

        return $iauthority;
    }
}

class_alias('SimplePie\IRI', 'SimplePie_IRI');

}

namespace SimplePie {

/**
 * Manages all item-related data
 *
 * Used by {@see \SimplePie\SimplePie::get_item()} and {@see \SimplePie\SimplePie::get_items()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_item_class()}
 */
class Item implements RegistryAware
{
    /**
     * Parent feed
     *
     * @access private
     * @var \SimplePie\SimplePie
     */
    public $feed;

    /**
     * Raw data
     *
     * @access private
     * @var array<string, mixed>
     */
    public $data = [];

    /**
     * Registry object
     *
     * @see set_registry
     * @var \SimplePie\Registry
     */
    protected $registry;

    /**
     * @var Sanitize|null
     */
    private $sanitize = null;

    /**
     * Create a new item object
     *
     * This is usually used by {@see \SimplePie\SimplePie::get_items} and
     * {@see \SimplePie\SimplePie::get_item}. Avoid creating this manually.
     *
     * @param \SimplePie\SimplePie $feed Parent feed
     * @param array<string, mixed> $data Raw data
     */
    public function __construct(\SimplePie\SimplePie $feed, array $data)
    {
        $this->feed = $feed;
        $this->data = $data;
    }

    /**
     * Set the registry handler
     *
     * This is usually used by {@see \SimplePie\Registry::create}
     *
     * @since 1.3
     * @param \SimplePie\Registry $registry
     * @return void
     */
    public function set_registry(\SimplePie\Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * Get a string representation of the item
     *
     * @return string
     */
    public function __toString()
    {
        return md5(serialize($this->data));
    }

    /**
     * Remove items that link back to this before destroying this object
     */
    public function __destruct()
    {
        if (!gc_enabled()) {
            unset($this->feed);
        }
    }

    /**
     * Get data for an item-level element
     *
     * This method allows you to get access to ANY element/attribute that is a
     * sub-element of the item/entry tag.
     *
     * See {@see \SimplePie\SimplePie::get_feed_tags()} for a description of the return value
     *
     * @since 1.0
     * @see http://simplepie.org/wiki/faq/supported_xml_namespaces
     * @param string $namespace The URL of the XML namespace of the elements you're trying to access
     * @param string $tag Tag name
     * @return array<array<string, mixed>>|null
     */
    public function get_item_tags(string $namespace, string $tag)
    {
        if (isset($this->data['child'][$namespace][$tag])) {
            return $this->data['child'][$namespace][$tag];
        }

        return null;
    }

    /**
     * Get base URL of the item itself.
     * Returns `<xml:base>` or feed base URL.
     * Similar to `Item::get_base()` but can safely be used during initialisation methods
     * such as `Item::get_links()` (`Item::get_base()` and `Item::get_links()` call each-other)
     * and is not affected by enclosures.
     *
     * @param array<string, mixed> $element
     * @see get_base
     */
    private function get_own_base(array $element = []): string
    {
        if (!empty($element['xml_base_explicit']) && isset($element['xml_base'])) {
            return $element['xml_base'];
        }
        return $this->feed->get_base();
    }

    /**
     * Get the base URL value.
     * Uses `<xml:base>`, or item link, or enclosure link, or feed base URL.
     *
     * @param array<string, mixed> $element
     * @return string
     */
    public function get_base(array $element = [])
    {
        if (!empty($element['xml_base_explicit']) && isset($element['xml_base'])) {
            return $element['xml_base'];
        }
        $link = $this->get_permalink();
        if ($link != null) {
            return $link;
        }
        return $this->feed->get_base($element);
    }

    /**
     * Sanitize feed data
     *
     * @access private
     * @see \SimplePie\SimplePie::sanitize()
     * @param string $data Data to sanitize
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @param string $base Base URL to resolve URLs against
     * @return string Sanitized data
     */
    public function sanitize(string $data, int $type, string $base = '')
    {
        // This really returns string|false but changing encoding is uncommon and we are going to deprecate it, so let’s just lie to PHPStan in the interest of cleaner annotations.
        return $this->feed->sanitize($data, $type, $base);
    }

    /**
     * Get the parent feed
     *
     * Note: this may not work as you think for multifeeds!
     *
     * @link http://simplepie.org/faq/typical_multifeed_gotchas#missing_data_from_feed
     * @since 1.0
     * @return \SimplePie\SimplePie
     */
    public function get_feed()
    {
        return $this->feed;
    }

    /**
     * Get the unique identifier for the item
     *
     * This is usually used when writing code to check for new items in a feed.
     *
     * Uses `<atom:id>`, `<guid>`, `<dc:identifier>` or the `about` attribute
     * for RDF. If none of these are supplied (or `$hash` is true), creates an
     * MD5 hash based on the permalink, title and content.
     *
     * @since Beta 2
     * @param bool $hash Should we force using a hash instead of the supplied ID?
     * @param string|false $fn User-supplied function to generate an hash
     * @return string|null
     */
    public function get_id(bool $hash = false, $fn = 'md5')
    {
        if (!$hash) {
            if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'id')) {
                return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'id')) {
                return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'guid')) {
                return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'identifier')) {
                return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'identifier')) {
                return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif (isset($this->data['attribs'][\SimplePie\SimplePie::NAMESPACE_RDF]['about'])) {
                return $this->sanitize($this->data['attribs'][\SimplePie\SimplePie::NAMESPACE_RDF]['about'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
        }
        if ($fn === false) {
            return null;
        } elseif (!is_callable($fn)) {
            trigger_error('User-supplied function $fn must be callable', E_USER_WARNING);
            $fn = 'md5';
        }
        return call_user_func(
            $fn,
            $this->get_permalink().$this->get_title().$this->get_content()
        );
    }

    /**
     * Get the title of the item
     *
     * Uses `<atom:title>`, `<title>` or `<dc:title>`
     *
     * @since Beta 2 (previously called `get_item_title` since 0.8)
     * @return string|null
     */
    public function get_title()
    {
        if (!isset($this->data['title'])) {
            if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'title')) {
                $this->data['title'] = $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } else {
                $this->data['title'] = null;
            }
        }
        return $this->data['title'];
    }

    /**
     * Get the content for the item
     *
     * Prefers summaries over full content , but will return full content if a
     * summary does not exist.
     *
     * To prefer full content instead, use {@see get_content}
     *
     * Uses `<atom:summary>`, `<description>`, `<dc:description>` or
     * `<itunes:subtitle>`
     *
     * @since 0.8
     * @param bool $description_only Should we avoid falling back to the content?
     * @return string|null
     */
    public function get_description(bool $description_only = false)
    {
        if (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'summary')) &&
            ($return = $this->sanitize($tags[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$tags[0]['attribs']]), $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'summary')) &&
                ($return = $this->sanitize($tags[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$tags[0]['attribs']]), $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'description')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'description')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML, $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'description')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'description')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'summary')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML, $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'subtitle')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'description')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML))) {
            return $return;
        } elseif (!$description_only) {
            return $this->get_content(true);
        }

        return null;
    }

    /**
     * Get the content for the item
     *
     * Prefers full content over summaries, but will return a summary if full
     * content does not exist.
     *
     * To prefer summaries instead, use {@see get_description}
     *
     * Uses `<atom:content>` or `<content:encoded>` (RSS 1.0 Content Module)
     *
     * @since 1.0
     * @param bool $content_only Should we avoid falling back to the description?
     * @return string|null
     */
    public function get_content(bool $content_only = false)
    {
        if (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'content')) &&
            ($return = $this->sanitize($tags[0]['data'], $this->registry->call(Misc::class, 'atom_10_content_construct_type', [$tags[0]['attribs']]), $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'content')) &&
                ($return = $this->sanitize($tags[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$tags[0]['attribs']]), $this->get_base($tags[0])))) {
            return $return;
        } elseif (($tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10_MODULES_CONTENT, 'encoded')) &&
                ($return = $this->sanitize($tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML, $this->get_base($tags[0])))) {
            return $return;
        } elseif (!$content_only) {
            return $this->get_description(true);
        }

        return null;
    }

    /**
     * Get the media:thumbnail of the item
     *
     * Uses `<media:thumbnail>`
     *
     *
     * @return array{url: string, height?: string, width?: string, time?: string}|null
     */
    public function get_thumbnail()
    {
        if (!isset($this->data['thumbnail'])) {
            if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'thumbnail')) {
                $thumbnail = $return[0]['attribs'][''];
                if (empty($thumbnail['url'])) {
                    $this->data['thumbnail'] = null;
                } else {
                    $thumbnail['url'] = $this->sanitize($thumbnail['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($return[0]));
                    $this->data['thumbnail'] = $thumbnail;
                }
            } else {
                $this->data['thumbnail'] = null;
            }
        }
        return $this->data['thumbnail'];
    }

    /**
     * Get a category for the item
     *
     * @since Beta 3 (previously called `get_categories()` since Beta 2)
     * @param int $key The category that you want to return.  Remember that arrays begin with 0, not 1
     * @return \SimplePie\Category|null
     */
    public function get_category(int $key = 0)
    {
        $categories = $this->get_categories();
        if (isset($categories[$key])) {
            return $categories[$key];
        }

        return null;
    }

    /**
     * Get all categories for the item
     *
     * Uses `<atom:category>`, `<category>` or `<dc:subject>`
     *
     * @since Beta 3
     * @return \SimplePie\Category[]|null List of {@see \SimplePie\Category} objects
     */
    public function get_categories()
    {
        $categories = [];

        $type = 'category';
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, $type) as $category) {
            $term = null;
            $scheme = null;
            $label = null;
            if (isset($category['attribs']['']['term'])) {
                $term = $this->sanitize($category['attribs']['']['term'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['scheme'])) {
                $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['label'])) {
                $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label, $type]);
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, $type) as $category) {
            // This is really the label, but keep this as the term also for BC.
            // Label will also work on retrieving because that falls back to term.
            $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            if (isset($category['attribs']['']['domain'])) {
                $scheme = $this->sanitize($category['attribs']['']['domain'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } else {
                $scheme = null;
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, null, $type]);
        }

        $type = 'subject';
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, $type) as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null, $type]);
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, $type) as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null, $type]);
        }

        if (!empty($categories)) {
            return array_unique($categories);
        }

        return null;
    }

    /**
     * Get an author for the item
     *
     * @since Beta 2
     * @param int $key The author that you want to return.  Remember that arrays begin with 0, not 1
     * @return \SimplePie\Author|null
     */
    public function get_author(int $key = 0)
    {
        $authors = $this->get_authors();
        if (isset($authors[$key])) {
            return $authors[$key];
        }

        return null;
    }

    /**
     * Get a contributor for the item
     *
     * @since 1.1
     * @param int $key The contrbutor that you want to return.  Remember that arrays begin with 0, not 1
     * @return \SimplePie\Author|null
     */
    public function get_contributor(int $key = 0)
    {
        $contributors = $this->get_contributors();
        if (isset($contributors[$key])) {
            return $contributors[$key];
        }

        return null;
    }

    /**
     * Get all contributors for the item
     *
     * Uses `<atom:contributor>`
     *
     * @since 1.1
     * @return \SimplePie\Author[]|null List of {@see \SimplePie\Author} objects
     */
    public function get_contributors()
    {
        $contributors = [];
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'contributor') as $contributor) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'contributor') as $contributor) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }

        if (!empty($contributors)) {
            return array_unique($contributors);
        }

        return null;
    }

    /**
     * Get all authors for the item
     *
     * Uses `<atom:author>`, `<author>`, `<dc:creator>` or `<itunes:author>`
     *
     * @since Beta 2
     * @return \SimplePie\Author[]|null List of {@see \SimplePie\Author} objects
     */
    public function get_authors()
    {
        $authors = [];
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'author') as $author) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        if ($author = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'author')) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }
        if ($author = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'author')) {
            $authors[] = $this->registry->create(Author::class, [null, null, $this->sanitize($author[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT)]);
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'author') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }

        if (!empty($authors)) {
            return array_unique($authors);
        } elseif (($source = $this->get_source()) && ($authors = $source->get_authors())) {
            return $authors;
        } elseif ($authors = $this->feed->get_authors()) {
            return $authors;
        }

        return null;
    }

    /**
     * Get the copyright info for the item
     *
     * Uses `<atom:rights>` or `<dc:rights>`
     *
     * @since 1.1
     * @return string|null
     */
    public function get_copyright()
    {
        if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'rights')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'rights')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'rights')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * Get the posting date/time for the item
     *
     * Uses `<atom:published>`, `<atom:updated>`, `<atom:issued>`,
     * `<atom:modified>`, `<pubDate>` or `<dc:date>`
     *
     * Note: obeys PHP's timezone setting. To get a UTC date/time, use
     * {@see get_gmdate}
     *
     * @since Beta 2 (previously called `get_item_date` since 0.8)
     *
     * @param string $date_format Supports any PHP date format from {@see http://php.net/date} (empty for the raw data)
     * @return ($date_format is 'U' ? ?int : ?string)
     */
    public function get_date(string $date_format = 'j F Y, g:i a')
    {
        if (!isset($this->data['date'])) {
            if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'published')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'pubDate')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'date')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'date')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'updated')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'issued')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'created')) {
                $this->data['date']['raw'] = $return[0]['data'];
            } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'modified')) {
                $this->data['date']['raw'] = $return[0]['data'];
            }

            if (!empty($this->data['date']['raw'])) {
                $parser = $this->registry->call(Parse\Date::class, 'get');
                $this->data['date']['parsed'] = $parser->parse($this->data['date']['raw']) ?: null;
            } else {
                $this->data['date'] = null;
            }
        }
        if ($this->data['date']) {
            switch ($date_format) {
                case '':
                    return $this->sanitize($this->data['date']['raw'], \SimplePie\SimplePie::CONSTRUCT_TEXT);

                case 'U':
                    return $this->data['date']['parsed'];

                default:
                    return date($date_format, $this->data['date']['parsed']);
            }
        }

        return null;
    }

    /**
     * Get the update date/time for the item
     *
     * Uses `<atom:updated>`
     *
     * Note: obeys PHP's timezone setting. To get a UTC date/time, use
     * {@see get_gmdate}
     *
     * @param string $date_format Supports any PHP date format from {@see http://php.net/date} (empty for the raw data)
     * @return ($date_format is 'U' ? ?int : ?string)
     */
    public function get_updated_date(string $date_format = 'j F Y, g:i a')
    {
        if (!isset($this->data['updated'])) {
            if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'updated')) {
                $this->data['updated']['raw'] = $return[0]['data'];
            }

            if (!empty($this->data['updated']['raw'])) {
                $parser = $this->registry->call(Parse\Date::class, 'get');
                $this->data['updated']['parsed'] = $parser->parse($this->data['updated']['raw']) ?: null;
            } else {
                $this->data['updated'] = null;
            }
        }
        if ($this->data['updated']) {
            switch ($date_format) {
                case '':
                    return $this->sanitize($this->data['updated']['raw'], \SimplePie\SimplePie::CONSTRUCT_TEXT);

                case 'U':
                    return $this->data['updated']['parsed'];

                default:
                    return date($date_format, $this->data['updated']['parsed']);
            }
        }

        return null;
    }

    /**
     * Get the localized posting date/time for the item
     *
     * Returns the date formatted in the localized language. To display in
     * languages other than the server's default, you need to change the locale
     * with {@link http://php.net/setlocale setlocale()}. The available
     * localizations depend on which ones are installed on your web server.
     *
     * @since 1.0
     *
     * @param string $date_format Supports any PHP date format from {@see http://php.net/strftime} (empty for the raw data)
     * @return string|null|false see `strftime` for when this can return `false`
     */
    public function get_local_date(string $date_format = '%c')
    {
        if ($date_format === '') {
            if (($raw_date = $this->get_date('')) === null) {
                return null;
            }

            return $this->sanitize($raw_date, \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif (($date = $this->get_date('U')) !== null && $date !== false) {
            return strftime($date_format, $date);
        }

        return null;
    }

    /**
     * Get the posting date/time for the item (UTC time)
     *
     * @see get_date
     * @param string $date_format Supports any PHP date format from {@see http://php.net/date}
     * @return string|null
     */
    public function get_gmdate(string $date_format = 'j F Y, g:i a')
    {
        $date = $this->get_date('U');
        if ($date === null) {
            return null;
        }

        return gmdate($date_format, $date);
    }

    /**
     * Get the update date/time for the item (UTC time)
     *
     * @see get_updated_date
     * @param string $date_format Supports any PHP date format from {@see http://php.net/date}
     * @return string|null
     */
    public function get_updated_gmdate(string $date_format = 'j F Y, g:i a')
    {
        $date = $this->get_updated_date('U');
        if ($date === null) {
            return null;
        }

        return gmdate($date_format, $date);
    }

    /**
     * Get the permalink for the item
     *
     * Returns the first link available with a relationship of "alternate".
     * Identical to {@see get_link()} with key 0
     *
     * @see get_link
     * @since 0.8
     * @return string|null Permalink URL
     */
    public function get_permalink()
    {
        $link = $this->get_link();
        $enclosure = $this->get_enclosure(0);
        if ($link !== null) {
            return $link;
        } elseif ($enclosure !== null) {
            return $enclosure->get_link();
        }

        return null;
    }

    /**
     * Get a single link for the item
     *
     * @since Beta 3
     * @param int $key The link that you want to return.  Remember that arrays begin with 0, not 1
     * @param string $rel The relationship of the link to return
     * @return string|null Link URL
     */
    public function get_link(int $key = 0, string $rel = 'alternate')
    {
        $links = $this->get_links($rel);
        if ($links && $links[$key] !== null) {
            return $links[$key];
        }

        return null;
    }

    /**
     * Get all links for the item
     *
     * Uses `<atom:link>`, `<link>` or `<guid>`
     *
     * @since Beta 2
     * @param string $rel The relationship of links to return
     * @return array<string>|null Links found for the item (strings)
     */
    public function get_links(string $rel = 'alternate')
    {
        if (!isset($this->data['links'])) {
            $this->data['links'] = [];
            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'link') as $link) {
                if (isset($link['attribs']['']['href'])) {
                    $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                    $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($link));
                }
            }
            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'link') as $link) {
                if (isset($link['attribs']['']['href'])) {
                    $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                    $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($link));
                }
            }
            if ($links = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($links[0]));
            }
            if ($links = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($links[0]));
            }
            if ($links = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($links[0]));
            }
            if ($links = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'guid')) {
                if (!isset($links[0]['attribs']['']['isPermaLink']) || strtolower(trim($links[0]['attribs']['']['isPermaLink'])) === 'true') {
                    $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($links[0]));
                }
            }

            $keys = array_keys($this->data['links']);
            foreach ($keys as $key) {
                if ($this->registry->call(Misc::class, 'is_isegment_nz_nc', [$key])) {
                    if (isset($this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key])) {
                        $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key] = array_merge($this->data['links'][$key], $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key]);
                        $this->data['links'][$key] = &$this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key];
                    } else {
                        $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key] = &$this->data['links'][$key];
                    }
                } elseif (substr((string) $key, 0, 41) === \SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY) {
                    $this->data['links'][substr((string) $key, 41)] = &$this->data['links'][$key];
                }
                $this->data['links'][$key] = array_unique($this->data['links'][$key]);
            }
        }
        if (isset($this->data['links'][$rel])) {
            return $this->data['links'][$rel];
        }

        return null;
    }

    /**
     * Get an enclosure from the item
     *
     * Supports the <enclosure> RSS tag, as well as Media RSS and iTunes RSS.
     *
     * @since Beta 2
     * @todo Add ability to prefer one type of content over another (in a media group).
     * @param int $key The enclosure that you want to return.  Remember that arrays begin with 0, not 1
     * @return \SimplePie\Enclosure|null
     */
    public function get_enclosure(int $key = 0)
    {
        $enclosures = $this->get_enclosures();
        if (isset($enclosures[$key])) {
            return $enclosures[$key];
        }

        return null;
    }

    /**
     * Get all available enclosures (podcasts, etc.)
     *
     * Supports the <enclosure> RSS tag, as well as Media RSS and iTunes RSS.
     *
     * At this point, we're pretty much assuming that all enclosures for an item
     * are the same content.  Anything else is too complicated to
     * properly support.
     *
     * @since Beta 2
     * @todo Add support for end-user defined sorting of enclosures by type/handler (so we can prefer the faster-loading FLV over MP4).
     * @todo If an element exists at a level, but its value is empty, we should fall back to the value from the parent (if it exists).
     * @return \SimplePie\Enclosure[]|null List of \SimplePie\Enclosure items
     */
    public function get_enclosures()
    {
        if (!isset($this->data['enclosures'])) {
            $this->data['enclosures'] = [];

            // Elements
            $captions_parent = null;
            $categories_parent = null;
            $copyrights_parent = null;
            $credits_parent = null;
            $description_parent = null;
            $duration_parent = null;
            $hashes_parent = null;
            $keywords_parent = null;
            $player_parent = null;
            $ratings_parent = null;
            $restrictions_parent = [];
            $thumbnails_parent = null;
            $title_parent = null;

            // Let's do the channel and item-level ones first, and just re-use them if we need to.
            $parent = $this->get_feed();

            // CAPTIONS
            if ($captions = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'text')) {
                foreach ($captions as $caption) {
                    $caption_type = null;
                    $caption_lang = null;
                    $caption_startTime = null;
                    $caption_endTime = null;
                    $caption_text = null;
                    if (isset($caption['attribs']['']['type'])) {
                        $caption_type = $this->sanitize($caption['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['lang'])) {
                        $caption_lang = $this->sanitize($caption['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['start'])) {
                        $caption_startTime = $this->sanitize($caption['attribs']['']['start'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['end'])) {
                        $caption_endTime = $this->sanitize($caption['attribs']['']['end'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['data'])) {
                        $caption_text = $this->sanitize($caption['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $captions_parent[] = $this->registry->create(Caption::class, [$caption_type, $caption_lang, $caption_startTime, $caption_endTime, $caption_text]);
                }
            } elseif ($captions = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'text')) {
                foreach ($captions as $caption) {
                    $caption_type = null;
                    $caption_lang = null;
                    $caption_startTime = null;
                    $caption_endTime = null;
                    $caption_text = null;
                    if (isset($caption['attribs']['']['type'])) {
                        $caption_type = $this->sanitize($caption['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['lang'])) {
                        $caption_lang = $this->sanitize($caption['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['start'])) {
                        $caption_startTime = $this->sanitize($caption['attribs']['']['start'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['attribs']['']['end'])) {
                        $caption_endTime = $this->sanitize($caption['attribs']['']['end'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($caption['data'])) {
                        $caption_text = $this->sanitize($caption['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $captions_parent[] = $this->registry->create(Caption::class, [$caption_type, $caption_lang, $caption_startTime, $caption_endTime, $caption_text]);
                }
            }
            if (is_array($captions_parent)) {
                $captions_parent = array_values(array_unique($captions_parent));
            }

            // CATEGORIES
            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'category') as $category) {
                $term = null;
                $scheme = null;
                $label = null;
                if (isset($category['data'])) {
                    $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                if (isset($category['attribs']['']['scheme'])) {
                    $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                } else {
                    $scheme = 'http://search.yahoo.com/mrss/category_schema';
                }
                if (isset($category['attribs']['']['label'])) {
                    $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                $categories_parent[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
            }
            foreach ((array) $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'category') as $category) {
                $term = null;
                $scheme = null;
                $label = null;
                if (isset($category['data'])) {
                    $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                if (isset($category['attribs']['']['scheme'])) {
                    $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                } else {
                    $scheme = 'http://search.yahoo.com/mrss/category_schema';
                }
                if (isset($category['attribs']['']['label'])) {
                    $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                $categories_parent[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
            }
            foreach ((array) $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'category') as $category) {
                $term = null;
                $scheme = 'http://www.itunes.com/dtds/podcast-1.0.dtd';
                $label = null;
                if (isset($category['attribs']['']['text'])) {
                    $label = $this->sanitize($category['attribs']['']['text'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                $categories_parent[] = $this->registry->create(Category::class, [$term, $scheme, $label]);

                if (isset($category['child'][\SimplePie\SimplePie::NAMESPACE_ITUNES]['category'])) {
                    foreach ((array) $category['child'][\SimplePie\SimplePie::NAMESPACE_ITUNES]['category'] as $subcategory) {
                        if (isset($subcategory['attribs']['']['text'])) {
                            $label = $this->sanitize($subcategory['attribs']['']['text'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        $categories_parent[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
                    }
                }
            }
            if (is_array($categories_parent)) {
                $categories_parent = array_values(array_unique($categories_parent));
            }

            // COPYRIGHT
            if ($copyright = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'copyright')) {
                $copyright_url = null;
                $copyright_label = null;
                if (isset($copyright[0]['attribs']['']['url'])) {
                    $copyright_url = $this->sanitize($copyright[0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                if (isset($copyright[0]['data'])) {
                    $copyright_label = $this->sanitize($copyright[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                $copyrights_parent = $this->registry->create(Copyright::class, [$copyright_url, $copyright_label]);
            } elseif ($copyright = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'copyright')) {
                $copyright_url = null;
                $copyright_label = null;
                if (isset($copyright[0]['attribs']['']['url'])) {
                    $copyright_url = $this->sanitize($copyright[0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                if (isset($copyright[0]['data'])) {
                    $copyright_label = $this->sanitize($copyright[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
                $copyrights_parent = $this->registry->create(Copyright::class, [$copyright_url, $copyright_label]);
            }

            // CREDITS
            if ($credits = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'credit')) {
                foreach ($credits as $credit) {
                    $credit_role = null;
                    $credit_scheme = null;
                    $credit_name = null;
                    if (isset($credit['attribs']['']['role'])) {
                        $credit_role = $this->sanitize($credit['attribs']['']['role'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($credit['attribs']['']['scheme'])) {
                        $credit_scheme = $this->sanitize($credit['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $credit_scheme = 'urn:ebu';
                    }
                    if (isset($credit['data'])) {
                        $credit_name = $this->sanitize($credit['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $credits_parent[] = $this->registry->create(Credit::class, [$credit_role, $credit_scheme, $credit_name]);
                }
            } elseif ($credits = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'credit')) {
                foreach ($credits as $credit) {
                    $credit_role = null;
                    $credit_scheme = null;
                    $credit_name = null;
                    if (isset($credit['attribs']['']['role'])) {
                        $credit_role = $this->sanitize($credit['attribs']['']['role'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($credit['attribs']['']['scheme'])) {
                        $credit_scheme = $this->sanitize($credit['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $credit_scheme = 'urn:ebu';
                    }
                    if (isset($credit['data'])) {
                        $credit_name = $this->sanitize($credit['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $credits_parent[] = $this->registry->create(Credit::class, [$credit_role, $credit_scheme, $credit_name]);
                }
            }
            if (is_array($credits_parent)) {
                $credits_parent = array_values(array_unique($credits_parent));
            }

            // DESCRIPTION
            if ($description_parent = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'description')) {
                if (isset($description_parent[0]['data'])) {
                    $description_parent = $this->sanitize($description_parent[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
            } elseif ($description_parent = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'description')) {
                if (isset($description_parent[0]['data'])) {
                    $description_parent = $this->sanitize($description_parent[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
            }

            // DURATION
            $duration_tags = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'duration');
            if ($duration_tags !== null) {
                $seconds = null;
                $minutes = null;
                $hours = null;
                if (isset($duration_tags[0]['data'])) {
                    $temp = explode(':', $this->sanitize($duration_tags[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                    $seconds = (int) array_pop($temp);
                    if (count($temp) > 0) {
                        $minutes = (int) array_pop($temp);
                        $seconds += $minutes * 60;
                    }
                    if (count($temp) > 0) {
                        $hours = (int) array_pop($temp);
                        $seconds += $hours * 3600;
                    }
                    unset($temp);
                    $duration_parent = $seconds;
                }
            }

            // HASHES
            if ($hashes_iterator = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'hash')) {
                foreach ($hashes_iterator as $hash) {
                    $value = null;
                    $algo = null;
                    if (isset($hash['data'])) {
                        $value = $this->sanitize($hash['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($hash['attribs']['']['algo'])) {
                        $algo = $this->sanitize($hash['attribs']['']['algo'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $algo = 'md5';
                    }
                    $hashes_parent[] = $algo.':'.$value;
                }
            } elseif ($hashes_iterator = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'hash')) {
                foreach ($hashes_iterator as $hash) {
                    $value = null;
                    $algo = null;
                    if (isset($hash['data'])) {
                        $value = $this->sanitize($hash['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($hash['attribs']['']['algo'])) {
                        $algo = $this->sanitize($hash['attribs']['']['algo'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $algo = 'md5';
                    }
                    $hashes_parent[] = $algo.':'.$value;
                }
            }
            if (is_array($hashes_parent)) {
                $hashes_parent = array_values(array_unique($hashes_parent));
            }

            // KEYWORDS
            if ($keywords = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'keywords')) {
                if (isset($keywords[0]['data'])) {
                    $temp = explode(',', $this->sanitize($keywords[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                    foreach ($temp as $word) {
                        $keywords_parent[] = trim($word);
                    }
                }
                unset($temp);
            } elseif ($keywords = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'keywords')) {
                if (isset($keywords[0]['data'])) {
                    $temp = explode(',', $this->sanitize($keywords[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                    foreach ($temp as $word) {
                        $keywords_parent[] = trim($word);
                    }
                }
                unset($temp);
            } elseif ($keywords = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'keywords')) {
                if (isset($keywords[0]['data'])) {
                    $temp = explode(',', $this->sanitize($keywords[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                    foreach ($temp as $word) {
                        $keywords_parent[] = trim($word);
                    }
                }
                unset($temp);
            } elseif ($keywords = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'keywords')) {
                if (isset($keywords[0]['data'])) {
                    $temp = explode(',', $this->sanitize($keywords[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                    foreach ($temp as $word) {
                        $keywords_parent[] = trim($word);
                    }
                }
                unset($temp);
            }
            if (is_array($keywords_parent)) {
                $keywords_parent = array_values(array_unique($keywords_parent));
            }

            // PLAYER
            if ($player_parent = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'player')) {
                if (isset($player_parent[0]['attribs']['']['url'])) {
                    $player_parent = $this->sanitize($player_parent[0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($player_parent[0]));
                }
            } elseif ($player_parent = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'player')) {
                if (isset($player_parent[0]['attribs']['']['url'])) {
                    $player_parent = $this->sanitize($player_parent[0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($player_parent[0]));
                }
            }

            // RATINGS
            if ($ratings = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'rating')) {
                foreach ($ratings as $rating) {
                    $rating_scheme = null;
                    $rating_value = null;
                    if (isset($rating['attribs']['']['scheme'])) {
                        $rating_scheme = $this->sanitize($rating['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $rating_scheme = 'urn:simple';
                    }
                    if (isset($rating['data'])) {
                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $ratings_parent[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                }
            } elseif ($ratings = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'explicit')) {
                foreach ($ratings as $rating) {
                    $rating_scheme = 'urn:itunes';
                    $rating_value = null;
                    if (isset($rating['data'])) {
                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $ratings_parent[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                }
            } elseif ($ratings = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'rating')) {
                foreach ($ratings as $rating) {
                    $rating_scheme = null;
                    $rating_value = null;
                    if (isset($rating['attribs']['']['scheme'])) {
                        $rating_scheme = $this->sanitize($rating['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $rating_scheme = 'urn:simple';
                    }
                    if (isset($rating['data'])) {
                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $ratings_parent[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                }
            } elseif ($ratings = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'explicit')) {
                foreach ($ratings as $rating) {
                    $rating_scheme = 'urn:itunes';
                    $rating_value = null;
                    if (isset($rating['data'])) {
                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $ratings_parent[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                }
            }
            if (is_array($ratings_parent)) {
                $ratings_parent = array_values(array_unique($ratings_parent));
            }

            // RESTRICTIONS
            if ($restrictions = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'restriction')) {
                foreach ($restrictions as $restriction) {
                    $restriction_relationship = null;
                    $restriction_type = null;
                    $restriction_value = null;
                    if (isset($restriction['attribs']['']['relationship'])) {
                        $restriction_relationship = $this->sanitize($restriction['attribs']['']['relationship'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($restriction['attribs']['']['type'])) {
                        $restriction_type = $this->sanitize($restriction['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($restriction['data'])) {
                        $restriction_value = $this->sanitize($restriction['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $restrictions_parent[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                }
            } elseif ($restrictions = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'block')) {
                foreach ($restrictions as $restriction) {
                    $restriction_relationship = Restriction::RELATIONSHIP_ALLOW;
                    $restriction_type = null;
                    $restriction_value = 'itunes';
                    if (isset($restriction['data']) && strtolower($restriction['data']) === 'yes') {
                        $restriction_relationship = Restriction::RELATIONSHIP_DENY;
                    }
                    $restrictions_parent[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                }
            } elseif ($restrictions = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'restriction')) {
                foreach ($restrictions as $restriction) {
                    $restriction_relationship = null;
                    $restriction_type = null;
                    $restriction_value = null;
                    if (isset($restriction['attribs']['']['relationship'])) {
                        $restriction_relationship = $this->sanitize($restriction['attribs']['']['relationship'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($restriction['attribs']['']['type'])) {
                        $restriction_type = $this->sanitize($restriction['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($restriction['data'])) {
                        $restriction_value = $this->sanitize($restriction['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    $restrictions_parent[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                }
            } elseif ($restrictions = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'block')) {
                foreach ($restrictions as $restriction) {
                    $restriction_relationship = Restriction::RELATIONSHIP_ALLOW;
                    $restriction_type = null;
                    $restriction_value = 'itunes';
                    if (isset($restriction['data']) && strtolower($restriction['data']) === 'yes') {
                        $restriction_relationship = Restriction::RELATIONSHIP_DENY;
                    }
                    $restrictions_parent[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                }
            }
            if (count($restrictions_parent) > 0) {
                $restrictions_parent = array_values(array_unique($restrictions_parent));
            } else {
                $restrictions_parent = [new \SimplePie\Restriction(Restriction::RELATIONSHIP_ALLOW, null, 'default')];
            }

            // THUMBNAILS
            if ($thumbnails = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'thumbnail')) {
                foreach ($thumbnails as $thumbnail) {
                    if (isset($thumbnail['attribs']['']['url'])) {
                        $thumbnails_parent[] = $this->sanitize($thumbnail['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($thumbnail));
                    }
                }
            } elseif ($thumbnails = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'thumbnail')) {
                foreach ($thumbnails as $thumbnail) {
                    if (isset($thumbnail['attribs']['']['url'])) {
                        $thumbnails_parent[] = $this->sanitize($thumbnail['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($thumbnail));
                    }
                }
            }

            // TITLES
            if ($title_parent = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'title')) {
                if (isset($title_parent[0]['data'])) {
                    $title_parent = $this->sanitize($title_parent[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
            } elseif ($title_parent = $parent->get_channel_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'title')) {
                if (isset($title_parent[0]['data'])) {
                    $title_parent = $this->sanitize($title_parent[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                }
            }

            // Clear the memory
            unset($parent);

            // Attributes
            $bitrate = null;
            $channels = null;
            $duration = null;
            $expression = null;
            $framerate = null;
            $height = null;
            $javascript = null;
            $lang = null;
            $length = null;
            $medium = null;
            $samplingrate = null;
            $type = null;
            $url = null;
            $width = null;

            // Elements
            $captions = null;
            $categories = null;
            $copyrights = null;
            $credits = null;
            $description = null;
            $hashes = null;
            $keywords = null;
            $player = null;
            $ratings = null;
            $restrictions = null;
            $thumbnails = null;
            $title = null;

            // If we have media:group tags, loop through them.
            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_MEDIARSS, 'group') as $group) {
                if (isset($group['child']) && isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['content'])) {
                    // If we have media:content tags, loop through them.
                    foreach ((array) $group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['content'] as $content) {
                        if (isset($content['attribs']['']['url'])) {
                            // Attributes
                            $bitrate = null;
                            $channels = null;
                            $duration = null;
                            $expression = null;
                            $framerate = null;
                            $height = null;
                            $javascript = null;
                            $lang = null;
                            $length = null;
                            $medium = null;
                            $samplingrate = null;
                            $type = null;
                            $url = null;
                            $width = null;

                            // Elements
                            $captions = null;
                            $categories = null;
                            $copyrights = null;
                            $credits = null;
                            $description = null;
                            $hashes = null;
                            $keywords = null;
                            $player = null;
                            $ratings = null;
                            $restrictions = null;
                            $thumbnails = null;
                            $title = null;

                            // Start checking the attributes of media:content
                            if (isset($content['attribs']['']['bitrate'])) {
                                $bitrate = $this->sanitize($content['attribs']['']['bitrate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['channels'])) {
                                $channels = $this->sanitize($content['attribs']['']['channels'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['duration'])) {
                                $duration = $this->sanitize($content['attribs']['']['duration'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            } else {
                                $duration = $duration_parent;
                            }
                            if (isset($content['attribs']['']['expression'])) {
                                $expression = $this->sanitize($content['attribs']['']['expression'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['framerate'])) {
                                $framerate = $this->sanitize($content['attribs']['']['framerate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['height'])) {
                                $height = $this->sanitize($content['attribs']['']['height'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['lang'])) {
                                $lang = $this->sanitize($content['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['fileSize'])) {
                                $length = intval($content['attribs']['']['fileSize']);
                            }
                            if (isset($content['attribs']['']['medium'])) {
                                $medium = $this->sanitize($content['attribs']['']['medium'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['samplingrate'])) {
                                $samplingrate = $this->sanitize($content['attribs']['']['samplingrate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['type'])) {
                                $type = $this->sanitize($content['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['attribs']['']['width'])) {
                                $width = $this->sanitize($content['attribs']['']['width'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            $url = $this->sanitize($content['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($content));

                            // Checking the other optional media: elements. Priority: media:content, media:group, item, channel

                            // CAPTIONS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'] as $caption) {
                                    $caption_type = null;
                                    $caption_lang = null;
                                    $caption_startTime = null;
                                    $caption_endTime = null;
                                    $caption_text = null;
                                    if (isset($caption['attribs']['']['type'])) {
                                        $caption_type = $this->sanitize($caption['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['lang'])) {
                                        $caption_lang = $this->sanitize($caption['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['start'])) {
                                        $caption_startTime = $this->sanitize($caption['attribs']['']['start'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['end'])) {
                                        $caption_endTime = $this->sanitize($caption['attribs']['']['end'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['data'])) {
                                        $caption_text = $this->sanitize($caption['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $captions[] = $this->registry->create(Caption::class, [$caption_type, $caption_lang, $caption_startTime, $caption_endTime, $caption_text]);
                                }
                                if (is_array($captions)) {
                                    $captions = array_values(array_unique($captions));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'] as $caption) {
                                    $caption_type = null;
                                    $caption_lang = null;
                                    $caption_startTime = null;
                                    $caption_endTime = null;
                                    $caption_text = null;
                                    if (isset($caption['attribs']['']['type'])) {
                                        $caption_type = $this->sanitize($caption['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['lang'])) {
                                        $caption_lang = $this->sanitize($caption['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['start'])) {
                                        $caption_startTime = $this->sanitize($caption['attribs']['']['start'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['attribs']['']['end'])) {
                                        $caption_endTime = $this->sanitize($caption['attribs']['']['end'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($caption['data'])) {
                                        $caption_text = $this->sanitize($caption['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $captions[] = $this->registry->create(Caption::class, [$caption_type, $caption_lang, $caption_startTime, $caption_endTime, $caption_text]);
                                }
                                if (is_array($captions)) {
                                    $captions = array_values(array_unique($captions));
                                }
                            } else {
                                $captions = $captions_parent;
                            }

                            // CATEGORIES
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'])) {
                                foreach ((array) $content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'] as $category) {
                                    $term = null;
                                    $scheme = null;
                                    $label = null;
                                    if (isset($category['data'])) {
                                        $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($category['attribs']['']['scheme'])) {
                                        $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $scheme = 'http://search.yahoo.com/mrss/category_schema';
                                    }
                                    if (isset($category['attribs']['']['label'])) {
                                        $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
                                }
                            }
                            if (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'])) {
                                foreach ((array) $group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'] as $category) {
                                    $term = null;
                                    $scheme = null;
                                    $label = null;
                                    if (isset($category['data'])) {
                                        $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($category['attribs']['']['scheme'])) {
                                        $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $scheme = 'http://search.yahoo.com/mrss/category_schema';
                                    }
                                    if (isset($category['attribs']['']['label'])) {
                                        $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
                                }
                            }
                            if (is_array($categories) && is_array($categories_parent)) {
                                $categories = array_values(array_unique(array_merge($categories, $categories_parent)));
                            } elseif (is_array($categories)) {
                                $categories = array_values(array_unique($categories));
                            } elseif (is_array($categories_parent)) {
                                $categories = array_values(array_unique($categories_parent));
                            }

                            // COPYRIGHTS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'])) {
                                $copyright_url = null;
                                $copyright_label = null;
                                if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'])) {
                                    $copyright_url = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'])) {
                                    $copyright_label = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $copyrights = $this->registry->create(Copyright::class, [$copyright_url, $copyright_label]);
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'])) {
                                $copyright_url = null;
                                $copyright_label = null;
                                if (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'])) {
                                    $copyright_url = $this->sanitize($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'])) {
                                    $copyright_label = $this->sanitize($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $copyrights = $this->registry->create(Copyright::class, [$copyright_url, $copyright_label]);
                            } else {
                                $copyrights = $copyrights_parent;
                            }

                            // CREDITS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'] as $credit) {
                                    $credit_role = null;
                                    $credit_scheme = null;
                                    $credit_name = null;
                                    if (isset($credit['attribs']['']['role'])) {
                                        $credit_role = $this->sanitize($credit['attribs']['']['role'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($credit['attribs']['']['scheme'])) {
                                        $credit_scheme = $this->sanitize($credit['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $credit_scheme = 'urn:ebu';
                                    }
                                    if (isset($credit['data'])) {
                                        $credit_name = $this->sanitize($credit['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $credits[] = $this->registry->create(Credit::class, [$credit_role, $credit_scheme, $credit_name]);
                                }
                                if (is_array($credits)) {
                                    $credits = array_values(array_unique($credits));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'] as $credit) {
                                    $credit_role = null;
                                    $credit_scheme = null;
                                    $credit_name = null;
                                    if (isset($credit['attribs']['']['role'])) {
                                        $credit_role = $this->sanitize($credit['attribs']['']['role'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($credit['attribs']['']['scheme'])) {
                                        $credit_scheme = $this->sanitize($credit['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $credit_scheme = 'urn:ebu';
                                    }
                                    if (isset($credit['data'])) {
                                        $credit_name = $this->sanitize($credit['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $credits[] = $this->registry->create(Credit::class, [$credit_role, $credit_scheme, $credit_name]);
                                }
                                if (is_array($credits)) {
                                    $credits = array_values(array_unique($credits));
                                }
                            } else {
                                $credits = $credits_parent;
                            }

                            // DESCRIPTION
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'])) {
                                $description = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'])) {
                                $description = $this->sanitize($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            } else {
                                $description = $description_parent;
                            }

                            // HASHES
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'] as $hash) {
                                    $value = null;
                                    $algo = null;
                                    if (isset($hash['data'])) {
                                        $value = $this->sanitize($hash['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($hash['attribs']['']['algo'])) {
                                        $algo = $this->sanitize($hash['attribs']['']['algo'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $algo = 'md5';
                                    }
                                    $hashes[] = $algo.':'.$value;
                                }
                                if (is_array($hashes)) {
                                    $hashes = array_values(array_unique($hashes));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'] as $hash) {
                                    $value = null;
                                    $algo = null;
                                    if (isset($hash['data'])) {
                                        $value = $this->sanitize($hash['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($hash['attribs']['']['algo'])) {
                                        $algo = $this->sanitize($hash['attribs']['']['algo'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $algo = 'md5';
                                    }
                                    $hashes[] = $algo.':'.$value;
                                }
                                if (is_array($hashes)) {
                                    $hashes = array_values(array_unique($hashes));
                                }
                            } else {
                                $hashes = $hashes_parent;
                            }

                            // KEYWORDS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'])) {
                                if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'])) {
                                    $temp = explode(',', $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                                    foreach ($temp as $word) {
                                        $keywords[] = trim($word);
                                    }
                                    unset($temp);
                                }
                                if (is_array($keywords)) {
                                    $keywords = array_values(array_unique($keywords));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'])) {
                                if (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'])) {
                                    $temp = explode(',', $this->sanitize($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                                    foreach ($temp as $word) {
                                        $keywords[] = trim($word);
                                    }
                                    unset($temp);
                                }
                                if (is_array($keywords)) {
                                    $keywords = array_values(array_unique($keywords));
                                }
                            } else {
                                $keywords = $keywords_parent;
                            }

                            // PLAYER
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'])) {
                                $playerElem = $content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'][0];
                                $player = $this->sanitize($playerElem['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($playerElem));
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'])) {
                                $playerElem = $group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'][0];
                                $player = $this->sanitize($playerElem['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($playerElem));
                            } else {
                                $player = $player_parent;
                            }

                            // RATINGS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'] as $rating) {
                                    $rating_scheme = null;
                                    $rating_value = null;
                                    if (isset($rating['attribs']['']['scheme'])) {
                                        $rating_scheme = $this->sanitize($rating['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $rating_scheme = 'urn:simple';
                                    }
                                    if (isset($rating['data'])) {
                                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $ratings[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                                }
                                if (is_array($ratings)) {
                                    $ratings = array_values(array_unique($ratings));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'] as $rating) {
                                    $rating_scheme = null;
                                    $rating_value = null;
                                    if (isset($rating['attribs']['']['scheme'])) {
                                        $rating_scheme = $this->sanitize($rating['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    } else {
                                        $rating_scheme = 'urn:simple';
                                    }
                                    if (isset($rating['data'])) {
                                        $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $ratings[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                                }
                                if (is_array($ratings)) {
                                    $ratings = array_values(array_unique($ratings));
                                }
                            } else {
                                $ratings = $ratings_parent;
                            }

                            // RESTRICTIONS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'] as $restriction) {
                                    $restriction_relationship = null;
                                    $restriction_type = null;
                                    $restriction_value = null;
                                    if (isset($restriction['attribs']['']['relationship'])) {
                                        $restriction_relationship = $this->sanitize($restriction['attribs']['']['relationship'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($restriction['attribs']['']['type'])) {
                                        $restriction_type = $this->sanitize($restriction['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($restriction['data'])) {
                                        $restriction_value = $this->sanitize($restriction['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $restrictions[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                                }
                                if (is_array($restrictions)) {
                                    $restrictions = array_values(array_unique($restrictions));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'] as $restriction) {
                                    $restriction_relationship = null;
                                    $restriction_type = null;
                                    $restriction_value = null;
                                    if (isset($restriction['attribs']['']['relationship'])) {
                                        $restriction_relationship = $this->sanitize($restriction['attribs']['']['relationship'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($restriction['attribs']['']['type'])) {
                                        $restriction_type = $this->sanitize($restriction['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    if (isset($restriction['data'])) {
                                        $restriction_value = $this->sanitize($restriction['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                    }
                                    $restrictions[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                                }
                                if (is_array($restrictions)) {
                                    $restrictions = array_values(array_unique($restrictions));
                                }
                            } else {
                                $restrictions = $restrictions_parent;
                            }

                            // THUMBNAILS
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'])) {
                                foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'] as $thumbnail) {
                                    $thumbnails[] = $this->sanitize($thumbnail['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($thumbnail));
                                }
                                if (is_array($thumbnails)) {
                                    $thumbnails = array_values(array_unique($thumbnails));
                                }
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'])) {
                                foreach ($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'] as $thumbnail) {
                                    $thumbnails[] = $this->sanitize($thumbnail['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($thumbnail));
                                }
                                if (is_array($thumbnails)) {
                                    $thumbnails = array_values(array_unique($thumbnails));
                                }
                            } else {
                                $thumbnails = $thumbnails_parent;
                            }

                            // TITLES
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'])) {
                                $title = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            } elseif (isset($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'])) {
                                $title = $this->sanitize($group['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            } else {
                                $title = $title_parent;
                            }

                            $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions, $categories, $channels, $copyrights, $credits, $description, $duration, $expression, $framerate, $hashes, $height, $keywords, $lang, $medium, $player, $ratings, $restrictions, $samplingrate, $thumbnails, $title, $width]);
                        }
                    }
                }
            }

            // If we have standalone media:content tags, loop through them.
            if (isset($this->data['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['content'])) {
                foreach ((array) $this->data['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['content'] as $content) {
                    if (isset($content['attribs']['']['url']) || isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'])) {
                        // Attributes
                        $bitrate = null;
                        $channels = null;
                        $duration = null;
                        $expression = null;
                        $framerate = null;
                        $height = null;
                        $javascript = null;
                        $lang = null;
                        $length = null;
                        $medium = null;
                        $samplingrate = null;
                        $type = null;
                        $url = null;
                        $width = null;

                        // Elements
                        $captions = null;
                        $categories = null;
                        $copyrights = null;
                        $credits = null;
                        $description = null;
                        $hashes = null;
                        $keywords = null;
                        $player = null;
                        $ratings = null;
                        $restrictions = null;
                        $thumbnails = null;
                        $title = null;

                        // Start checking the attributes of media:content
                        if (isset($content['attribs']['']['bitrate'])) {
                            $bitrate = $this->sanitize($content['attribs']['']['bitrate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['channels'])) {
                            $channels = $this->sanitize($content['attribs']['']['channels'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['duration'])) {
                            $duration = $this->sanitize($content['attribs']['']['duration'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        } else {
                            $duration = $duration_parent;
                        }
                        if (isset($content['attribs']['']['expression'])) {
                            $expression = $this->sanitize($content['attribs']['']['expression'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['framerate'])) {
                            $framerate = $this->sanitize($content['attribs']['']['framerate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['height'])) {
                            $height = $this->sanitize($content['attribs']['']['height'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['lang'])) {
                            $lang = $this->sanitize($content['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['fileSize'])) {
                            $length = intval($content['attribs']['']['fileSize']);
                        }
                        if (isset($content['attribs']['']['medium'])) {
                            $medium = $this->sanitize($content['attribs']['']['medium'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['samplingrate'])) {
                            $samplingrate = $this->sanitize($content['attribs']['']['samplingrate'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['type'])) {
                            $type = $this->sanitize($content['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['width'])) {
                            $width = $this->sanitize($content['attribs']['']['width'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        }
                        if (isset($content['attribs']['']['url'])) {
                            $url = $this->sanitize($content['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($content));
                        }
                        // Checking the other optional media: elements. Priority: media:content, media:group, item, channel

                        // CAPTIONS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['text'] as $caption) {
                                $caption_type = null;
                                $caption_lang = null;
                                $caption_startTime = null;
                                $caption_endTime = null;
                                $caption_text = null;
                                if (isset($caption['attribs']['']['type'])) {
                                    $caption_type = $this->sanitize($caption['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($caption['attribs']['']['lang'])) {
                                    $caption_lang = $this->sanitize($caption['attribs']['']['lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($caption['attribs']['']['start'])) {
                                    $caption_startTime = $this->sanitize($caption['attribs']['']['start'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($caption['attribs']['']['end'])) {
                                    $caption_endTime = $this->sanitize($caption['attribs']['']['end'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($caption['data'])) {
                                    $caption_text = $this->sanitize($caption['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $captions[] = $this->registry->create(Caption::class, [$caption_type, $caption_lang, $caption_startTime, $caption_endTime, $caption_text]);
                            }
                            if (is_array($captions)) {
                                $captions = array_values(array_unique($captions));
                            }
                        } else {
                            $captions = $captions_parent;
                        }

                        // CATEGORIES
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'])) {
                            foreach ((array) $content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['category'] as $category) {
                                $term = null;
                                $scheme = null;
                                $label = null;
                                if (isset($category['data'])) {
                                    $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($category['attribs']['']['scheme'])) {
                                    $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                } else {
                                    $scheme = 'http://search.yahoo.com/mrss/category_schema';
                                }
                                if (isset($category['attribs']['']['label'])) {
                                    $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
                            }
                        }
                        if (is_array($categories) && is_array($categories_parent)) {
                            $categories = array_values(array_unique(array_merge($categories, $categories_parent)));
                        } elseif (is_array($categories)) {
                            $categories = array_values(array_unique($categories));
                        } elseif (is_array($categories_parent)) {
                            $categories = array_values(array_unique($categories_parent));
                        } else {
                            $categories = null;
                        }

                        // COPYRIGHTS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'])) {
                            $copyright_url = null;
                            $copyright_label = null;
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'])) {
                                $copyright_url = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'])) {
                                $copyright_label = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['copyright'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                            }
                            $copyrights = $this->registry->create(Copyright::class, [$copyright_url, $copyright_label]);
                        } else {
                            $copyrights = $copyrights_parent;
                        }

                        // CREDITS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['credit'] as $credit) {
                                $credit_role = null;
                                $credit_scheme = null;
                                $credit_name = null;
                                if (isset($credit['attribs']['']['role'])) {
                                    $credit_role = $this->sanitize($credit['attribs']['']['role'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($credit['attribs']['']['scheme'])) {
                                    $credit_scheme = $this->sanitize($credit['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                } else {
                                    $credit_scheme = 'urn:ebu';
                                }
                                if (isset($credit['data'])) {
                                    $credit_name = $this->sanitize($credit['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $credits[] = $this->registry->create(Credit::class, [$credit_role, $credit_scheme, $credit_name]);
                            }
                            if (is_array($credits)) {
                                $credits = array_values(array_unique($credits));
                            }
                        } else {
                            $credits = $credits_parent;
                        }

                        // DESCRIPTION
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'])) {
                            $description = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['description'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        } else {
                            $description = $description_parent;
                        }

                        // HASHES
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['hash'] as $hash) {
                                $value = null;
                                $algo = null;
                                if (isset($hash['data'])) {
                                    $value = $this->sanitize($hash['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($hash['attribs']['']['algo'])) {
                                    $algo = $this->sanitize($hash['attribs']['']['algo'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                } else {
                                    $algo = 'md5';
                                }
                                $hashes[] = $algo.':'.$value;
                            }
                            if (is_array($hashes)) {
                                $hashes = array_values(array_unique($hashes));
                            }
                        } else {
                            $hashes = $hashes_parent;
                        }

                        // KEYWORDS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'])) {
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'])) {
                                $temp = explode(',', $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['keywords'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT));
                                foreach ($temp as $word) {
                                    $keywords[] = trim($word);
                                }
                                unset($temp);
                            }
                            if (is_array($keywords)) {
                                $keywords = array_values(array_unique($keywords));
                            }
                        } else {
                            $keywords = $keywords_parent;
                        }

                        // PLAYER
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'])) {
                            if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'][0]['attribs']['']['url'])) {
                                $playerElem = $content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['player'][0];
                                $player = $this->sanitize($playerElem['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($playerElem));
                            }
                        } else {
                            $player = $player_parent;
                        }

                        // RATINGS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['rating'] as $rating) {
                                $rating_scheme = null;
                                $rating_value = null;
                                if (isset($rating['attribs']['']['scheme'])) {
                                    $rating_scheme = $this->sanitize($rating['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                } else {
                                    $rating_scheme = 'urn:simple';
                                }
                                if (isset($rating['data'])) {
                                    $rating_value = $this->sanitize($rating['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $ratings[] = $this->registry->create(Rating::class, [$rating_scheme, $rating_value]);
                            }
                            if (is_array($ratings)) {
                                $ratings = array_values(array_unique($ratings));
                            }
                        } else {
                            $ratings = $ratings_parent;
                        }

                        // RESTRICTIONS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['restriction'] as $restriction) {
                                $restriction_relationship = null;
                                $restriction_type = null;
                                $restriction_value = null;
                                if (isset($restriction['attribs']['']['relationship'])) {
                                    $restriction_relationship = $this->sanitize($restriction['attribs']['']['relationship'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($restriction['attribs']['']['type'])) {
                                    $restriction_type = $this->sanitize($restriction['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                if (isset($restriction['data'])) {
                                    $restriction_value = $this->sanitize($restriction['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                                }
                                $restrictions[] = $this->registry->create(Restriction::class, [$restriction_relationship, $restriction_type, $restriction_value]);
                            }
                            if (is_array($restrictions)) {
                                $restrictions = array_values(array_unique($restrictions));
                            }
                        } else {
                            $restrictions = $restrictions_parent;
                        }

                        // THUMBNAILS
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'])) {
                            foreach ($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['thumbnail'] as $thumbnail) {
                                if (isset($thumbnail['attribs']['']['url'])) {
                                    $thumbnails[] = $this->sanitize($thumbnail['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($thumbnail));
                                }
                            }
                            if (is_array($thumbnails)) {
                                $thumbnails = array_values(array_unique($thumbnails));
                            }
                        } else {
                            $thumbnails = $thumbnails_parent;
                        }

                        // TITLES
                        if (isset($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'])) {
                            $title = $this->sanitize($content['child'][\SimplePie\SimplePie::NAMESPACE_MEDIARSS]['title'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                        } else {
                            $title = $title_parent;
                        }

                        $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions, $categories, $channels, $copyrights, $credits, $description, $duration, $expression, $framerate, $hashes, $height, $keywords, $lang, $medium, $player, $ratings, $restrictions, $samplingrate, $thumbnails, $title, $width]);
                    }
                }
            }

            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'link') as $link) {
                if (isset($link['attribs']['']['href']) && !empty($link['attribs']['']['rel']) && $link['attribs']['']['rel'] === 'enclosure') {
                    // Attributes
                    $bitrate = null;
                    $channels = null;
                    $duration = null;
                    $expression = null;
                    $framerate = null;
                    $height = null;
                    $javascript = null;
                    $lang = null;
                    $length = null;
                    $medium = null;
                    $samplingrate = null;
                    $type = null;
                    $url = null;
                    $width = null;

                    $url = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($link));
                    if (isset($link['attribs']['']['type'])) {
                        $type = $this->sanitize($link['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($link['attribs']['']['length'])) {
                        $length = intval($link['attribs']['']['length']);
                    }
                    if (isset($link['attribs']['']['title'])) {
                        $title = $this->sanitize($link['attribs']['']['title'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    } else {
                        $title = $title_parent;
                    }

                    // Since we don't have group or content for these, we'll just pass the '*_parent' variables directly to the constructor
                    $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions_parent, $categories_parent, $channels, $copyrights_parent, $credits_parent, $description_parent, $duration_parent, $expression, $framerate, $hashes_parent, $height, $keywords_parent, $lang, $medium, $player_parent, $ratings_parent, $restrictions_parent, $samplingrate, $thumbnails_parent, $title, $width]);
                }
            }

            foreach ((array) $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'link') as $link) {
                if (isset($link['attribs']['']['href']) && !empty($link['attribs']['']['rel']) && $link['attribs']['']['rel'] === 'enclosure') {
                    // Attributes
                    $bitrate = null;
                    $channels = null;
                    $duration = null;
                    $expression = null;
                    $framerate = null;
                    $height = null;
                    $javascript = null;
                    $lang = null;
                    $length = null;
                    $medium = null;
                    $samplingrate = null;
                    $type = null;
                    $url = null;
                    $width = null;

                    $url = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($link));
                    if (isset($link['attribs']['']['type'])) {
                        $type = $this->sanitize($link['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($link['attribs']['']['length'])) {
                        $length = intval($link['attribs']['']['length']);
                    }

                    // Since we don't have group or content for these, we'll just pass the '*_parent' variables directly to the constructor
                    $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions_parent, $categories_parent, $channels, $copyrights_parent, $credits_parent, $description_parent, $duration_parent, $expression, $framerate, $hashes_parent, $height, $keywords_parent, $lang, $medium, $player_parent, $ratings_parent, $restrictions_parent, $samplingrate, $thumbnails_parent, $title_parent, $width]);
                }
            }

            foreach ($this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'enclosure') ?? [] as $enclosure) {
                if (isset($enclosure['attribs']['']['url'])) {
                    // Attributes
                    $bitrate = null;
                    $channels = null;
                    $duration = null;
                    $expression = null;
                    $framerate = null;
                    $height = null;
                    $javascript = null;
                    $lang = null;
                    $length = null;
                    $medium = null;
                    $samplingrate = null;
                    $type = null;
                    $url = null;
                    $width = null;

                    $url = $this->sanitize($enclosure['attribs']['']['url'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_own_base($enclosure));
                    $url = $this->get_sanitize()->https_url($url);
                    if (isset($enclosure['attribs']['']['type'])) {
                        $type = $this->sanitize($enclosure['attribs']['']['type'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
                    }
                    if (isset($enclosure['attribs']['']['length'])) {
                        $length = intval($enclosure['attribs']['']['length']);
                    }

                    // Since we don't have group or content for these, we'll just pass the '*_parent' variables directly to the constructor
                    $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions_parent, $categories_parent, $channels, $copyrights_parent, $credits_parent, $description_parent, $duration_parent, $expression, $framerate, $hashes_parent, $height, $keywords_parent, $lang, $medium, $player_parent, $ratings_parent, $restrictions_parent, $samplingrate, $thumbnails_parent, $title_parent, $width]);
                }
            }

            if (count($this->data['enclosures']) === 0 && ($url || $type || $length || $bitrate || $captions_parent || $categories_parent || $channels || $copyrights_parent || $credits_parent || $description_parent || $duration_parent || $expression || $framerate || $hashes_parent || $height || $keywords_parent || $lang || $medium || $player_parent || $ratings_parent || $samplingrate || $thumbnails_parent || $title_parent || $width)) {
                // Since we don't have group or content for these, we'll just pass the '*_parent' variables directly to the constructor
                $this->data['enclosures'][] = $this->registry->create(Enclosure::class, [$url, $type, $length, null, $bitrate, $captions_parent, $categories_parent, $channels, $copyrights_parent, $credits_parent, $description_parent, $duration_parent, $expression, $framerate, $hashes_parent, $height, $keywords_parent, $lang, $medium, $player_parent, $ratings_parent, $restrictions_parent, $samplingrate, $thumbnails_parent, $title_parent, $width]);
            }

            $this->data['enclosures'] = array_values(array_unique($this->data['enclosures']));
        }
        if (!empty($this->data['enclosures'])) {
            return $this->data['enclosures'];
        }

        return null;
    }

    /**
     * Get the latitude coordinates for the item
     *
     * Compatible with the W3C WGS84 Basic Geo and GeoRSS specifications
     *
     * Uses `<geo:lat>` or `<georss:point>`
     *
     * @since 1.0
     * @link http://www.w3.org/2003/01/geo/ W3C WGS84 Basic Geo
     * @link http://www.georss.org/ GeoRSS
     * @return float|null
     */
    public function get_latitude()
    {
        if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'lat')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[1];
        }

        return null;
    }

    /**
     * Get the longitude coordinates for the item
     *
     * Compatible with the W3C WGS84 Basic Geo and GeoRSS specifications
     *
     * Uses `<geo:long>`, `<geo:lon>` or `<georss:point>`
     *
     * @since 1.0
     * @link http://www.w3.org/2003/01/geo/ W3C WGS84 Basic Geo
     * @link http://www.georss.org/ GeoRSS
     * @return float|null
     */
    public function get_longitude()
    {
        if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'long')) {
            return (float) $return[0]['data'];
        } elseif ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'lon')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[2];
        }

        return null;
    }

    /**
     * Get the `<atom:source>` for the item
     *
     * @since 1.1
     * @return \SimplePie\Source|null
     */
    public function get_source()
    {
        if ($return = $this->get_item_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'source')) {
            return $this->registry->create(Source::class, [$this, $return[0]]);
        }

        return null;
    }

    public function set_sanitize(Sanitize $sanitize): void
    {
        $this->sanitize = $sanitize;
    }

    protected function get_sanitize(): Sanitize
    {
        if ($this->sanitize === null) {
            $this->sanitize = new Sanitize();
        }

        return $this->sanitize;
    }
}

class_alias('SimplePie\Item', 'SimplePie_Item');

}

namespace SimplePie {

use DomDocument;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\UriFactoryInterface;
use SimplePie\HTTP\Client;
use SimplePie\HTTP\ClientException;
use SimplePie\HTTP\FileClient;
use SimplePie\HTTP\Psr18Client;
use SimplePie\HTTP\Response;

/**
 * Used for feed auto-discovery
 *
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_locator_class()}
 */
class Locator implements RegistryAware
{
    /** @var ?string */
    public $useragent = null;
    /** @var int */
    public $timeout = 10;
    /** @var File */
    public $file;
    /** @var string[] */
    public $local = [];
    /** @var string[] */
    public $elsewhere = [];
    /** @var array<mixed> */
    public $cached_entities = [];
    /** @var string */
    public $http_base;
    /** @var string */
    public $base;
    /** @var int */
    public $base_location = 0;
    /** @var int */
    public $checked_feeds = 0;
    /** @var int */
    public $max_checked_feeds = 10;
    /** @var bool */
    public $force_fsockopen = false;
    /** @var array<int, mixed> */
    public $curl_options = [];
    /** @var ?\DomDocument */
    public $dom;
    /** @var ?Registry */
    protected $registry;

    /**
     * @var Client|null
     */
    private $http_client = null;

    /**
     * @param array<int, mixed> $curl_options
     */
    public function __construct(File $file, int $timeout = 10, ?string $useragent = null, int $max_checked_feeds = 10, bool $force_fsockopen = false, array $curl_options = [])
    {
        $this->file = $file;
        $this->useragent = $useragent;
        $this->timeout = $timeout;
        $this->max_checked_feeds = $max_checked_feeds;
        $this->force_fsockopen = $force_fsockopen;
        $this->curl_options = $curl_options;

        $body = $this->file->get_body_content();

        if (class_exists('DOMDocument') && $body != '') {
            $this->dom = new \DOMDocument();

            set_error_handler([Misc::class, 'silence_errors']);
            try {
                $this->dom->loadHTML($body);
            } catch (\Throwable $ex) {
                $this->dom = null;
            }
            restore_error_handler();
        } else {
            $this->dom = null;
        }
    }

    /**
     * Set a PSR-18 client and PSR-17 factories
     *
     * Allows you to use your own HTTP client implementations.
     */
    final public function set_http_client(
        ClientInterface $http_client,
        RequestFactoryInterface $request_factory,
        UriFactoryInterface $uri_factory
    ): void {
        $this->http_client = new Psr18Client($http_client, $request_factory, $uri_factory);
    }

    /**
     * @return void
     */
    public function set_registry(\SimplePie\Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * @param SimplePie::LOCATOR_* $type
     * @param array<Response>|null $working
     * @return Response|null
     */
    public function find(int $type = \SimplePie\SimplePie::LOCATOR_ALL, ?array &$working = null)
    {
        assert($this->registry !== null);

        if ($this->is_feed($this->file)) {
            return $this->file;
        }

        if (Misc::is_remote_uri($this->file->get_final_requested_uri())) {
            $sniffer = $this->registry->create(Content\Type\Sniffer::class, [$this->file]);
            if ($sniffer->get_type() !== 'text/html') {
                return null;
            }
        }

        if ($type & ~\SimplePie\SimplePie::LOCATOR_NONE) {
            $this->get_base();
        }

        if ($type & \SimplePie\SimplePie::LOCATOR_AUTODISCOVERY && $working = $this->autodiscovery()) {
            return $working[0];
        }

        if ($type & (\SimplePie\SimplePie::LOCATOR_LOCAL_EXTENSION | \SimplePie\SimplePie::LOCATOR_LOCAL_BODY | \SimplePie\SimplePie::LOCATOR_REMOTE_EXTENSION | \SimplePie\SimplePie::LOCATOR_REMOTE_BODY) && $this->get_links()) {
            if ($type & \SimplePie\SimplePie::LOCATOR_LOCAL_EXTENSION && $working = $this->extension($this->local)) {
                return $working[0];
            }

            if ($type & \SimplePie\SimplePie::LOCATOR_LOCAL_BODY && $working = $this->body($this->local)) {
                return $working[0];
            }

            if ($type & \SimplePie\SimplePie::LOCATOR_REMOTE_EXTENSION && $working = $this->extension($this->elsewhere)) {
                return $working[0];
            }

            if ($type & \SimplePie\SimplePie::LOCATOR_REMOTE_BODY && $working = $this->body($this->elsewhere)) {
                return $working[0];
            }
        }
        return null;
    }

    /**
     * @return bool
     */
    public function is_feed(Response $file, bool $check_html = false)
    {
        assert($this->registry !== null);

        if (Misc::is_remote_uri($file->get_final_requested_uri())) {
            $sniffer = $this->registry->create(Content\Type\Sniffer::class, [$file]);
            $sniffed = $sniffer->get_type();
            $mime_types = ['application/rss+xml', 'application/rdf+xml',
                                'text/rdf', 'application/atom+xml', 'text/xml',
                                'application/xml', 'application/x-rss+xml'];
            if ($check_html) {
                $mime_types[] = 'text/html';
            }

            return in_array($sniffed, $mime_types);
        } elseif (is_file($file->get_final_requested_uri())) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * @return void
     */
    public function get_base()
    {
        assert($this->registry !== null);

        if ($this->dom === null) {
            throw new \SimplePie\Exception('DOMDocument not found, unable to use locator');
        }
        $this->http_base = $this->file->get_final_requested_uri();
        $this->base = $this->http_base;
        $elements = $this->dom->getElementsByTagName('base');
        foreach ($elements as $element) {
            if ($element->hasAttribute('href')) {
                $base = $this->registry->call(Misc::class, 'absolutize_url', [trim($element->getAttribute('href')), $this->http_base]);
                if ($base === false) {
                    continue;
                }
                $this->base = $base;
                $this->base_location = method_exists($element, 'getLineNo') ? $element->getLineNo() : 0;
                break;
            }
        }
    }

    /**
     * @return array<Response>|null
     */
    public function autodiscovery()
    {
        $done = [];
        $feeds = [];
        $feeds = array_merge($feeds, $this->search_elements_by_tag('link', $done, $feeds));
        $feeds = array_merge($feeds, $this->search_elements_by_tag('a', $done, $feeds));
        $feeds = array_merge($feeds, $this->search_elements_by_tag('area', $done, $feeds));

        if (!empty($feeds)) {
            return array_values($feeds);
        }

        return null;
    }

    /**
     * @param string[] $done
     * @param array<string, Response> $feeds
     * @return array<string, Response>
     */
    protected function search_elements_by_tag(string $name, array &$done, array $feeds)
    {
        assert($this->registry !== null);

        if ($this->dom === null) {
            throw new \SimplePie\Exception('DOMDocument not found, unable to use locator');
        }

        $links = $this->dom->getElementsByTagName($name);
        foreach ($links as $link) {
            if ($this->checked_feeds === $this->max_checked_feeds) {
                break;
            }
            if ($link->hasAttribute('href') && $link->hasAttribute('rel')) {
                $rel = array_unique($this->registry->call(Misc::class, 'space_separated_tokens', [strtolower($link->getAttribute('rel'))]));
                $line = method_exists($link, 'getLineNo') ? $link->getLineNo() : 1;

                if ($this->base_location < $line) {
                    $href = $this->registry->call(Misc::class, 'absolutize_url', [trim($link->getAttribute('href')), $this->base]);
                } else {
                    $href = $this->registry->call(Misc::class, 'absolutize_url', [trim($link->getAttribute('href')), $this->http_base]);
                }
                if ($href === false) {
                    continue;
                }

                if (!in_array($href, $done) && in_array('feed', $rel) || (in_array('alternate', $rel) && !in_array('stylesheet', $rel) && $link->hasAttribute('type') && in_array(strtolower($this->registry->call(Misc::class, 'parse_mime', [$link->getAttribute('type')])), ['text/html', 'application/rss+xml', 'application/atom+xml'])) && !isset($feeds[$href])) {
                    $this->checked_feeds++;
                    $headers = [
                        'Accept' => SimplePie::DEFAULT_HTTP_ACCEPT_HEADER,
                    ];

                    try {
                        $feed = $this->get_http_client()->request(Client::METHOD_GET, $href, $headers);

                        if ((!Misc::is_remote_uri($feed->get_final_requested_uri()) || ($feed->get_status_code() === 200 || $feed->get_status_code() > 206 && $feed->get_status_code() < 300)) && $this->is_feed($feed, true)) {
                            $feeds[$href] = $feed;
                        }
                    } catch (ClientException $th) {
                        // Just mark it as done and continue.
                    }
                }
                $done[] = $href;
            }
        }

        return $feeds;
    }

    /**
     * @return true|null
     */
    public function get_links()
    {
        assert($this->registry !== null);

        if ($this->dom === null) {
            throw new \SimplePie\Exception('DOMDocument not found, unable to use locator');
        }

        $links = $this->dom->getElementsByTagName('a');
        foreach ($links as $link) {
            if ($link->hasAttribute('href')) {
                $href = trim($link->getAttribute('href'));
                $parsed = $this->registry->call(Misc::class, 'parse_url', [$href]);
                if ($parsed['scheme'] === '' || preg_match('/^(https?|feed)?$/i', $parsed['scheme'])) {
                    if (method_exists($link, 'getLineNo') && $this->base_location < $link->getLineNo()) {
                        $href = $this->registry->call(Misc::class, 'absolutize_url', [trim($link->getAttribute('href')), $this->base]);
                    } else {
                        $href = $this->registry->call(Misc::class, 'absolutize_url', [trim($link->getAttribute('href')), $this->http_base]);
                    }
                    if ($href === false) {
                        continue;
                    }

                    $current = $this->registry->call(Misc::class, 'parse_url', [$this->file->get_final_requested_uri()]);

                    if ($parsed['authority'] === '' || $parsed['authority'] === $current['authority']) {
                        $this->local[] = $href;
                    } else {
                        $this->elsewhere[] = $href;
                    }
                }
            }
        }
        $this->local = array_unique($this->local);
        $this->elsewhere = array_unique($this->elsewhere);
        if (!empty($this->local) || !empty($this->elsewhere)) {
            return true;
        }
        return null;
    }

    /**
     * Extracts first `link` element with given `rel` attribute inside the `head` element.
     *
     * @return string|null
     */
    public function get_rel_link(string $rel)
    {
        assert($this->registry !== null);

        if ($this->dom === null) {
            throw new \SimplePie\Exception('DOMDocument not found, unable to use '.
                                          'locator');
        }
        if (!class_exists('DOMXpath')) {
            throw new \SimplePie\Exception('DOMXpath not found, unable to use '.
                                          'get_rel_link');
        }

        $xpath = new \DOMXpath($this->dom);
        $query = '(//head)[1]/link[@rel and @href]';
        /** @var \DOMNodeList<\DOMElement> */
        $queryResult = $xpath->query($query);
        foreach ($queryResult as $link) {
            $href = trim($link->getAttribute('href'));
            $parsed = $this->registry->call(Misc::class, 'parse_url', [$href]);
            if ($parsed['scheme'] === '' ||
                preg_match('/^https?$/i', $parsed['scheme'])) {
                if (method_exists($link, 'getLineNo') &&
                    $this->base_location < $link->getLineNo()) {
                    $href = $this->registry->call(
                        Misc::class,
                        'absolutize_url',
                        [trim($link->getAttribute('href')), $this->base]
                    );
                } else {
                    $href = $this->registry->call(
                        Misc::class,
                        'absolutize_url',
                        [trim($link->getAttribute('href')), $this->http_base]
                    );
                }
                if ($href === false) {
                    return null;
                }
                $rel_values = explode(' ', strtolower($link->getAttribute('rel')));
                if (in_array($rel, $rel_values)) {
                    return $href;
                }
            }
        }

        return null;
    }

    /**
     * @param string[] $array
     * @return array<Response>|null
     */
    public function extension(array &$array)
    {
        foreach ($array as $key => $value) {
            if ($this->checked_feeds === $this->max_checked_feeds) {
                break;
            }
            $extension = strrchr($value, '.');
            if ($extension !== false && in_array(strtolower($extension), ['.rss', '.rdf', '.atom', '.xml'])) {
                $this->checked_feeds++;

                $headers = [
                    'Accept' => SimplePie::DEFAULT_HTTP_ACCEPT_HEADER,
                ];

                try {
                    $feed = $this->get_http_client()->request(Client::METHOD_GET, $value, $headers);

                    if ((!Misc::is_remote_uri($feed->get_final_requested_uri()) || ($feed->get_status_code() === 200 || $feed->get_status_code() > 206 && $feed->get_status_code() < 300)) && $this->is_feed($feed)) {
                        return [$feed];
                    }
                } catch (ClientException $th) {
                    // Just unset and continue.
                }

                unset($array[$key]);
            }
        }
        return null;
    }

    /**
     * @param string[] $array
     * @return array<Response>|null
     */
    public function body(array &$array)
    {
        foreach ($array as $key => $value) {
            if ($this->checked_feeds === $this->max_checked_feeds) {
                break;
            }
            if (preg_match('/(feed|rss|rdf|atom|xml)/i', $value)) {
                $this->checked_feeds++;
                $headers = [
                    'Accept' => SimplePie::DEFAULT_HTTP_ACCEPT_HEADER,
                ];

                try {
                    $feed = $this->get_http_client()->request(Client::METHOD_GET, $value, $headers);

                    if ((!Misc::is_remote_uri($feed->get_final_requested_uri()) || ($feed->get_status_code() === 200 || $feed->get_status_code() > 206 && $feed->get_status_code() < 300)) && $this->is_feed($feed)) {
                        return [$feed];
                    }
                } catch (ClientException $th) {
                    // Just unset and continue.
                }

                unset($array[$key]);
            }
        }
        return null;
    }

    /**
     * Get a HTTP client
     */
    private function get_http_client(): Client
    {
        assert($this->registry !== null);

        if ($this->http_client === null) {
            $options = [
                'timeout' => $this->timeout,
                'redirects' => 5,
                'force_fsockopen' => $this->force_fsockopen,
                'curl_options' => $this->curl_options,
            ];

            if ($this->useragent !== null) {
                $options['useragent'] = $this->useragent;
            }

            return new FileClient(
                $this->registry,
                $options
            );
        }

        return $this->http_client;
    }
}

class_alias('SimplePie\Locator', 'SimplePie_Locator', false);

}

namespace SimplePie {

use SimplePie\XML\Declaration\Parser;

/**
 * Miscellaneous utilities
 */
class Misc
{
    /** @var int|null */
    private static $SIMPLEPIE_BUILD = null;

    /**
     * @return string
     */
    public static function time_hms(int $seconds)
    {
        $time = '';

        $hours = floor($seconds / 3600);
        $remainder = $seconds % 3600;
        if ($hours > 0) {
            $time .= $hours.':';
        }

        $minutes = floor($remainder / 60);
        $seconds = $remainder % 60;
        if ($minutes < 10 && $hours > 0) {
            $minutes = '0' . $minutes;
        }
        if ($seconds < 10) {
            $seconds = '0' . $seconds;
        }

        $time .= $minutes.':';
        $time .= $seconds;

        return $time;
    }

    /**
     * @return string|false
     */
    public static function absolutize_url(string $relative, string $base)
    {
        $iri = \SimplePie\IRI::absolutize(new \SimplePie\IRI($base), $relative);
        if ($iri === false) {
            return false;
        }
        return $iri->get_uri();
    }

    /**
     * @internal
     */
    public static function is_remote_uri(string $uri): bool
    {
        return preg_match('/^https?:\/\//i', $uri) === 1;
    }

    /**
     * Get a HTML/XML element from a HTML string
     *
     * @deprecated since SimplePie 1.3, use DOMDocument instead (parsing HTML with regex is bad!)
     * @param string $realname Element name (including namespace prefix if applicable)
     * @param string $string HTML document
     * @return array<array{tag: string, self_closing: bool, attribs: array<string, array{data: string}>, content?: string}>
     */
    public static function get_element(string $realname, string $string)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.3, use "DOMDocument" instead.'), \E_USER_DEPRECATED);

        $return = [];
        $name = preg_quote($realname, '/');
        if (preg_match_all("/<($name)" . \SimplePie\SimplePie::PCRE_HTML_ATTRIBUTE . "(>(.*)<\/$name>|(\/)?>)/siU", $string, $matches, PREG_SET_ORDER | PREG_OFFSET_CAPTURE)) {
            for ($i = 0, $total_matches = count($matches); $i < $total_matches; $i++) {
                $return[$i]['tag'] = $realname;
                $return[$i]['full'] = $matches[$i][0][0];
                $return[$i]['offset'] = $matches[$i][0][1];
                if (strlen($matches[$i][3][0]) <= 2) {
                    $return[$i]['self_closing'] = true;
                } else {
                    $return[$i]['self_closing'] = false;
                    $return[$i]['content'] = $matches[$i][4][0];
                }
                $return[$i]['attribs'] = [];
                if (isset($matches[$i][2][0]) && preg_match_all('/[\x09\x0A\x0B\x0C\x0D\x20]+([^\x09\x0A\x0B\x0C\x0D\x20\x2F\x3E][^\x09\x0A\x0B\x0C\x0D\x20\x2F\x3D\x3E]*)(?:[\x09\x0A\x0B\x0C\x0D\x20]*=[\x09\x0A\x0B\x0C\x0D\x20]*(?:"([^"]*)"|\'([^\']*)\'|([^\x09\x0A\x0B\x0C\x0D\x20\x22\x27\x3E][^\x09\x0A\x0B\x0C\x0D\x20\x3E]*)?))?/', ' ' . $matches[$i][2][0] . ' ', $attribs, PREG_SET_ORDER)) {
                    foreach ($attribs as $attrib) {
                        if (count($attrib) === 2) {
                            $attrib[2] = $attrib[1];
                        }
                        $return[$i]['attribs'][strtolower($attrib[1])]['data'] = Misc::entities_decode(end($attrib));
                    }
                }
            }
        }
        return $return;
    }

    /**
     * @deprecated since SimplePie 1.9.0. If you need it, you can copy the function to your codebase. But you should consider using `DOMDocument` for any DOM wrangling.
     * @param array{tag: string, self_closing: bool, attribs: array<string, array{data: string}>, content: string} $element
     * @return string
     */
    public static function element_implode(array $element)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.9.'), \E_USER_DEPRECATED);

        $full = "<{$element['tag']}";
        foreach ($element['attribs'] as $key => $value) {
            $key = strtolower($key);
            $full .= " $key=\"" . htmlspecialchars($value['data'], ENT_COMPAT, 'UTF-8') . '"';
        }
        if ($element['self_closing']) {
            $full .= ' />';
        } else {
            $full .= ">{$element['content']}</{$element['tag']}>";
        }
        return $full;
    }

    /**
     * @param string $message
     * @param int $level
     * @param string $file
     * @param int $line
     * @return string
     */
    public static function error(string $message, int $level, string $file, int $line)
    {
        if ((error_reporting() & $level) > 0) {
            switch ($level) {
                case E_USER_ERROR:
                    $note = 'PHP Error';
                    break;
                case E_USER_WARNING:
                    $note = 'PHP Warning';
                    break;
                case E_USER_NOTICE:
                    $note = 'PHP Notice';
                    break;
                default:
                    $note = 'Unknown Error';
                    break;
            }

            $log_error = true;
            if (!function_exists('error_log')) {
                $log_error = false;
            }

            $log_file = @ini_get('error_log');
            if (!empty($log_file) && ('syslog' !== $log_file) && !@is_writable($log_file)) {
                $log_error = false;
            }

            if ($log_error) {
                @error_log("$note: $message in $file on line $line", 0);
            }
        }

        return $message;
    }

    /**
     * @return string
     */
    public static function fix_protocol(string $url, int $http = 1)
    {
        $url = Misc::normalize_url($url);
        $parsed = Misc::parse_url($url);
        if ($parsed['scheme'] !== '' && $parsed['scheme'] !== 'http' && $parsed['scheme'] !== 'https') {
            return Misc::fix_protocol(Misc::compress_parse_url('http', $parsed['authority'], $parsed['path'], $parsed['query'], $parsed['fragment']), $http);
        }

        if ($parsed['scheme'] === '' && $parsed['authority'] === '' && !file_exists($url)) {
            return Misc::fix_protocol(Misc::compress_parse_url('http', $parsed['path'], '', $parsed['query'], $parsed['fragment']), $http);
        }

        if ($http === 2 && $parsed['scheme'] !== '') {
            return "feed:$url";
        } elseif ($http === 3 && strtolower($parsed['scheme']) === 'http') {
            return substr_replace($url, 'podcast', 0, 4);
        } elseif ($http === 4 && strtolower($parsed['scheme']) === 'http') {
            return substr_replace($url, 'itpc', 0, 4);
        }

        return $url;
    }

    /**
     * @deprecated since SimplePie 1.8.0, use PHP native array_replace_recursive() instead.
     * @param array<mixed> $array1
     * @param array<mixed> $array2
     * @return array<mixed>
     */
    public static function array_merge_recursive(array $array1, array $array2)
    {
        foreach ($array2 as $key => $value) {
            if (is_array($value)) {
                $array1[$key] = Misc::array_merge_recursive($array1[$key], $value);
            } else {
                $array1[$key] = $value;
            }
        }

        return $array1;
    }

    /**
     * @return array<string, string>
     */
    public static function parse_url(string $url)
    {
        $iri = new \SimplePie\IRI($url);
        return [
            'scheme' => (string) $iri->scheme,
            'authority' => (string) $iri->authority,
            'path' => (string) $iri->path,
            'query' => (string) $iri->query,
            'fragment' => (string) $iri->fragment
        ];
    }

    /**
     * @return string
     */
    public static function compress_parse_url(string $scheme = '', string $authority = '', string $path = '', string $query = '', ?string $fragment = '')
    {
        $iri = new \SimplePie\IRI('');
        $iri->scheme = $scheme;
        $iri->authority = $authority;
        $iri->path = $path;
        $iri->query = $query;
        $iri->fragment = $fragment;
        return $iri->get_uri();
    }

    /**
     * @return string
     */
    public static function normalize_url(string $url)
    {
        $iri = new \SimplePie\IRI($url);
        return $iri->get_uri();
    }

    /**
     * @deprecated since SimplePie 1.9.0. This functionality is part of `IRI` – if you need it standalone, consider copying the function to your codebase.
     * @param array<int, string> $match
     * @return string
     */
    public static function percent_encoding_normalization(array $match)
    {
        $integer = hexdec($match[1]);
        if ($integer >= 0x41 && $integer <= 0x5A || $integer >= 0x61 && $integer <= 0x7A || $integer >= 0x30 && $integer <= 0x39 || $integer === 0x2D || $integer === 0x2E || $integer === 0x5F || $integer === 0x7E) {
            // Cast for PHPStan, the value would only be float when above PHP_INT_MAX, which would not go in this branch.
            return chr((int) $integer);
        }

        return strtoupper($match[0]);
    }

    /**
     * Converts a Windows-1252 encoded string to a UTF-8 encoded string
     *
     * @static
     * @param string $string Windows-1252 encoded string
     * @return string UTF-8 encoded string
     */
    public static function windows_1252_to_utf8(string $string)
    {
        static $convert_table = ["\x80" => "\xE2\x82\xAC", "\x81" => "\xEF\xBF\xBD", "\x82" => "\xE2\x80\x9A", "\x83" => "\xC6\x92", "\x84" => "\xE2\x80\x9E", "\x85" => "\xE2\x80\xA6", "\x86" => "\xE2\x80\xA0", "\x87" => "\xE2\x80\xA1", "\x88" => "\xCB\x86", "\x89" => "\xE2\x80\xB0", "\x8A" => "\xC5\xA0", "\x8B" => "\xE2\x80\xB9", "\x8C" => "\xC5\x92", "\x8D" => "\xEF\xBF\xBD", "\x8E" => "\xC5\xBD", "\x8F" => "\xEF\xBF\xBD", "\x90" => "\xEF\xBF\xBD", "\x91" => "\xE2\x80\x98", "\x92" => "\xE2\x80\x99", "\x93" => "\xE2\x80\x9C", "\x94" => "\xE2\x80\x9D", "\x95" => "\xE2\x80\xA2", "\x96" => "\xE2\x80\x93", "\x97" => "\xE2\x80\x94", "\x98" => "\xCB\x9C", "\x99" => "\xE2\x84\xA2", "\x9A" => "\xC5\xA1", "\x9B" => "\xE2\x80\xBA", "\x9C" => "\xC5\x93", "\x9D" => "\xEF\xBF\xBD", "\x9E" => "\xC5\xBE", "\x9F" => "\xC5\xB8", "\xA0" => "\xC2\xA0", "\xA1" => "\xC2\xA1", "\xA2" => "\xC2\xA2", "\xA3" => "\xC2\xA3", "\xA4" => "\xC2\xA4", "\xA5" => "\xC2\xA5", "\xA6" => "\xC2\xA6", "\xA7" => "\xC2\xA7", "\xA8" => "\xC2\xA8", "\xA9" => "\xC2\xA9", "\xAA" => "\xC2\xAA", "\xAB" => "\xC2\xAB", "\xAC" => "\xC2\xAC", "\xAD" => "\xC2\xAD", "\xAE" => "\xC2\xAE", "\xAF" => "\xC2\xAF", "\xB0" => "\xC2\xB0", "\xB1" => "\xC2\xB1", "\xB2" => "\xC2\xB2", "\xB3" => "\xC2\xB3", "\xB4" => "\xC2\xB4", "\xB5" => "\xC2\xB5", "\xB6" => "\xC2\xB6", "\xB7" => "\xC2\xB7", "\xB8" => "\xC2\xB8", "\xB9" => "\xC2\xB9", "\xBA" => "\xC2\xBA", "\xBB" => "\xC2\xBB", "\xBC" => "\xC2\xBC", "\xBD" => "\xC2\xBD", "\xBE" => "\xC2\xBE", "\xBF" => "\xC2\xBF", "\xC0" => "\xC3\x80", "\xC1" => "\xC3\x81", "\xC2" => "\xC3\x82", "\xC3" => "\xC3\x83", "\xC4" => "\xC3\x84", "\xC5" => "\xC3\x85", "\xC6" => "\xC3\x86", "\xC7" => "\xC3\x87", "\xC8" => "\xC3\x88", "\xC9" => "\xC3\x89", "\xCA" => "\xC3\x8A", "\xCB" => "\xC3\x8B", "\xCC" => "\xC3\x8C", "\xCD" => "\xC3\x8D", "\xCE" => "\xC3\x8E", "\xCF" => "\xC3\x8F", "\xD0" => "\xC3\x90", "\xD1" => "\xC3\x91", "\xD2" => "\xC3\x92", "\xD3" => "\xC3\x93", "\xD4" => "\xC3\x94", "\xD5" => "\xC3\x95", "\xD6" => "\xC3\x96", "\xD7" => "\xC3\x97", "\xD8" => "\xC3\x98", "\xD9" => "\xC3\x99", "\xDA" => "\xC3\x9A", "\xDB" => "\xC3\x9B", "\xDC" => "\xC3\x9C", "\xDD" => "\xC3\x9D", "\xDE" => "\xC3\x9E", "\xDF" => "\xC3\x9F", "\xE0" => "\xC3\xA0", "\xE1" => "\xC3\xA1", "\xE2" => "\xC3\xA2", "\xE3" => "\xC3\xA3", "\xE4" => "\xC3\xA4", "\xE5" => "\xC3\xA5", "\xE6" => "\xC3\xA6", "\xE7" => "\xC3\xA7", "\xE8" => "\xC3\xA8", "\xE9" => "\xC3\xA9", "\xEA" => "\xC3\xAA", "\xEB" => "\xC3\xAB", "\xEC" => "\xC3\xAC", "\xED" => "\xC3\xAD", "\xEE" => "\xC3\xAE", "\xEF" => "\xC3\xAF", "\xF0" => "\xC3\xB0", "\xF1" => "\xC3\xB1", "\xF2" => "\xC3\xB2", "\xF3" => "\xC3\xB3", "\xF4" => "\xC3\xB4", "\xF5" => "\xC3\xB5", "\xF6" => "\xC3\xB6", "\xF7" => "\xC3\xB7", "\xF8" => "\xC3\xB8", "\xF9" => "\xC3\xB9", "\xFA" => "\xC3\xBA", "\xFB" => "\xC3\xBB", "\xFC" => "\xC3\xBC", "\xFD" => "\xC3\xBD", "\xFE" => "\xC3\xBE", "\xFF" => "\xC3\xBF"];

        return strtr($string, $convert_table);
    }

    /**
     * Change a string from one encoding to another
     *
     * @param string $data Raw data in $input encoding
     * @param string $input Encoding of $data
     * @param string $output Encoding you want
     * @return string|false False if we can't convert it
     */
    public static function change_encoding(string $data, string $input, string $output)
    {
        $input = Misc::encoding($input);
        $output = Misc::encoding($output);

        // We fail to fail on non US-ASCII bytes
        if ($input === 'US-ASCII') {
            static $non_ascii_octets = '';
            if (!$non_ascii_octets) {
                for ($i = 0x80; $i <= 0xFF; $i++) {
                    $non_ascii_octets .= chr($i);
                }
            }
            $data = substr($data, 0, strcspn($data, $non_ascii_octets));
        }

        // This is first, as behaviour of this is completely predictable
        if ($input === 'windows-1252' && $output === 'UTF-8') {
            return Misc::windows_1252_to_utf8($data);
        }
        // This is second, as behaviour of this varies only with PHP version (the middle part of this expression checks the encoding is supported).
        elseif (function_exists('mb_convert_encoding') && ($return = Misc::change_encoding_mbstring($data, $input, $output))) {
            return $return;
        }
        // This is third, as behaviour of this varies with OS userland and PHP version
        elseif (function_exists('iconv') && ($return = Misc::change_encoding_iconv($data, $input, $output))) {
            return $return;
        }
        // This is last, as behaviour of this varies with OS userland and PHP version
        elseif (class_exists('\UConverter') && ($return = Misc::change_encoding_uconverter($data, $input, $output))) {
            return $return;
        }

        // If we can't do anything, just fail
        return false;
    }

    /**
     * @return string|false
     */
    protected static function change_encoding_mbstring(string $data, string $input, string $output)
    {
        if ($input === 'windows-949') {
            $input = 'EUC-KR';
        }
        if ($output === 'windows-949') {
            $output = 'EUC-KR';
        }
        if ($input === 'Windows-31J') {
            $input = 'SJIS';
        }
        if ($output === 'Windows-31J') {
            $output = 'SJIS';
        }

        // Check that the encoding is supported
        if (!in_array($input, mb_list_encodings())) {
            return false;
        }

        if (@mb_convert_encoding("\x80", 'UTF-16BE', $input) === "\x00\x80") {
            return false;
        }

        // Let's do some conversion
        if ($return = @mb_convert_encoding($data, $output, $input)) {
            return $return;
        }

        return false;
    }

    /**
     * @return string|false
     */
    protected static function change_encoding_iconv(string $data, string $input, string $output)
    {
        return @iconv($input, $output, $data);
    }

    /**
     * @return string|false
     */
    protected static function change_encoding_uconverter(string $data, string $input, string $output)
    {
        return @\UConverter::transcode($data, $output, $input);
    }

    /**
     * Normalize an encoding name
     *
     * This is automatically generated by create.php
     *
     * To generate it, run `php create.php` on the command line, and copy the
     * output to replace this function.
     *
     * @param string $charset Character set to standardise
     * @return string Standardised name
     */
    public static function encoding(string $charset)
    {
        // Normalization from UTS #22
        // Cast for PHPStan, the regex should not fail.
        switch (strtolower((string) preg_replace('/(?:[^a-zA-Z0-9]+|([^0-9])0+)/', '\1', $charset))) {
            case 'adobestandardencoding':
            case 'csadobestandardencoding':
                return 'Adobe-Standard-Encoding';

            case 'adobesymbolencoding':
            case 'cshppsmath':
                return 'Adobe-Symbol-Encoding';

            case 'ami1251':
            case 'amiga1251':
                return 'Amiga-1251';

            case 'ansix31101983':
            case 'csat5001983':
            case 'csiso99naplps':
            case 'isoir99':
            case 'naplps':
                return 'ANSI_X3.110-1983';

            case 'arabic7':
            case 'asmo449':
            case 'csiso89asmo449':
            case 'iso9036':
            case 'isoir89':
                return 'ASMO_449';

            case 'big5':
            case 'csbig5':
                return 'Big5';

            case 'big5hkscs':
                return 'Big5-HKSCS';

            case 'bocu1':
            case 'csbocu1':
                return 'BOCU-1';

            case 'brf':
            case 'csbrf':
                return 'BRF';

            case 'bs4730':
            case 'csiso4unitedkingdom':
            case 'gb':
            case 'iso646gb':
            case 'isoir4':
            case 'uk':
                return 'BS_4730';

            case 'bsviewdata':
            case 'csiso47bsviewdata':
            case 'isoir47':
                return 'BS_viewdata';

            case 'cesu8':
            case 'cscesu8':
                return 'CESU-8';

            case 'ca':
            case 'csa71':
            case 'csaz243419851':
            case 'csiso121canadian1':
            case 'iso646ca':
            case 'isoir121':
                return 'CSA_Z243.4-1985-1';

            case 'csa72':
            case 'csaz243419852':
            case 'csiso122canadian2':
            case 'iso646ca2':
            case 'isoir122':
                return 'CSA_Z243.4-1985-2';

            case 'csaz24341985gr':
            case 'csiso123csaz24341985gr':
            case 'isoir123':
                return 'CSA_Z243.4-1985-gr';

            case 'csiso139csn369103':
            case 'csn369103':
            case 'isoir139':
                return 'CSN_369103';

            case 'csdecmcs':
            case 'dec':
            case 'decmcs':
                return 'DEC-MCS';

            case 'csiso21german':
            case 'de':
            case 'din66003':
            case 'iso646de':
            case 'isoir21':
                return 'DIN_66003';

            case 'csdkus':
            case 'dkus':
                return 'dk-us';

            case 'csiso646danish':
            case 'dk':
            case 'ds2089':
            case 'iso646dk':
                return 'DS_2089';

            case 'csibmebcdicatde':
            case 'ebcdicatde':
                return 'EBCDIC-AT-DE';

            case 'csebcdicatdea':
            case 'ebcdicatdea':
                return 'EBCDIC-AT-DE-A';

            case 'csebcdiccafr':
            case 'ebcdiccafr':
                return 'EBCDIC-CA-FR';

            case 'csebcdicdkno':
            case 'ebcdicdkno':
                return 'EBCDIC-DK-NO';

            case 'csebcdicdknoa':
            case 'ebcdicdknoa':
                return 'EBCDIC-DK-NO-A';

            case 'csebcdices':
            case 'ebcdices':
                return 'EBCDIC-ES';

            case 'csebcdicesa':
            case 'ebcdicesa':
                return 'EBCDIC-ES-A';

            case 'csebcdicess':
            case 'ebcdicess':
                return 'EBCDIC-ES-S';

            case 'csebcdicfise':
            case 'ebcdicfise':
                return 'EBCDIC-FI-SE';

            case 'csebcdicfisea':
            case 'ebcdicfisea':
                return 'EBCDIC-FI-SE-A';

            case 'csebcdicfr':
            case 'ebcdicfr':
                return 'EBCDIC-FR';

            case 'csebcdicit':
            case 'ebcdicit':
                return 'EBCDIC-IT';

            case 'csebcdicpt':
            case 'ebcdicpt':
                return 'EBCDIC-PT';

            case 'csebcdicuk':
            case 'ebcdicuk':
                return 'EBCDIC-UK';

            case 'csebcdicus':
            case 'ebcdicus':
                return 'EBCDIC-US';

            case 'csiso111ecmacyrillic':
            case 'ecmacyrillic':
            case 'isoir111':
            case 'koi8e':
                return 'ECMA-cyrillic';

            case 'csiso17spanish':
            case 'es':
            case 'iso646es':
            case 'isoir17':
                return 'ES';

            case 'csiso85spanish2':
            case 'es2':
            case 'iso646es2':
            case 'isoir85':
                return 'ES2';

            case 'cseucpkdfmtjapanese':
            case 'eucjp':
            case 'extendedunixcodepackedformatforjapanese':
                return 'EUC-JP';

            case 'cseucfixwidjapanese':
            case 'extendedunixcodefixedwidthforjapanese':
                return 'Extended_UNIX_Code_Fixed_Width_for_Japanese';

            case 'gb18030':
                return 'GB18030';

            case 'chinese':
            case 'cp936':
            case 'csgb2312':
            case 'csiso58gb231280':
            case 'gb2312':
            case 'gb231280':
            case 'gbk':
            case 'isoir58':
            case 'ms936':
            case 'windows936':
                return 'GBK';

            case 'cn':
            case 'csiso57gb1988':
            case 'gb198880':
            case 'iso646cn':
            case 'isoir57':
                return 'GB_1988-80';

            case 'csiso153gost1976874':
            case 'gost1976874':
            case 'isoir153':
            case 'stsev35888':
                return 'GOST_19768-74';

            case 'csiso150':
            case 'csiso150greekccitt':
            case 'greekccitt':
            case 'isoir150':
                return 'greek-ccitt';

            case 'csiso88greek7':
            case 'greek7':
            case 'isoir88':
                return 'greek7';

            case 'csiso18greek7old':
            case 'greek7old':
            case 'isoir18':
                return 'greek7-old';

            case 'cshpdesktop':
            case 'hpdesktop':
                return 'HP-DeskTop';

            case 'cshplegal':
            case 'hplegal':
                return 'HP-Legal';

            case 'cshpmath8':
            case 'hpmath8':
                return 'HP-Math8';

            case 'cshppifont':
            case 'hppifont':
                return 'HP-Pi-font';

            case 'cshproman8':
            case 'hproman8':
            case 'r8':
            case 'roman8':
                return 'hp-roman8';

            case 'hzgb2312':
                return 'HZ-GB-2312';

            case 'csibmsymbols':
            case 'ibmsymbols':
                return 'IBM-Symbols';

            case 'csibmthai':
            case 'ibmthai':
                return 'IBM-Thai';

            case 'cp37':
            case 'csibm37':
            case 'ebcdiccpca':
            case 'ebcdiccpnl':
            case 'ebcdiccpus':
            case 'ebcdiccpwt':
            case 'ibm37':
                return 'IBM037';

            case 'cp38':
            case 'csibm38':
            case 'ebcdicint':
            case 'ibm38':
                return 'IBM038';

            case 'cp273':
            case 'csibm273':
            case 'ibm273':
                return 'IBM273';

            case 'cp274':
            case 'csibm274':
            case 'ebcdicbe':
            case 'ibm274':
                return 'IBM274';

            case 'cp275':
            case 'csibm275':
            case 'ebcdicbr':
            case 'ibm275':
                return 'IBM275';

            case 'csibm277':
            case 'ebcdiccpdk':
            case 'ebcdiccpno':
            case 'ibm277':
                return 'IBM277';

            case 'cp278':
            case 'csibm278':
            case 'ebcdiccpfi':
            case 'ebcdiccpse':
            case 'ibm278':
                return 'IBM278';

            case 'cp280':
            case 'csibm280':
            case 'ebcdiccpit':
            case 'ibm280':
                return 'IBM280';

            case 'cp281':
            case 'csibm281':
            case 'ebcdicjpe':
            case 'ibm281':
                return 'IBM281';

            case 'cp284':
            case 'csibm284':
            case 'ebcdiccpes':
            case 'ibm284':
                return 'IBM284';

            case 'cp285':
            case 'csibm285':
            case 'ebcdiccpgb':
            case 'ibm285':
                return 'IBM285';

            case 'cp290':
            case 'csibm290':
            case 'ebcdicjpkana':
            case 'ibm290':
                return 'IBM290';

            case 'cp297':
            case 'csibm297':
            case 'ebcdiccpfr':
            case 'ibm297':
                return 'IBM297';

            case 'cp420':
            case 'csibm420':
            case 'ebcdiccpar1':
            case 'ibm420':
                return 'IBM420';

            case 'cp423':
            case 'csibm423':
            case 'ebcdiccpgr':
            case 'ibm423':
                return 'IBM423';

            case 'cp424':
            case 'csibm424':
            case 'ebcdiccphe':
            case 'ibm424':
                return 'IBM424';

            case '437':
            case 'cp437':
            case 'cspc8codepage437':
            case 'ibm437':
                return 'IBM437';

            case 'cp500':
            case 'csibm500':
            case 'ebcdiccpbe':
            case 'ebcdiccpch':
            case 'ibm500':
                return 'IBM500';

            case 'cp775':
            case 'cspc775baltic':
            case 'ibm775':
                return 'IBM775';

            case '850':
            case 'cp850':
            case 'cspc850multilingual':
            case 'ibm850':
                return 'IBM850';

            case '851':
            case 'cp851':
            case 'csibm851':
            case 'ibm851':
                return 'IBM851';

            case '852':
            case 'cp852':
            case 'cspcp852':
            case 'ibm852':
                return 'IBM852';

            case '855':
            case 'cp855':
            case 'csibm855':
            case 'ibm855':
                return 'IBM855';

            case '857':
            case 'cp857':
            case 'csibm857':
            case 'ibm857':
                return 'IBM857';

            case 'ccsid858':
            case 'cp858':
            case 'ibm858':
            case 'pcmultilingual850euro':
                return 'IBM00858';

            case '860':
            case 'cp860':
            case 'csibm860':
            case 'ibm860':
                return 'IBM860';

            case '861':
            case 'cp861':
            case 'cpis':
            case 'csibm861':
            case 'ibm861':
                return 'IBM861';

            case '862':
            case 'cp862':
            case 'cspc862latinhebrew':
            case 'ibm862':
                return 'IBM862';

            case '863':
            case 'cp863':
            case 'csibm863':
            case 'ibm863':
                return 'IBM863';

            case 'cp864':
            case 'csibm864':
            case 'ibm864':
                return 'IBM864';

            case '865':
            case 'cp865':
            case 'csibm865':
            case 'ibm865':
                return 'IBM865';

            case '866':
            case 'cp866':
            case 'csibm866':
            case 'ibm866':
                return 'IBM866';

            case 'cp868':
            case 'cpar':
            case 'csibm868':
            case 'ibm868':
                return 'IBM868';

            case '869':
            case 'cp869':
            case 'cpgr':
            case 'csibm869':
            case 'ibm869':
                return 'IBM869';

            case 'cp870':
            case 'csibm870':
            case 'ebcdiccproece':
            case 'ebcdiccpyu':
            case 'ibm870':
                return 'IBM870';

            case 'cp871':
            case 'csibm871':
            case 'ebcdiccpis':
            case 'ibm871':
                return 'IBM871';

            case 'cp880':
            case 'csibm880':
            case 'ebcdiccyrillic':
            case 'ibm880':
                return 'IBM880';

            case 'cp891':
            case 'csibm891':
            case 'ibm891':
                return 'IBM891';

            case 'cp903':
            case 'csibm903':
            case 'ibm903':
                return 'IBM903';

            case '904':
            case 'cp904':
            case 'csibbm904':
            case 'ibm904':
                return 'IBM904';

            case 'cp905':
            case 'csibm905':
            case 'ebcdiccptr':
            case 'ibm905':
                return 'IBM905';

            case 'cp918':
            case 'csibm918':
            case 'ebcdiccpar2':
            case 'ibm918':
                return 'IBM918';

            case 'ccsid924':
            case 'cp924':
            case 'ebcdiclatin9euro':
            case 'ibm924':
                return 'IBM00924';

            case 'cp1026':
            case 'csibm1026':
            case 'ibm1026':
                return 'IBM1026';

            case 'ibm1047':
                return 'IBM1047';

            case 'ccsid1140':
            case 'cp1140':
            case 'ebcdicus37euro':
            case 'ibm1140':
                return 'IBM01140';

            case 'ccsid1141':
            case 'cp1141':
            case 'ebcdicde273euro':
            case 'ibm1141':
                return 'IBM01141';

            case 'ccsid1142':
            case 'cp1142':
            case 'ebcdicdk277euro':
            case 'ebcdicno277euro':
            case 'ibm1142':
                return 'IBM01142';

            case 'ccsid1143':
            case 'cp1143':
            case 'ebcdicfi278euro':
            case 'ebcdicse278euro':
            case 'ibm1143':
                return 'IBM01143';

            case 'ccsid1144':
            case 'cp1144':
            case 'ebcdicit280euro':
            case 'ibm1144':
                return 'IBM01144';

            case 'ccsid1145':
            case 'cp1145':
            case 'ebcdices284euro':
            case 'ibm1145':
                return 'IBM01145';

            case 'ccsid1146':
            case 'cp1146':
            case 'ebcdicgb285euro':
            case 'ibm1146':
                return 'IBM01146';

            case 'ccsid1147':
            case 'cp1147':
            case 'ebcdicfr297euro':
            case 'ibm1147':
                return 'IBM01147';

            case 'ccsid1148':
            case 'cp1148':
            case 'ebcdicinternational500euro':
            case 'ibm1148':
                return 'IBM01148';

            case 'ccsid1149':
            case 'cp1149':
            case 'ebcdicis871euro':
            case 'ibm1149':
                return 'IBM01149';

            case 'csiso143iecp271':
            case 'iecp271':
            case 'isoir143':
                return 'IEC_P27-1';

            case 'csiso49inis':
            case 'inis':
            case 'isoir49':
                return 'INIS';

            case 'csiso50inis8':
            case 'inis8':
            case 'isoir50':
                return 'INIS-8';

            case 'csiso51iniscyrillic':
            case 'iniscyrillic':
            case 'isoir51':
                return 'INIS-cyrillic';

            case 'csinvariant':
            case 'invariant':
                return 'INVARIANT';

            case 'iso2022cn':
                return 'ISO-2022-CN';

            case 'iso2022cnext':
                return 'ISO-2022-CN-EXT';

            case 'csiso2022jp':
            case 'iso2022jp':
                return 'ISO-2022-JP';

            case 'csiso2022jp2':
            case 'iso2022jp2':
                return 'ISO-2022-JP-2';

            case 'csiso2022kr':
            case 'iso2022kr':
                return 'ISO-2022-KR';

            case 'cswindows30latin1':
            case 'iso88591windows30latin1':
                return 'ISO-8859-1-Windows-3.0-Latin-1';

            case 'cswindows31latin1':
            case 'iso88591windows31latin1':
                return 'ISO-8859-1-Windows-3.1-Latin-1';

            case 'csisolatin2':
            case 'iso88592':
            case 'iso885921987':
            case 'isoir101':
            case 'l2':
            case 'latin2':
                return 'ISO-8859-2';

            case 'cswindows31latin2':
            case 'iso88592windowslatin2':
                return 'ISO-8859-2-Windows-Latin-2';

            case 'csisolatin3':
            case 'iso88593':
            case 'iso885931988':
            case 'isoir109':
            case 'l3':
            case 'latin3':
                return 'ISO-8859-3';

            case 'csisolatin4':
            case 'iso88594':
            case 'iso885941988':
            case 'isoir110':
            case 'l4':
            case 'latin4':
                return 'ISO-8859-4';

            case 'csisolatincyrillic':
            case 'cyrillic':
            case 'iso88595':
            case 'iso885951988':
            case 'isoir144':
                return 'ISO-8859-5';

            case 'arabic':
            case 'asmo708':
            case 'csisolatinarabic':
            case 'ecma114':
            case 'iso88596':
            case 'iso885961987':
            case 'isoir127':
                return 'ISO-8859-6';

            case 'csiso88596e':
            case 'iso88596e':
                return 'ISO-8859-6-E';

            case 'csiso88596i':
            case 'iso88596i':
                return 'ISO-8859-6-I';

            case 'csisolatingreek':
            case 'ecma118':
            case 'elot928':
            case 'greek':
            case 'greek8':
            case 'iso88597':
            case 'iso885971987':
            case 'isoir126':
                return 'ISO-8859-7';

            case 'csisolatinhebrew':
            case 'hebrew':
            case 'iso88598':
            case 'iso885981988':
            case 'isoir138':
                return 'ISO-8859-8';

            case 'csiso88598e':
            case 'iso88598e':
                return 'ISO-8859-8-E';

            case 'csiso88598i':
            case 'iso88598i':
                return 'ISO-8859-8-I';

            case 'cswindows31latin5':
            case 'iso88599windowslatin5':
                return 'ISO-8859-9-Windows-Latin-5';

            case 'csisolatin6':
            case 'iso885910':
            case 'iso8859101992':
            case 'isoir157':
            case 'l6':
            case 'latin6':
                return 'ISO-8859-10';

            case 'iso885913':
                return 'ISO-8859-13';

            case 'iso885914':
            case 'iso8859141998':
            case 'isoceltic':
            case 'isoir199':
            case 'l8':
            case 'latin8':
                return 'ISO-8859-14';

            case 'iso885915':
            case 'latin9':
                return 'ISO-8859-15';

            case 'iso885916':
            case 'iso8859162001':
            case 'isoir226':
            case 'l10':
            case 'latin10':
                return 'ISO-8859-16';

            case 'iso10646j1':
                return 'ISO-10646-J-1';

            case 'csunicode':
            case 'iso10646ucs2':
                return 'ISO-10646-UCS-2';

            case 'csucs4':
            case 'iso10646ucs4':
                return 'ISO-10646-UCS-4';

            case 'csunicodeascii':
            case 'iso10646ucsbasic':
                return 'ISO-10646-UCS-Basic';

            case 'csunicodelatin1':
            case 'iso10646':
            case 'iso10646unicodelatin1':
                return 'ISO-10646-Unicode-Latin1';

            case 'csiso10646utf1':
            case 'iso10646utf1':
                return 'ISO-10646-UTF-1';

            case 'csiso115481':
            case 'iso115481':
            case 'isotr115481':
                return 'ISO-11548-1';

            case 'csiso90':
            case 'isoir90':
                return 'iso-ir-90';

            case 'csunicodeibm1261':
            case 'isounicodeibm1261':
                return 'ISO-Unicode-IBM-1261';

            case 'csunicodeibm1264':
            case 'isounicodeibm1264':
                return 'ISO-Unicode-IBM-1264';

            case 'csunicodeibm1265':
            case 'isounicodeibm1265':
                return 'ISO-Unicode-IBM-1265';

            case 'csunicodeibm1268':
            case 'isounicodeibm1268':
                return 'ISO-Unicode-IBM-1268';

            case 'csunicodeibm1276':
            case 'isounicodeibm1276':
                return 'ISO-Unicode-IBM-1276';

            case 'csiso646basic1983':
            case 'iso646basic1983':
            case 'ref':
                return 'ISO_646.basic:1983';

            case 'csiso2intlrefversion':
            case 'irv':
            case 'iso646irv1983':
            case 'isoir2':
                return 'ISO_646.irv:1983';

            case 'csiso2033':
            case 'e13b':
            case 'iso20331983':
            case 'isoir98':
                return 'ISO_2033-1983';

            case 'csiso5427cyrillic':
            case 'iso5427':
            case 'isoir37':
                return 'ISO_5427';

            case 'iso5427cyrillic1981':
            case 'iso54271981':
            case 'isoir54':
                return 'ISO_5427:1981';

            case 'csiso5428greek':
            case 'iso54281980':
            case 'isoir55':
                return 'ISO_5428:1980';

            case 'csiso6937add':
            case 'iso6937225':
            case 'isoir152':
                return 'ISO_6937-2-25';

            case 'csisotextcomm':
            case 'iso69372add':
            case 'isoir142':
                return 'ISO_6937-2-add';

            case 'csiso8859supp':
            case 'iso8859supp':
            case 'isoir154':
            case 'latin125':
                return 'ISO_8859-supp';

            case 'csiso10367box':
            case 'iso10367box':
            case 'isoir155':
                return 'ISO_10367-box';

            case 'csiso15italian':
            case 'iso646it':
            case 'isoir15':
            case 'it':
                return 'IT';

            case 'csiso13jisc6220jp':
            case 'isoir13':
            case 'jisc62201969':
            case 'jisc62201969jp':
            case 'katakana':
            case 'x2017':
                return 'JIS_C6220-1969-jp';

            case 'csiso14jisc6220ro':
            case 'iso646jp':
            case 'isoir14':
            case 'jisc62201969ro':
            case 'jp':
                return 'JIS_C6220-1969-ro';

            case 'csiso42jisc62261978':
            case 'isoir42':
            case 'jisc62261978':
                return 'JIS_C6226-1978';

            case 'csiso87jisx208':
            case 'isoir87':
            case 'jisc62261983':
            case 'jisx2081983':
            case 'x208':
                return 'JIS_C6226-1983';

            case 'csiso91jisc62291984a':
            case 'isoir91':
            case 'jisc62291984a':
            case 'jpocra':
                return 'JIS_C6229-1984-a';

            case 'csiso92jisc62991984b':
            case 'iso646jpocrb':
            case 'isoir92':
            case 'jisc62291984b':
            case 'jpocrb':
                return 'JIS_C6229-1984-b';

            case 'csiso93jis62291984badd':
            case 'isoir93':
            case 'jisc62291984badd':
            case 'jpocrbadd':
                return 'JIS_C6229-1984-b-add';

            case 'csiso94jis62291984hand':
            case 'isoir94':
            case 'jisc62291984hand':
            case 'jpocrhand':
                return 'JIS_C6229-1984-hand';

            case 'csiso95jis62291984handadd':
            case 'isoir95':
            case 'jisc62291984handadd':
            case 'jpocrhandadd':
                return 'JIS_C6229-1984-hand-add';

            case 'csiso96jisc62291984kana':
            case 'isoir96':
            case 'jisc62291984kana':
                return 'JIS_C6229-1984-kana';

            case 'csjisencoding':
            case 'jisencoding':
                return 'JIS_Encoding';

            case 'cshalfwidthkatakana':
            case 'jisx201':
            case 'x201':
                return 'JIS_X0201';

            case 'csiso159jisx2121990':
            case 'isoir159':
            case 'jisx2121990':
            case 'x212':
                return 'JIS_X0212-1990';

            case 'csiso141jusib1002':
            case 'iso646yu':
            case 'isoir141':
            case 'js':
            case 'jusib1002':
            case 'yu':
                return 'JUS_I.B1.002';

            case 'csiso147macedonian':
            case 'isoir147':
            case 'jusib1003mac':
            case 'macedonian':
                return 'JUS_I.B1.003-mac';

            case 'csiso146serbian':
            case 'isoir146':
            case 'jusib1003serb':
            case 'serbian':
                return 'JUS_I.B1.003-serb';

            case 'koi7switched':
                return 'KOI7-switched';

            case 'cskoi8r':
            case 'koi8r':
                return 'KOI8-R';

            case 'koi8u':
                return 'KOI8-U';

            case 'csksc5636':
            case 'iso646kr':
            case 'ksc5636':
                return 'KSC5636';

            case 'cskz1048':
            case 'kz1048':
            case 'rk1048':
            case 'strk10482002':
                return 'KZ-1048';

            case 'csiso19latingreek':
            case 'isoir19':
            case 'latingreek':
                return 'latin-greek';

            case 'csiso27latingreek1':
            case 'isoir27':
            case 'latingreek1':
                return 'Latin-greek-1';

            case 'csiso158lap':
            case 'isoir158':
            case 'lap':
            case 'latinlap':
                return 'latin-lap';

            case 'csmacintosh':
            case 'mac':
            case 'macintosh':
                return 'macintosh';

            case 'csmicrosoftpublishing':
            case 'microsoftpublishing':
                return 'Microsoft-Publishing';

            case 'csmnem':
            case 'mnem':
                return 'MNEM';

            case 'csmnemonic':
            case 'mnemonic':
                return 'MNEMONIC';

            case 'csiso86hungarian':
            case 'hu':
            case 'iso646hu':
            case 'isoir86':
            case 'msz77953':
                return 'MSZ_7795.3';

            case 'csnatsdano':
            case 'isoir91':
            case 'natsdano':
                return 'NATS-DANO';

            case 'csnatsdanoadd':
            case 'isoir92':
            case 'natsdanoadd':
                return 'NATS-DANO-ADD';

            case 'csnatssefi':
            case 'isoir81':
            case 'natssefi':
                return 'NATS-SEFI';

            case 'csnatssefiadd':
            case 'isoir82':
            case 'natssefiadd':
                return 'NATS-SEFI-ADD';

            case 'csiso151cuba':
            case 'cuba':
            case 'iso646cu':
            case 'isoir151':
            case 'ncnc1081':
                return 'NC_NC00-10:81';

            case 'csiso69french':
            case 'fr':
            case 'iso646fr':
            case 'isoir69':
            case 'nfz62010':
                return 'NF_Z_62-010';

            case 'csiso25french':
            case 'iso646fr1':
            case 'isoir25':
            case 'nfz620101973':
                return 'NF_Z_62-010_(1973)';

            case 'csiso60danishnorwegian':
            case 'csiso60norwegian1':
            case 'iso646no':
            case 'isoir60':
            case 'no':
            case 'ns45511':
                return 'NS_4551-1';

            case 'csiso61norwegian2':
            case 'iso646no2':
            case 'isoir61':
            case 'no2':
            case 'ns45512':
                return 'NS_4551-2';

            case 'osdebcdicdf3irv':
                return 'OSD_EBCDIC_DF03_IRV';

            case 'osdebcdicdf41':
                return 'OSD_EBCDIC_DF04_1';

            case 'osdebcdicdf415':
                return 'OSD_EBCDIC_DF04_15';

            case 'cspc8danishnorwegian':
            case 'pc8danishnorwegian':
                return 'PC8-Danish-Norwegian';

            case 'cspc8turkish':
            case 'pc8turkish':
                return 'PC8-Turkish';

            case 'csiso16portuguese':
            case 'iso646pt':
            case 'isoir16':
            case 'pt':
                return 'PT';

            case 'csiso84portuguese2':
            case 'iso646pt2':
            case 'isoir84':
            case 'pt2':
                return 'PT2';

            case 'cp154':
            case 'csptcp154':
            case 'cyrillicasian':
            case 'pt154':
            case 'ptcp154':
                return 'PTCP154';

            case 'scsu':
                return 'SCSU';

            case 'csiso10swedish':
            case 'fi':
            case 'iso646fi':
            case 'iso646se':
            case 'isoir10':
            case 'se':
            case 'sen850200b':
                return 'SEN_850200_B';

            case 'csiso11swedishfornames':
            case 'iso646se2':
            case 'isoir11':
            case 'se2':
            case 'sen850200c':
                return 'SEN_850200_C';

            case 'csiso102t617bit':
            case 'isoir102':
            case 't617bit':
                return 'T.61-7bit';

            case 'csiso103t618bit':
            case 'isoir103':
            case 't61':
            case 't618bit':
                return 'T.61-8bit';

            case 'csiso128t101g2':
            case 'isoir128':
            case 't101g2':
                return 'T.101-G2';

            case 'cstscii':
            case 'tscii':
                return 'TSCII';

            case 'csunicode11':
            case 'unicode11':
                return 'UNICODE-1-1';

            case 'csunicode11utf7':
            case 'unicode11utf7':
                return 'UNICODE-1-1-UTF-7';

            case 'csunknown8bit':
            case 'unknown8bit':
                return 'UNKNOWN-8BIT';

            case 'ansix341968':
            case 'ansix341986':
            case 'ascii':
            case 'cp367':
            case 'csascii':
            case 'ibm367':
            case 'iso646irv1991':
            case 'iso646us':
            case 'isoir6':
            case 'us':
            case 'usascii':
                return 'US-ASCII';

            case 'csusdk':
            case 'usdk':
                return 'us-dk';

            case 'utf7':
                return 'UTF-7';

            case 'utf8':
                return 'UTF-8';

            case 'utf16':
                return 'UTF-16';

            case 'utf16be':
                return 'UTF-16BE';

            case 'utf16le':
                return 'UTF-16LE';

            case 'utf32':
                return 'UTF-32';

            case 'utf32be':
                return 'UTF-32BE';

            case 'utf32le':
                return 'UTF-32LE';

            case 'csventurainternational':
            case 'venturainternational':
                return 'Ventura-International';

            case 'csventuramath':
            case 'venturamath':
                return 'Ventura-Math';

            case 'csventuraus':
            case 'venturaus':
                return 'Ventura-US';

            case 'csiso70videotexsupp1':
            case 'isoir70':
            case 'videotexsuppl':
                return 'videotex-suppl';

            case 'csviqr':
            case 'viqr':
                return 'VIQR';

            case 'csviscii':
            case 'viscii':
                return 'VISCII';

            case 'csshiftjis':
            case 'cswindows31j':
            case 'mskanji':
            case 'shiftjis':
            case 'windows31j':
                return 'Windows-31J';

            case 'iso885911':
            case 'tis620':
                return 'windows-874';

            case 'cseuckr':
            case 'csksc56011987':
            case 'euckr':
            case 'isoir149':
            case 'korean':
            case 'ksc5601':
            case 'ksc56011987':
            case 'ksc56011989':
            case 'windows949':
                return 'windows-949';

            case 'windows1250':
                return 'windows-1250';

            case 'windows1251':
                return 'windows-1251';

            case 'cp819':
            case 'csisolatin1':
            case 'ibm819':
            case 'iso88591':
            case 'iso885911987':
            case 'isoir100':
            case 'l1':
            case 'latin1':
            case 'windows1252':
                return 'windows-1252';

            case 'windows1253':
                return 'windows-1253';

            case 'csisolatin5':
            case 'iso88599':
            case 'iso885991989':
            case 'isoir148':
            case 'l5':
            case 'latin5':
            case 'windows1254':
                return 'windows-1254';

            case 'windows1255':
                return 'windows-1255';

            case 'windows1256':
                return 'windows-1256';

            case 'windows1257':
                return 'windows-1257';

            case 'windows1258':
                return 'windows-1258';

            default:
                return $charset;
        }
    }

    /**
     * @return string
     */
    public static function get_curl_version()
    {
        if (is_array($curl = curl_version())) {
            $curl = $curl['version'];
        } else {
            $curl = '0';
        }
        return $curl;
    }

    /**
     * Strip HTML comments
     *
     * @deprecated since SimplePie 1.9.0. If you need it, you can copy the function to your codebase. But you should consider using `DOMDocument` for any DOM wrangling.
     * @param string $data Data to strip comments from
     * @return string Comment stripped string
     */
    public static function strip_comments(string $data)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.9.'), \E_USER_DEPRECATED);

        $output = '';
        while (($start = strpos($data, '<!--')) !== false) {
            $output .= substr($data, 0, $start);
            if (($end = strpos($data, '-->', $start)) !== false) {
                $data = substr_replace($data, '', 0, $end + 3);
            } else {
                $data = '';
            }
        }
        return $output . $data;
    }

    /**
     * @return int|false
     */
    public static function parse_date(string $dt)
    {
        $parser = \SimplePie\Parse\Date::get();
        return $parser->parse($dt);
    }

    /**
     * Decode HTML entities
     *
     * @deprecated since SimplePie 1.3, use DOMDocument instead
     * @param string $data Input data
     * @return string Output data
     */
    public static function entities_decode(string $data)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.3, use "DOMDocument" instead.'), \E_USER_DEPRECATED);

        $decoder = new \SimplePie_Decode_HTML_Entities($data);
        return $decoder->parse();
    }

    /**
     * Remove RFC822 comments
     *
     * @deprecated since SimplePie 1.9.0. If you need it, consider copying the function to your codebase.
     * @param string $string Data to strip comments from
     * @return string Comment stripped string
     */
    public static function uncomment_rfc822(string $string)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.9.'), \E_USER_DEPRECATED);

        $position = 0;
        $length = strlen($string);
        $depth = 0;

        $output = '';

        while ($position < $length && ($pos = strpos($string, '(', $position)) !== false) {
            $output .= substr($string, $position, $pos - $position);
            $position = $pos + 1;
            if ($string[$pos - 1] !== '\\') {
                $depth++;
                while ($depth && $position < $length) {
                    $position += strcspn($string, '()', $position);
                    if ($string[$position - 1] === '\\') {
                        $position++;
                        continue;
                    } elseif (isset($string[$position])) {
                        switch ($string[$position]) {
                            case '(':
                                $depth++;
                                break;

                            case ')':
                                $depth--;
                                break;
                        }
                        $position++;
                    } else {
                        break;
                    }
                }
            } else {
                $output .= '(';
            }
        }
        $output .= substr($string, $position);

        return $output;
    }

    /**
     * @return string
     */
    public static function parse_mime(string $mime)
    {
        if (($pos = strpos($mime, ';')) === false) {
            return trim($mime);
        }

        return trim(substr($mime, 0, $pos));
    }

    /**
     * @param array<string, array<string, string>> $attribs
     * @return int-mask-of<SimplePie::CONSTRUCT_*>
     */
    public static function atom_03_construct_type(array $attribs)
    {
        if (isset($attribs['']['mode']) && strtolower(trim($attribs['']['mode'])) === 'base64') {
            $mode = \SimplePie\SimplePie::CONSTRUCT_BASE64;
        } else {
            $mode = \SimplePie\SimplePie::CONSTRUCT_NONE;
        }
        if (isset($attribs['']['type'])) {
            switch (strtolower(trim($attribs['']['type']))) {
                case 'text':
                case 'text/plain':
                    return \SimplePie\SimplePie::CONSTRUCT_TEXT | $mode;

                case 'html':
                case 'text/html':
                    return \SimplePie\SimplePie::CONSTRUCT_HTML | $mode;

                case 'xhtml':
                case 'application/xhtml+xml':
                    return \SimplePie\SimplePie::CONSTRUCT_XHTML | $mode;

                default:
                    return \SimplePie\SimplePie::CONSTRUCT_NONE | $mode;
            }
        }

        return \SimplePie\SimplePie::CONSTRUCT_TEXT | $mode;
    }

    /**
     * @param array<string, array<string, string>> $attribs
     * @return int-mask-of<SimplePie::CONSTRUCT_*>
     */
    public static function atom_10_construct_type(array $attribs)
    {
        if (isset($attribs['']['type'])) {
            switch (strtolower(trim($attribs['']['type']))) {
                case 'text':
                    return \SimplePie\SimplePie::CONSTRUCT_TEXT;

                case 'html':
                    return \SimplePie\SimplePie::CONSTRUCT_HTML;

                case 'xhtml':
                    return \SimplePie\SimplePie::CONSTRUCT_XHTML;

                default:
                    return \SimplePie\SimplePie::CONSTRUCT_NONE;
            }
        }
        return \SimplePie\SimplePie::CONSTRUCT_TEXT;
    }

    /**
     * @param array<string, array<string, string>> $attribs
     * @return int-mask-of<SimplePie::CONSTRUCT_*>
     */
    public static function atom_10_content_construct_type(array $attribs)
    {
        if (isset($attribs['']['type'])) {
            $type = strtolower(trim($attribs['']['type']));
            switch ($type) {
                case 'text':
                    return \SimplePie\SimplePie::CONSTRUCT_TEXT;

                case 'html':
                    return \SimplePie\SimplePie::CONSTRUCT_HTML;

                case 'xhtml':
                    return \SimplePie\SimplePie::CONSTRUCT_XHTML;
            }
            if (in_array(substr($type, -4), ['+xml', '/xml']) || substr($type, 0, 5) === 'text/') {
                return \SimplePie\SimplePie::CONSTRUCT_NONE;
            } else {
                return \SimplePie\SimplePie::CONSTRUCT_BASE64;
            }
        }

        return \SimplePie\SimplePie::CONSTRUCT_TEXT;
    }

    /**
     * @return bool
     */
    public static function is_isegment_nz_nc(string $string)
    {
        return (bool) preg_match('/^([A-Za-z0-9\-._~\x{A0}-\x{D7FF}\x{F900}-\x{FDCF}\x{FDF0}-\x{FFEF}\x{10000}-\x{1FFFD}\x{20000}-\x{2FFFD}\x{30000}-\x{3FFFD}\x{40000}-\x{4FFFD}\x{50000}-\x{5FFFD}\x{60000}-\x{6FFFD}\x{70000}-\x{7FFFD}\x{80000}-\x{8FFFD}\x{90000}-\x{9FFFD}\x{A0000}-\x{AFFFD}\x{B0000}-\x{BFFFD}\x{C0000}-\x{CFFFD}\x{D0000}-\x{DFFFD}\x{E1000}-\x{EFFFD}!$&\'()*+,;=@]|(%[0-9ABCDEF]{2}))+$/u', $string);
    }

    /**
     * @return string[]
     */
    public static function space_separated_tokens(string $string)
    {
        $space_characters = "\x20\x09\x0A\x0B\x0C\x0D";
        $string_length = strlen($string);

        $position = strspn($string, $space_characters);
        $tokens = [];

        while ($position < $string_length) {
            $len = strcspn($string, $space_characters, $position);
            $tokens[] = substr($string, $position, $len);
            $position += $len;
            $position += strspn($string, $space_characters, $position);
        }

        return $tokens;
    }

    /**
     * Converts a unicode codepoint to a UTF-8 character
     *
     * @static
     * @param int $codepoint Unicode codepoint
     * @return string|false UTF-8 character
     */
    public static function codepoint_to_utf8(int $codepoint)
    {
        if ($codepoint < 0) {
            return false;
        } elseif ($codepoint <= 0x7f) {
            return chr($codepoint);
        } elseif ($codepoint <= 0x7ff) {
            return chr(0xc0 | ($codepoint >> 6)) . chr(0x80 | ($codepoint & 0x3f));
        } elseif ($codepoint <= 0xffff) {
            return chr(0xe0 | ($codepoint >> 12)) . chr(0x80 | (($codepoint >> 6) & 0x3f)) . chr(0x80 | ($codepoint & 0x3f));
        } elseif ($codepoint <= 0x10ffff) {
            return chr(0xf0 | ($codepoint >> 18)) . chr(0x80 | (($codepoint >> 12) & 0x3f)) . chr(0x80 | (($codepoint >> 6) & 0x3f)) . chr(0x80 | ($codepoint & 0x3f));
        }

        // U+FFFD REPLACEMENT CHARACTER
        return "\xEF\xBF\xBD";
    }

    /**
     * Similar to parse_str()
     *
     * Returns an associative array of name/value pairs, where the value is an
     * array of values that have used the same name
     *
     * @deprecated since SimplePie 1.9.0. If you need it, consider copying the function to your codebase.
     * @static
     * @param string $str The input string.
     * @return array<string, array<string|null>>
     */
    public static function parse_str(string $str)
    {
        // trigger_error(sprintf('Using method "' . __METHOD__ . '" is deprecated since SimplePie 1.9.'), \E_USER_DEPRECATED);

        $return = [];
        $str = explode('&', $str);

        foreach ($str as $section) {
            if (strpos($section, '=') !== false) {
                [$name, $value] = explode('=', $section, 2);
                $return[urldecode($name)][] = urldecode($value);
            } else {
                $return[urldecode($section)][] = null;
            }
        }

        return $return;
    }

    /**
     * Detect XML encoding, as per XML 1.0 Appendix F.1
     *
     * @todo Add support for EBCDIC
     * @param string $data XML data
     * @param \SimplePie\Registry $registry Class registry
     * @return array<string> Possible encodings
     */
    public static function xml_encoding(string $data, \SimplePie\Registry $registry)
    {
        // UTF-32 Big Endian BOM
        if (substr($data, 0, 4) === "\x00\x00\xFE\xFF") {
            $encoding[] = 'UTF-32BE';
        }
        // UTF-32 Little Endian BOM
        elseif (substr($data, 0, 4) === "\xFF\xFE\x00\x00") {
            $encoding[] = 'UTF-32LE';
        }
        // UTF-16 Big Endian BOM
        elseif (substr($data, 0, 2) === "\xFE\xFF") {
            $encoding[] = 'UTF-16BE';
        }
        // UTF-16 Little Endian BOM
        elseif (substr($data, 0, 2) === "\xFF\xFE") {
            $encoding[] = 'UTF-16LE';
        }
        // UTF-8 BOM
        elseif (substr($data, 0, 3) === "\xEF\xBB\xBF") {
            $encoding[] = 'UTF-8';
        }
        // UTF-32 Big Endian Without BOM
        elseif (substr($data, 0, 20) === "\x00\x00\x00\x3C\x00\x00\x00\x3F\x00\x00\x00\x78\x00\x00\x00\x6D\x00\x00\x00\x6C") {
            if ($pos = strpos($data, "\x00\x00\x00\x3F\x00\x00\x00\x3E")) {
                $parser = $registry->create(Parser::class, [Misc::change_encoding(substr($data, 20, $pos - 20), 'UTF-32BE', 'UTF-8')]);
                if ($parser->parse()) {
                    $encoding[] = $parser->encoding;
                }
            }
            $encoding[] = 'UTF-32BE';
        }
        // UTF-32 Little Endian Without BOM
        elseif (substr($data, 0, 20) === "\x3C\x00\x00\x00\x3F\x00\x00\x00\x78\x00\x00\x00\x6D\x00\x00\x00\x6C\x00\x00\x00") {
            if ($pos = strpos($data, "\x3F\x00\x00\x00\x3E\x00\x00\x00")) {
                $parser = $registry->create(Parser::class, [Misc::change_encoding(substr($data, 20, $pos - 20), 'UTF-32LE', 'UTF-8')]);
                if ($parser->parse()) {
                    $encoding[] = $parser->encoding;
                }
            }
            $encoding[] = 'UTF-32LE';
        }
        // UTF-16 Big Endian Without BOM
        elseif (substr($data, 0, 10) === "\x00\x3C\x00\x3F\x00\x78\x00\x6D\x00\x6C") {
            if ($pos = strpos($data, "\x00\x3F\x00\x3E")) {
                $parser = $registry->create(Parser::class, [Misc::change_encoding(substr($data, 20, $pos - 10), 'UTF-16BE', 'UTF-8')]);
                if ($parser->parse()) {
                    $encoding[] = $parser->encoding;
                }
            }
            $encoding[] = 'UTF-16BE';
        }
        // UTF-16 Little Endian Without BOM
        elseif (substr($data, 0, 10) === "\x3C\x00\x3F\x00\x78\x00\x6D\x00\x6C\x00") {
            if ($pos = strpos($data, "\x3F\x00\x3E\x00")) {
                $parser = $registry->create(Parser::class, [Misc::change_encoding(substr($data, 20, $pos - 10), 'UTF-16LE', 'UTF-8')]);
                if ($parser->parse()) {
                    $encoding[] = $parser->encoding;
                }
            }
            $encoding[] = 'UTF-16LE';
        }
        // US-ASCII (or superset)
        elseif (substr($data, 0, 5) === "\x3C\x3F\x78\x6D\x6C") {
            if ($pos = strpos($data, "\x3F\x3E")) {
                $parser = $registry->create(Parser::class, [substr($data, 5, $pos - 5)]);
                if ($parser->parse()) {
                    $encoding[] = $parser->encoding;
                }
            }
            $encoding[] = 'UTF-8';
        }
        // Fallback to UTF-8
        else {
            $encoding[] = 'UTF-8';
        }
        return $encoding;
    }

    /**
     * @return void
     */
    public static function output_javascript()
    {
        if (function_exists('ob_gzhandler')) {
            ob_start('ob_gzhandler');
        }
        header('Content-type: text/javascript; charset: UTF-8');
        header('Cache-Control: must-revalidate');
        header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 604800) . ' GMT'); // 7 days

        $body = <<<JS
function embed_quicktime(type, bgcolor, width, height, link, placeholder, loop) {
	if (placeholder != '') {
		document.writeln('<embed type="'+type+'" style="cursor:hand; cursor:pointer;" href="'+link+'" src="'+placeholder+'" width="'+width+'" height="'+height+'" autoplay="false" target="myself" controller="false" loop="'+loop+'" scale="aspect" bgcolor="'+bgcolor+'" pluginspage="http://www.apple.com/quicktime/download/"></embed>');
	}
	else {
		document.writeln('<embed type="'+type+'" style="cursor:hand; cursor:pointer;" src="'+link+'" width="'+width+'" height="'+height+'" autoplay="false" target="myself" controller="true" loop="'+loop+'" scale="aspect" bgcolor="'+bgcolor+'" pluginspage="http://www.apple.com/quicktime/download/"></embed>');
	}
}

function embed_flash(bgcolor, width, height, link, loop, type) {
	document.writeln('<embed src="'+link+'" pluginspage="http://www.macromedia.com/go/getflashplayer" type="'+type+'" quality="high" width="'+width+'" height="'+height+'" bgcolor="'+bgcolor+'" loop="'+loop+'"></embed>');
}

function embed_flv(width, height, link, placeholder, loop, player) {
	document.writeln('<embed src="'+player+'" pluginspage="http://www.macromedia.com/go/getflashplayer" type="application/x-shockwave-flash" quality="high" width="'+width+'" height="'+height+'" wmode="transparent" flashvars="file='+link+'&autostart=false&repeat='+loop+'&showdigits=true&showfsbutton=false"></embed>');
}

function embed_wmedia(width, height, link) {
	document.writeln('<embed type="application/x-mplayer2" src="'+link+'" autosize="1" width="'+width+'" height="'+height+'" showcontrols="1" showstatusbar="0" showdisplay="0" autostart="0"></embed>');
}
JS;
        echo $body;
    }

    /**
     * Get the SimplePie build timestamp
     *
     * Uses the git index if it exists, otherwise uses the modification time
     * of the newest file.
     *
     * @return int
     */
    public static function get_build()
    {
        if (self::$SIMPLEPIE_BUILD !== null) {
            return self::$SIMPLEPIE_BUILD;
        }

        $root = dirname(__FILE__, 2);
        if (file_exists($root . '/.git/index')) {
            self::$SIMPLEPIE_BUILD = (int) filemtime($root . '/.git/index');

            return self::$SIMPLEPIE_BUILD;
        } elseif (file_exists($root . '/src')) {
            $time = 0;
            foreach (glob($root . '/src/*.php') ?: [] as $file) {
                if (($mtime = filemtime($file)) > $time) {
                    $time = $mtime;
                }
            }

            self::$SIMPLEPIE_BUILD = $time;

            return self::$SIMPLEPIE_BUILD;
        }

        self::$SIMPLEPIE_BUILD = (int) filemtime(__FILE__);

        return self::$SIMPLEPIE_BUILD;
    }

    /**
     * Get the default user agent string
     *
     * @return string
     */
    public static function get_default_useragent()
    {
        return \SimplePie\SimplePie::NAME . '/' . \SimplePie\SimplePie::VERSION . ' (Feed Parser; ' . \SimplePie\SimplePie::URL . '; Allow like Gecko) Build/' . static::get_build();
    }

    /**
     * Format debugging information
     *
     * @return string
     */
    public static function debug(SimplePie &$sp)
    {
        $info = 'SimplePie ' . \SimplePie\SimplePie::VERSION . ' Build ' . static::get_build() . "\n";
        $info .= 'PHP ' . PHP_VERSION . "\n";
        if ($sp->error() !== null) {
            // TODO: Remove cast with multifeeds.
            $info .= 'Error occurred: ' . implode(', ', (array) $sp->error()) . "\n";
        } else {
            $info .= "No error found.\n";
        }
        $info .= "Extensions:\n";
        $extensions = ['pcre', 'curl', 'zlib', 'mbstring', 'iconv', 'xmlreader', 'xml'];
        foreach ($extensions as $ext) {
            if (extension_loaded($ext)) {
                $info .= "    $ext loaded\n";
                switch ($ext) {
                    case 'pcre':
                        $info .= '      Version ' . PCRE_VERSION . "\n";
                        break;
                    case 'curl':
                        $version = (array) curl_version();
                        $info .= '      Version ' . $version['version'] . "\n";
                        break;
                    case 'iconv':
                        $info .= '      Version ' . ICONV_VERSION . "\n";
                        break;
                    case 'xml':
                        $info .= '      Version ' . LIBXML_DOTTED_VERSION . "\n";
                        break;
                }
            } else {
                $info .= "    $ext not loaded\n";
            }
        }
        return $info;
    }

    /**
     * @return bool
     */
    public static function silence_errors(int $num, string $str)
    {
        // No-op
        return true;
    }

    /**
     * Sanitize a URL by removing HTTP credentials.
     * @param string $url the URL to sanitize.
     * @return string the same URL without HTTP credentials.
     */
    public static function url_remove_credentials(string $url)
    {
        // Cast for PHPStan: I do not think this can fail.
        // The regex is valid and there should be no backtracking.
        // https://github.com/phpstan/phpstan/issues/11547
        return (string) preg_replace('#^(https?://)[^/:@]+:[^/:@]+@#i', '$1', $url);
    }
}

class_alias('SimplePie\Misc', 'SimplePie_Misc', false);

}

namespace SimplePie\Net {

/**
 * Class to validate and to work with IPv6 addresses.
 *
 * @copyright 2003-2005 The PHP Group
 * @license http://www.opensource.org/licenses/bsd-license.php
 * @link http://pear.php.net/package/Net_IPv6
 * @author Alexander Merz <alexander.merz@web.de>
 * @author elfrink at introweb dot nl
 * @author Josh Peck <jmp at joshpeck dot org>
 * @author Sam Sneddon <geoffers@gmail.com>
 */
class IPv6
{
    /**
     * Uncompresses an IPv6 address
     *
     * RFC 4291 allows you to compress consecutive zero pieces in an address to
     * '::'. This method expects a valid IPv6 address and expands the '::' to
     * the required number of zero pieces.
     *
     * Example:  FF01::101   ->  FF01:0:0:0:0:0:0:101
     *           ::1         ->  0:0:0:0:0:0:0:1
     *
     * @author Alexander Merz <alexander.merz@web.de>
     * @author elfrink at introweb dot nl
     * @author Josh Peck <jmp at joshpeck dot org>
     * @copyright 2003-2005 The PHP Group
     * @license http://www.opensource.org/licenses/bsd-license.php
     * @param string $ip An IPv6 address
     * @return string The uncompressed IPv6 address
     */
    public static function uncompress(string $ip)
    {
        $c1 = -1;
        $c2 = -1;
        if (substr_count($ip, '::') === 1) {
            [$ip1, $ip2] = explode('::', $ip);
            if ($ip1 === '') {
                $c1 = -1;
            } else {
                $c1 = substr_count($ip1, ':');
            }
            if ($ip2 === '') {
                $c2 = -1;
            } else {
                $c2 = substr_count($ip2, ':');
            }
            if (strpos($ip2, '.') !== false) {
                $c2++;
            }
            // ::
            if ($c1 === -1 && $c2 === -1) {
                $ip = '0:0:0:0:0:0:0:0';
            }
            // ::xxx
            elseif ($c1 === -1) {
                $fill = str_repeat('0:', 7 - $c2);
                $ip = str_replace('::', $fill, $ip);
            }
            // xxx::
            elseif ($c2 === -1) {
                $fill = str_repeat(':0', 7 - $c1);
                $ip = str_replace('::', $fill, $ip);
            }
            // xxx::xxx
            else {
                $fill = ':' . str_repeat('0:', 6 - $c2 - $c1);
                $ip = str_replace('::', $fill, $ip);
            }
        }
        return $ip;
    }

    /**
     * Compresses an IPv6 address
     *
     * RFC 4291 allows you to compress consecutive zero pieces in an address to
     * '::'. This method expects a valid IPv6 address and compresses consecutive
     * zero pieces to '::'.
     *
     * Example:  FF01:0:0:0:0:0:0:101   ->  FF01::101
     *           0:0:0:0:0:0:0:1        ->  ::1
     *
     * @see uncompress()
     * @param string $ip An IPv6 address
     * @return string The compressed IPv6 address
     */
    public static function compress(string $ip)
    {
        // Prepare the IP to be compressed
        $ip = self::uncompress($ip);
        $ip_parts = self::split_v6_v4($ip);

        // Replace all leading zeros
        $ip_parts[0] = (string) preg_replace('/(^|:)0+([0-9])/', '\1\2', $ip_parts[0]);

        // Find bunches of zeros
        if (preg_match_all('/(?:^|:)(?:0(?::|$))+/', $ip_parts[0], $matches, PREG_OFFSET_CAPTURE)) {
            $max = 0;
            $pos = null;
            foreach ($matches[0] as $match) {
                if (strlen($match[0]) > $max) {
                    $max = strlen($match[0]);
                    $pos = $match[1];
                }
            }

            assert($pos !== null, 'For PHPStan: Since the regex matched, there is at least one match. And because the pattern is non-empty, the loop will always end with $pos ≥ 1.');
            $ip_parts[0] = substr_replace($ip_parts[0], '::', $pos, $max);
        }

        if ($ip_parts[1] !== '') {
            return implode(':', $ip_parts);
        }

        return $ip_parts[0];
    }

    /**
     * Splits an IPv6 address into the IPv6 and IPv4 representation parts
     *
     * RFC 4291 allows you to represent the last two parts of an IPv6 address
     * using the standard IPv4 representation
     *
     * Example:  0:0:0:0:0:0:13.1.68.3
     *           0:0:0:0:0:FFFF:129.144.52.38
     *
     * @param string $ip An IPv6 address
     * @return array{string, string} [0] contains the IPv6 represented part, and [1] the IPv4 represented part
     */
    private static function split_v6_v4(string $ip): array
    {
        if (strpos($ip, '.') !== false) {
            $pos = strrpos($ip, ':');
            assert($pos !== false, 'For PHPStan: IPv6 address must contain colon, since split_v6_v4 is only ever called after uncompress.');
            $ipv6_part = substr($ip, 0, $pos);
            $ipv4_part = substr($ip, $pos + 1);
            return [$ipv6_part, $ipv4_part];
        }

        return [$ip, ''];
    }

    /**
     * Checks an IPv6 address
     *
     * Checks if the given IP is a valid IPv6 address
     *
     * @param string $ip An IPv6 address
     * @return bool true if $ip is a valid IPv6 address
     */
    public static function check_ipv6(string $ip)
    {
        $ip = self::uncompress($ip);
        [$ipv6, $ipv4] = self::split_v6_v4($ip);
        $ipv6 = explode(':', $ipv6);
        $ipv4 = explode('.', $ipv4);
        if (count($ipv6) === 8 && count($ipv4) === 1 || count($ipv6) === 6 && count($ipv4) === 4) {
            foreach ($ipv6 as $ipv6_part) {
                // The section can't be empty
                if ($ipv6_part === '') {
                    return false;
                }

                // Nor can it be over four characters
                if (strlen($ipv6_part) > 4) {
                    return false;
                }

                // Remove leading zeros (this is safe because of the above)
                $ipv6_part = ltrim($ipv6_part, '0');
                if ($ipv6_part === '') {
                    $ipv6_part = '0';
                }

                // Check the value is valid
                $value = hexdec($ipv6_part);
                if ($value < 0 || $value > 0xFFFF) {
                    return false;
                }
                assert(is_int($value), 'For PHPStan: $value is only float when $ipv6_part > PHP_INT_MAX');
                if (dechex($value) !== strtolower($ipv6_part)) {
                    return false;
                }
            }
            if (count($ipv4) === 4) {
                foreach ($ipv4 as $ipv4_part) {
                    $value = (int) $ipv4_part;
                    if ((string) $value !== $ipv4_part || $value < 0 || $value > 0xFF) {
                        return false;
                    }
                }
            }
            return true;
        }

        return false;
    }

    /**
     * Checks if the given IP is a valid IPv6 address
     *
     * @codeCoverageIgnore
     * @deprecated Use {@see IPv6::check_ipv6()} instead
     * @see check_ipv6
     * @param string $ip An IPv6 address
     * @return bool true if $ip is a valid IPv6 address
     */
    public static function checkIPv6(string $ip)
    {
        return self::check_ipv6($ip);
    }
}

class_alias('SimplePie\Net\IPv6', 'SimplePie_Net_IPv6');

}

namespace SimplePie\Parse {

/**
 * Date Parser
 */
class Date
{
    /**
     * Input data
     *
     * @access protected
     * @var string
     */
    public $date;

    /**
     * List of days, calendar day name => ordinal day number in the week
     *
     * @access protected
     * @var array<string, int<1,7>>
     */
    public $day = [
        // English
        'mon' => 1,
        'monday' => 1,
        'tue' => 2,
        'tuesday' => 2,
        'wed' => 3,
        'wednesday' => 3,
        'thu' => 4,
        'thursday' => 4,
        'fri' => 5,
        'friday' => 5,
        'sat' => 6,
        'saturday' => 6,
        'sun' => 7,
        'sunday' => 7,
        // Dutch
        'maandag' => 1,
        'dinsdag' => 2,
        'woensdag' => 3,
        'donderdag' => 4,
        'vrijdag' => 5,
        'zaterdag' => 6,
        'zondag' => 7,
        // French
        'lundi' => 1,
        'mardi' => 2,
        'mercredi' => 3,
        'jeudi' => 4,
        'vendredi' => 5,
        'samedi' => 6,
        'dimanche' => 7,
        // German
        'montag' => 1,
        'mo' => 1,
        'dienstag' => 2,
        'di' => 2,
        'mittwoch' => 3,
        'mi' => 3,
        'donnerstag' => 4,
        'do' => 4,
        'freitag' => 5,
        'fr' => 5,
        'samstag' => 6,
        'sa' => 6,
        'sonnabend' => 6,
        // AFAIK no short form for sonnabend
        'so' => 7,
        'sonntag' => 7,
        // Italian
        'lunedì' => 1,
        'martedì' => 2,
        'mercoledì' => 3,
        'giovedì' => 4,
        'venerdì' => 5,
        'sabato' => 6,
        'domenica' => 7,
        // Spanish
        'lunes' => 1,
        'martes' => 2,
        'miércoles' => 3,
        'jueves' => 4,
        'viernes' => 5,
        'sábado' => 6,
        'domingo' => 7,
        // Finnish
        'maanantai' => 1,
        'tiistai' => 2,
        'keskiviikko' => 3,
        'torstai' => 4,
        'perjantai' => 5,
        'lauantai' => 6,
        'sunnuntai' => 7,
        // Hungarian
        'hétfő' => 1,
        'kedd' => 2,
        'szerda' => 3,
        'csütörtok' => 4,
        'péntek' => 5,
        'szombat' => 6,
        'vasárnap' => 7,
        // Greek
        'Δευ' => 1,
        'Τρι' => 2,
        'Τετ' => 3,
        'Πεμ' => 4,
        'Παρ' => 5,
        'Σαβ' => 6,
        'Κυρ' => 7,
        // Russian
        'Пн.' => 1,
        'Вт.' => 2,
        'Ср.' => 3,
        'Чт.' => 4,
        'Пт.' => 5,
        'Сб.' => 6,
        'Вс.' => 7,
    ];

    /**
     * List of months, calendar month name => calendar month number
     *
     * @access protected
     * @var array<string, int<1,12>>
     */
    public $month = [
        // English
        'jan' => 1,
        'january' => 1,
        'feb' => 2,
        'february' => 2,
        'mar' => 3,
        'march' => 3,
        'apr' => 4,
        'april' => 4,
        'may' => 5,
        // No long form of May
        'jun' => 6,
        'june' => 6,
        'jul' => 7,
        'july' => 7,
        'aug' => 8,
        'august' => 8,
        'sep' => 9,
        'september' => 9,
        'oct' => 10,
        'october' => 10,
        'nov' => 11,
        'november' => 11,
        'dec' => 12,
        'december' => 12,
        // Dutch
        'januari' => 1,
        'februari' => 2,
        'maart' => 3,
        // 'april' => 4,
        'mei' => 5,
        'juni' => 6,
        'juli' => 7,
        'augustus' => 8,
        // 'september' => 9,
        'oktober' => 10,
        // 'november' => 11,
        // 'december' => 12,
        // French
        'janvier' => 1,
        'février' => 2,
        'mars' => 3,
        'avril' => 4,
        'mai' => 5,
        'juin' => 6,
        'juillet' => 7,
        'août' => 8,
        'septembre' => 9,
        'octobre' => 10,
        'novembre' => 11,
        'décembre' => 12,
        // German
        'januar' => 1,
        // 'jan' => 1,
        'februar' => 2,
        // 'feb' => 2,
        'märz' => 3,
        'mär' => 3,
        // 'april' => 4,
        // 'apr' => 4,
        // 'mai' => 5, // no short form for may
        // 'juni' => 6,
        // 'jun' => 6,
        // 'juli' => 7,
        // 'jul' => 7,
        // 'august' => 8,
        // 'aug' => 8,
        // 'september' => 9,
        // 'sep' => 9,
        // 'oktober' => 10,
        'okt' => 10,
        // 'november' => 11,
        // 'nov' => 11,
        'dezember' => 12,
        'dez' => 12,
        // Italian
        'gennaio' => 1,
        'febbraio' => 2,
        'marzo' => 3,
        'aprile' => 4,
        'maggio' => 5,
        'giugno' => 6,
        'luglio' => 7,
        'agosto' => 8,
        'settembre' => 9,
        'ottobre' => 10,
        // 'novembre' => 11,
        'dicembre' => 12,
        // Spanish
        'enero' => 1,
        'febrero' => 2,
        // 'marzo' => 3,
        'abril' => 4,
        'mayo' => 5,
        'junio' => 6,
        'julio' => 7,
        // 'agosto' => 8,
        'septiembre' => 9,
        'setiembre' => 9,
        'octubre' => 10,
        'noviembre' => 11,
        'diciembre' => 12,
        // Finnish
        'tammikuu' => 1,
        'helmikuu' => 2,
        'maaliskuu' => 3,
        'huhtikuu' => 4,
        'toukokuu' => 5,
        'kesäkuu' => 6,
        'heinäkuu' => 7,
        'elokuu' => 8,
        'suuskuu' => 9,
        'lokakuu' => 10,
        'marras' => 11,
        'joulukuu' => 12,
        // Hungarian
        'január' => 1,
        'február' => 2,
        'március' => 3,
        'április' => 4,
        'május' => 5,
        'június' => 6,
        'július' => 7,
        'augusztus' => 8,
        'szeptember' => 9,
        'október' => 10,
        // 'november' => 11,
        // 'december' => 12,
        // Greek
        'Ιαν' => 1,
        'Φεβ' => 2,
        'Μάώ' => 3,
        'Μαώ' => 3,
        'Απρ' => 4,
        'Μάι' => 5,
        'Μαϊ' => 5,
        'Μαι' => 5,
        'Ιούν' => 6,
        'Ιον' => 6,
        'Ιούλ' => 7,
        'Ιολ' => 7,
        'Αύγ' => 8,
        'Αυγ' => 8,
        'Σεπ' => 9,
        'Οκτ' => 10,
        'Νοέ' => 11,
        'Δεκ' => 12,
        // Russian
        'Янв' => 1,
        'января' => 1,
        'Фев' => 2,
        'февраля' => 2,
        'Мар' => 3,
        'марта' => 3,
        'Апр' => 4,
        'апреля' => 4,
        'Май' => 5,
        'мая' => 5,
        'Июн' => 6,
        'июня' => 6,
        'Июл' => 7,
        'июля' => 7,
        'Авг' => 8,
        'августа' => 8,
        'Сен' => 9,
        'сентября' => 9,
        'Окт' => 10,
        'октября' => 10,
        'Ноя' => 11,
        'ноября' => 11,
        'Дек' => 12,
        'декабря' => 12,

    ];

    /**
     * List of timezones, abbreviation => offset from UTC
     *
     * @access protected
     * @var array<string, int>
     */
    public $timezone = [
        'ACDT' => 37800,
        'ACIT' => 28800,
        'ACST' => 34200,
        'ACT' => -18000,
        'ACWDT' => 35100,
        'ACWST' => 31500,
        'AEDT' => 39600,
        'AEST' => 36000,
        'AFT' => 16200,
        'AKDT' => -28800,
        'AKST' => -32400,
        'AMDT' => 18000,
        'AMT' => -14400,
        'ANAST' => 46800,
        'ANAT' => 43200,
        'ART' => -10800,
        'AZOST' => -3600,
        'AZST' => 18000,
        'AZT' => 14400,
        'BIOT' => 21600,
        'BIT' => -43200,
        'BOT' => -14400,
        'BRST' => -7200,
        'BRT' => -10800,
        'BST' => 3600,
        'BTT' => 21600,
        'CAST' => 18000,
        'CAT' => 7200,
        'CCT' => 23400,
        'CDT' => -18000,
        'CEDT' => 7200,
        'CEST' => 7200,
        'CET' => 3600,
        'CGST' => -7200,
        'CGT' => -10800,
        'CHADT' => 49500,
        'CHAST' => 45900,
        'CIST' => -28800,
        'CKT' => -36000,
        'CLDT' => -10800,
        'CLST' => -14400,
        'COT' => -18000,
        'CST' => -21600,
        'CVT' => -3600,
        'CXT' => 25200,
        'DAVT' => 25200,
        'DTAT' => 36000,
        'EADT' => -18000,
        'EAST' => -21600,
        'EAT' => 10800,
        'ECT' => -18000,
        'EDT' => -14400,
        'EEST' => 10800,
        'EET' => 7200,
        'EGT' => -3600,
        'EKST' => 21600,
        'EST' => -18000,
        'FJT' => 43200,
        'FKDT' => -10800,
        'FKST' => -14400,
        'FNT' => -7200,
        'GALT' => -21600,
        'GEDT' => 14400,
        'GEST' => 10800,
        'GFT' => -10800,
        'GILT' => 43200,
        'GIT' => -32400,
        'GST' => 14400,
        // 'GST' => -7200,
        'GYT' => -14400,
        'HAA' => -10800,
        'HAC' => -18000,
        'HADT' => -32400,
        'HAE' => -14400,
        'HAP' => -25200,
        'HAR' => -21600,
        'HAST' => -36000,
        'HAT' => -9000,
        'HAY' => -28800,
        'HKST' => 28800,
        'HMT' => 18000,
        'HNA' => -14400,
        'HNC' => -21600,
        'HNE' => -18000,
        'HNP' => -28800,
        'HNR' => -25200,
        'HNT' => -12600,
        'HNY' => -32400,
        'IRDT' => 16200,
        'IRKST' => 32400,
        'IRKT' => 28800,
        'IRST' => 12600,
        'JFDT' => -10800,
        'JFST' => -14400,
        'JST' => 32400,
        'KGST' => 21600,
        'KGT' => 18000,
        'KOST' => 39600,
        'KOVST' => 28800,
        'KOVT' => 25200,
        'KRAST' => 28800,
        'KRAT' => 25200,
        'KST' => 32400,
        'LHDT' => 39600,
        'LHST' => 37800,
        'LINT' => 50400,
        'LKT' => 21600,
        'MAGST' => 43200,
        'MAGT' => 39600,
        'MAWT' => 21600,
        'MDT' => -21600,
        'MESZ' => 7200,
        'MEZ' => 3600,
        'MHT' => 43200,
        'MIT' => -34200,
        'MNST' => 32400,
        'MSDT' => 14400,
        'MSST' => 10800,
        'MST' => -25200,
        'MUT' => 14400,
        'MVT' => 18000,
        'MYT' => 28800,
        'NCT' => 39600,
        'NDT' => -9000,
        'NFT' => 41400,
        'NMIT' => 36000,
        'NOVST' => 25200,
        'NOVT' => 21600,
        'NPT' => 20700,
        'NRT' => 43200,
        'NST' => -12600,
        'NUT' => -39600,
        'NZDT' => 46800,
        'NZST' => 43200,
        'OMSST' => 25200,
        'OMST' => 21600,
        'PDT' => -25200,
        'PET' => -18000,
        'PETST' => 46800,
        'PETT' => 43200,
        'PGT' => 36000,
        'PHOT' => 46800,
        'PHT' => 28800,
        'PKT' => 18000,
        'PMDT' => -7200,
        'PMST' => -10800,
        'PONT' => 39600,
        'PST' => -28800,
        'PWT' => 32400,
        'PYST' => -10800,
        'PYT' => -14400,
        'RET' => 14400,
        'ROTT' => -10800,
        'SAMST' => 18000,
        'SAMT' => 14400,
        'SAST' => 7200,
        'SBT' => 39600,
        'SCDT' => 46800,
        'SCST' => 43200,
        'SCT' => 14400,
        'SEST' => 3600,
        'SGT' => 28800,
        'SIT' => 28800,
        'SRT' => -10800,
        'SST' => -39600,
        'SYST' => 10800,
        'SYT' => 7200,
        'TFT' => 18000,
        'THAT' => -36000,
        'TJT' => 18000,
        'TKT' => -36000,
        'TMT' => 18000,
        'TOT' => 46800,
        'TPT' => 32400,
        'TRUT' => 36000,
        'TVT' => 43200,
        'TWT' => 28800,
        'UYST' => -7200,
        'UYT' => -10800,
        'UZT' => 18000,
        'VET' => -14400,
        'VLAST' => 39600,
        'VLAT' => 36000,
        'VOST' => 21600,
        'VUT' => 39600,
        'WAST' => 7200,
        'WAT' => 3600,
        'WDT' => 32400,
        'WEST' => 3600,
        'WFT' => 43200,
        'WIB' => 25200,
        'WIT' => 32400,
        'WITA' => 28800,
        'WKST' => 18000,
        'WST' => 28800,
        'YAKST' => 36000,
        'YAKT' => 32400,
        'YAPT' => 36000,
        'YEKST' => 21600,
        'YEKT' => 18000,
    ];

    /**
     * Cached PCRE for Date::$day
     *
     * @access protected
     * @var string
     */
    public $day_pcre;

    /**
     * Cached PCRE for Date::$month
     *
     * @access protected
     * @var string
     */
    public $month_pcre;

    /**
     * Array of user-added callback methods
     *
     * @access private
     * @var array<string>
     */
    public $built_in = [];

    /**
     * Array of user-added callback methods
     *
     * @access private
     * @var array<callable(string): (int|false)>
     */
    public $user = [];

    /**
     * Create new Date object, and set self::day_pcre,
     * self::month_pcre, and self::built_in
     *
     * @access private
     */
    public function __construct()
    {
        $this->day_pcre = '(' . implode('|', array_keys($this->day)) . ')';
        $this->month_pcre = '(' . implode('|', array_keys($this->month)) . ')';

        static $cache;
        if (!isset($cache[get_class($this)])) {
            $all_methods = get_class_methods($this);

            foreach ($all_methods as $method) {
                if (strtolower(substr($method, 0, 5)) === 'date_') {
                    $cache[get_class($this)][] = $method;
                }
            }
        }

        foreach ($cache[get_class($this)] as $method) {
            $this->built_in[] = $method;
        }
    }

    /**
     * Get the object
     *
     * @access public
     * @return Date
     */
    public static function get()
    {
        static $object;
        if (!$object) {
            $object = new Date();
        }
        return $object;
    }

    /**
     * Parse a date
     *
     * @final
     * @access public
     * @param string $date Date to parse
     * @return int|false Timestamp corresponding to date string, or false on failure
     */
    public function parse(string $date)
    {
        foreach ($this->user as $method) {
            if (($returned = call_user_func($method, $date)) !== false) {
                return (int) $returned;
            }
        }

        foreach ($this->built_in as $method) {
            // TODO: we should really check this in constructor but that would require private properties.
            /** @var callable(string): (int|false) */
            $callable = [$this, $method];
            if (($returned = call_user_func($callable, $date)) !== false) {
                return $returned;
            }
        }

        return false;
    }

    /**
     * Add a callback method to parse a date
     *
     * @final
     * @access public
     * @param callable $callback
     * @return void
     */
    public function add_callback(callable $callback)
    {
        $this->user[] = $callback;
    }

    /**
     * Parse a superset of W3C-DTF (allows hyphens and colons to be omitted, as
     * well as allowing any of upper or lower case "T", horizontal tabs, or
     * spaces to be used as the time separator (including more than one))
     *
     * @access protected
     * @param string $date
     * @return int|false Timestamp
     */
    public function date_w3cdtf(string $date)
    {
        $pcre = <<<'PCRE'
            /
            ^
            (?P<year>[0-9]{4})
            (?:
                -?
                (?P<month>[0-9]{2})
                (?:
                    -?
                    (?P<day>[0-9]{2})
                    (?:
                        [Tt\x09\x20]+
                        (?P<hour>[0-9]{2})
                        (?:
                            :?
                            (?P<minute>[0-9]{2})
                            (?:
                                :?
                                (?P<second>[0-9]{2})
                                (?:
                                    .
                                    (?P<second_fraction>[0-9]*)
                                )?
                            )?
                        )?
                        (?:
                            (?P<zulu>Z)
                            |   (?P<tz_sign>[+\-])
                                (?P<tz_hour>[0-9]{1,2})
                                :?
                                (?P<tz_minute>[0-9]{1,2})
                        )
                    )?
                )?
            )?
            $
            /x
PCRE;
        if (preg_match($pcre, $date, $match)) {
            // Fill in empty matches and convert to proper types.
            $year = (int) $match['year'];
            $month = isset($match['month']) ? (int) $match['month'] : 1;
            $day = isset($match['day']) ? (int) $match['day'] : 1;
            $hour = isset($match['hour']) ? (int) $match['hour'] : 0;
            $minute = isset($match['minute']) ? (int) $match['minute'] : 0;
            $second = isset($match['second']) ? (int) $match['second'] : 0;
            $second_fraction = isset($match['second_fraction']) ? ((int) $match['second_fraction']) / (10 ** strlen($match['second_fraction'])) : 0;
            $tz_sign = ($match['tz_sign'] ?? '') === '-' ? -1 : 1;
            $tz_hour = isset($match['tz_hour']) ? (int) $match['tz_hour'] : 0;
            $tz_minute = isset($match['tz_minute']) ? (int) $match['tz_minute'] : 0;

            // Numeric timezone
            $timezone = $tz_hour * 3600;
            $timezone += $tz_minute * 60;
            $timezone *= $tz_sign;

            // Convert the number of seconds to an integer, taking decimals into account
            $second = (int) round($second + $second_fraction);

            return gmmktime($hour, $minute, $second, $month, $day, $year) - $timezone;
        }

        return false;
    }

    /**
     * Remove RFC822 comments
     *
     * @access protected
     * @param string $string Data to strip comments from
     * @return string Comment stripped string
     */
    public function remove_rfc2822_comments(string $string)
    {
        $position = 0;
        $length = strlen($string);
        $depth = 0;

        $output = '';

        while ($position < $length && ($pos = strpos($string, '(', $position)) !== false) {
            $output .= substr($string, $position, $pos - $position);
            $position = $pos + 1;
            if ($pos === 0 || $string[$pos - 1] !== '\\') {
                $depth++;
                while ($depth && $position < $length) {
                    $position += strcspn($string, '()', $position);
                    if ($string[$position - 1] === '\\') {
                        $position++;
                        continue;
                    } elseif (isset($string[$position])) {
                        switch ($string[$position]) {
                            case '(':
                                $depth++;
                                break;

                            case ')':
                                $depth--;
                                break;
                        }
                        $position++;
                    } else {
                        break;
                    }
                }
            } else {
                $output .= '(';
            }
        }
        $output .= substr($string, $position);

        return $output;
    }

    /**
     * Parse RFC2822's date format
     *
     * @access protected
     * @param string $date
     * @return int|false Timestamp
     */
    public function date_rfc2822(string $date)
    {
        static $pcre;
        if (!$pcre) {
            $wsp = '[\x09\x20]';
            $fws = '(?:' . $wsp . '+|' . $wsp . '*(?:\x0D\x0A' . $wsp . '+)+)';
            $optional_fws = $fws . '?';
            $day_name = $this->day_pcre;
            $month = $this->month_pcre;
            $day = '([0-9]{1,2})';
            $hour = $minute = $second = '([0-9]{2})';
            $year = '([0-9]{2,4})';
            $num_zone = '([+\-])([0-9]{2})([0-9]{2})';
            $character_zone = '([A-Z]{1,5})';
            $zone = '(?:' . $num_zone . '|' . $character_zone . ')';
            $pcre = '/(?:' . $optional_fws . $day_name . $optional_fws . ',)?' . $optional_fws . $day . $fws . $month . $fws . $year . $fws . $hour . $optional_fws . ':' . $optional_fws . $minute . '(?:' . $optional_fws . ':' . $optional_fws . $second . ')?' . $fws . $zone . '/i';
        }
        if (preg_match($pcre, $this->remove_rfc2822_comments($date), $match)) {
            /*
            Capturing subpatterns:
            1: Day name
            2: Day
            3: Month
            4: Year
            5: Hour
            6: Minute
            7: Second
            8: Timezone ±
            9: Timezone hours
            10: Timezone minutes
            11: Alphabetic timezone
            */

            $day = (int) $match[2];
            // Find the month number
            $month = $this->month[strtolower($match[3])];
            $year = (int) $match[4];
            $hour = (int) $match[5];
            $minute = (int) $match[6];
            // Second is optional, if it is empty set it to zero
            $second = (int) $match[7];

            $tz_sign = $match[8];
            $tz_hour = (int) $match[9];
            $tz_minute = (int) $match[10];
            $tz_code = isset($match[11]) ? strtoupper($match[11]) : '';

            // Numeric timezone
            if ($tz_sign !== '') {
                $timezone = $tz_hour * 3600;
                $timezone += $tz_minute * 60;
                if ($tz_sign === '-') {
                    $timezone = 0 - $timezone;
                }
            }
            // Character timezone
            elseif (isset($this->timezone[$tz_code])) {
                $timezone = $this->timezone[$tz_code];
            }
            // Assume everything else to be -0000
            else {
                $timezone = 0;
            }

            // Deal with 2/3 digit years
            if ($year < 50) {
                $year += 2000;
            } elseif ($year < 1000) {
                $year += 1900;
            }

            return gmmktime($hour, $minute, $second, $month, $day, $year) - $timezone;
        }

        return false;
    }

    /**
     * Parse RFC850's date format
     *
     * @access protected
     * @param string $date
     * @return int|false Timestamp
     */
    public function date_rfc850(string $date)
    {
        static $pcre;
        if (!$pcre) {
            $space = '[\x09\x20]+';
            $day_name = $this->day_pcre;
            $month = $this->month_pcre;
            $day = '([0-9]{1,2})';
            $year = $hour = $minute = $second = '([0-9]{2})';
            $zone = '([A-Z]{1,5})';
            $pcre = '/^' . $day_name . ',' . $space . $day . '-' . $month . '-' . $year . $space . $hour . ':' . $minute . ':' . $second . $space . $zone . '$/i';
        }
        if (preg_match($pcre, $date, $match)) {
            /*
            Capturing subpatterns:
            1: Day name
            2: Day
            3: Month
            4: Year
            5: Hour
            6: Minute
            7: Second
            8: Timezone
            */

            $day = (int) $match[2];
            // Month
            $month = $this->month[strtolower($match[3])];
            $year = (int) $match[4];
            $hour = (int) $match[5];
            $minute = (int) $match[6];
            // Second is optional, if it is empty set it to zero
            $second = (int) $match[7];

            $tz_code = strtoupper($match[8]);

            // Character timezone
            if (isset($this->timezone[$tz_code])) {
                $timezone = $this->timezone[$tz_code];
            }
            // Assume everything else to be -0000
            else {
                $timezone = 0;
            }

            // Deal with 2 digit year
            if ($year < 50) {
                $year += 2000;
            } else {
                $year += 1900;
            }

            return gmmktime($hour, $minute, $second, $month, $day, $year) - $timezone;
        }

        return false;
    }

    /**
     * Parse C99's asctime()'s date format
     *
     * @access protected
     * @param string $date
     * @return int|false Timestamp
     */
    public function date_asctime(string $date)
    {
        static $pcre;
        if (!$pcre) {
            $space = '[\x09\x20]+';
            $wday_name = $this->day_pcre;
            $mon_name = $this->month_pcre;
            $day = '([0-9]{1,2})';
            $hour = $sec = $min = '([0-9]{2})';
            $year = '([0-9]{4})';
            $terminator = '\x0A?\x00?';
            $pcre = '/^' . $wday_name . $space . $mon_name . $space . $day . $space . $hour . ':' . $min . ':' . $sec . $space . $year . $terminator . '$/i';
        }
        if (preg_match($pcre, $date, $match)) {
            /*
            Capturing subpatterns:
            1: Day name
            2: Month
            3: Day
            4: Hour
            5: Minute
            6: Second
            7: Year
            */

            $month = $this->month[strtolower($match[2])];
            return gmmktime((int) $match[4], (int) $match[5], (int) $match[6], $month, (int) $match[3], (int) $match[7]);
        }

        return false;
    }

    /**
     * Parse dates using strtotime()
     *
     * @access protected
     * @param string $date
     * @return int|false Timestamp
     */
    public function date_strtotime(string $date)
    {
        $strtotime = strtotime($date);
        if ($strtotime === -1 || $strtotime === false) {
            return false;
        }

        return $strtotime;
    }
}

class_alias('SimplePie\Parse\Date', 'SimplePie_Parse_Date');

}

namespace SimplePie {

use SimplePie\XML\Declaration\Parser as DeclarationParser;
use XMLParser;

/**
 * Parses XML into something sane
 *
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_parser_class()}
 */
class Parser implements RegistryAware
{
    /** @var int */
    public $error_code;
    /** @var string */
    public $error_string;
    /** @var int */
    public $current_line;
    /** @var int */
    public $current_column;
    /** @var int */
    public $current_byte;
    /** @var string */
    public $separator = ' ';
    /** @var string[] */
    public $namespace = [''];
    /** @var string[] */
    public $element = [''];
    /** @var string[] */
    public $xml_base = [''];
    /** @var bool[] */
    public $xml_base_explicit = [false];
    /** @var string[] */
    public $xml_lang = [''];
    /** @var array<string, mixed> */
    public $data = [];
    /** @var array<array<string, mixed>> */
    public $datas = [[]];
    /** @var int */
    public $current_xhtml_construct = -1;
    /** @var string */
    public $encoding;
    /** @var Registry */
    protected $registry;

    /**
     * @return void
     */
    public function set_registry(\SimplePie\Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * @return bool
     */
    public function parse(string &$data, string $encoding, string $url = '')
    {
        if (class_exists('DOMXpath') && function_exists('Mf2\parse')) {
            $doc = new \DOMDocument();
            @$doc->loadHTML($data);
            $xpath = new \DOMXpath($doc);
            // Check for both h-feed and h-entry, as both a feed with no entries
            // and a list of entries without an h-feed wrapper are both valid.
            $query = '//*[contains(concat(" ", @class, " "), " h-feed ") or '.
                'contains(concat(" ", @class, " "), " h-entry ")]';
            /** @var \DOMNodeList<\DOMElement> $result */
            $result = $xpath->query($query);
            if ($result->length !== 0) {
                return $this->parse_microformats($data, $url);
            }
        }

        // Use UTF-8 if we get passed US-ASCII, as every US-ASCII character is a UTF-8 character
        if (strtoupper($encoding) === 'US-ASCII') {
            $this->encoding = 'UTF-8';
        } else {
            $this->encoding = $encoding;
        }

        // Strip BOM:
        // UTF-32 Big Endian BOM
        if (substr($data, 0, 4) === "\x00\x00\xFE\xFF") {
            $data = substr($data, 4);
        }
        // UTF-32 Little Endian BOM
        elseif (substr($data, 0, 4) === "\xFF\xFE\x00\x00") {
            $data = substr($data, 4);
        }
        // UTF-16 Big Endian BOM
        elseif (substr($data, 0, 2) === "\xFE\xFF") {
            $data = substr($data, 2);
        }
        // UTF-16 Little Endian BOM
        elseif (substr($data, 0, 2) === "\xFF\xFE") {
            $data = substr($data, 2);
        }
        // UTF-8 BOM
        elseif (substr($data, 0, 3) === "\xEF\xBB\xBF") {
            $data = substr($data, 3);
        }

        if (substr($data, 0, 5) === '<?xml' && strspn(substr($data, 5, 1), "\x09\x0A\x0D\x20") && ($pos = strpos($data, '?>')) !== false) {
            $declaration = $this->registry->create(DeclarationParser::class, [substr($data, 5, $pos - 5)]);
            if ($declaration->parse()) {
                $data = substr($data, $pos + 2);
                $data = '<?xml version="' . $declaration->version . '" encoding="' . $encoding . '" standalone="' . (($declaration->standalone) ? 'yes' : 'no') . '"?>' . "\n" .
                    self::set_doctype($data);
            } else {
                $this->error_string = 'SimplePie bug! Please report this!';
                return false;
            }
        } else {
            $data = self::set_doctype($data);
        }

        $return = true;

        static $xml_is_sane = null;
        if ($xml_is_sane === null) {
            $parser_check = xml_parser_create();
            xml_parse_into_struct($parser_check, '<foo>&amp;</foo>', $values);
            if (\PHP_VERSION_ID < 80000) {
                xml_parser_free($parser_check);
            }
            $xml_is_sane = isset($values[0]['value']);
        }

        // Create the parser
        if ($xml_is_sane) {
            $xml = xml_parser_create_ns($this->encoding, $this->separator);
            xml_parser_set_option($xml, XML_OPTION_SKIP_WHITE, 1);
            xml_parser_set_option($xml, XML_OPTION_CASE_FOLDING, 0);
            xml_set_character_data_handler($xml, [$this, 'cdata']);
            xml_set_element_handler($xml, [$this, 'tag_open'], [$this, 'tag_close']);

            // Parse!
            $wrapper = @is_writable(sys_get_temp_dir()) ? 'php://temp' : 'php://memory';
            if (($stream = fopen($wrapper, 'r+')) &&
                fwrite($stream, $data) &&
                rewind($stream)) {
                //Parse by chunks not to use too much memory
                do {
                    $stream_data = (string) fread($stream, 1048576);

                    if (!xml_parse($xml, $stream_data, feof($stream))) {
                        $this->error_code = xml_get_error_code($xml);
                        $this->error_string = xml_error_string($this->error_code) ?: "Unknown";
                        $return = false;
                        break;
                    }
                } while (!feof($stream));
                fclose($stream);
            } else {
                $return = false;
            }

            $this->current_line = xml_get_current_line_number($xml);
            $this->current_column = xml_get_current_column_number($xml);
            $this->current_byte = xml_get_current_byte_index($xml);
            if (\PHP_VERSION_ID < 80000) {
                xml_parser_free($xml);
            }
            return $return;
        }

        libxml_clear_errors();
        $xml = new \XMLReader();
        $xml->xml($data);
        while (@$xml->read()) {
            switch ($xml->nodeType) {
                case \XMLReader::END_ELEMENT:
                    if ($xml->namespaceURI !== '') {
                        $tagName = $xml->namespaceURI . $this->separator . $xml->localName;
                    } else {
                        $tagName = $xml->localName;
                    }
                    $this->tag_close(null, $tagName);
                    break;
                case \XMLReader::ELEMENT:
                    $empty = $xml->isEmptyElement;
                    if ($xml->namespaceURI !== '') {
                        $tagName = $xml->namespaceURI . $this->separator . $xml->localName;
                    } else {
                        $tagName = $xml->localName;
                    }
                    $attributes = [];
                    while ($xml->moveToNextAttribute()) {
                        if ($xml->namespaceURI !== '') {
                            $attrName = $xml->namespaceURI . $this->separator . $xml->localName;
                        } else {
                            $attrName = $xml->localName;
                        }
                        $attributes[$attrName] = $xml->value;
                    }
                    $this->tag_open(null, $tagName, $attributes);
                    if ($empty) {
                        $this->tag_close(null, $tagName);
                    }
                    break;
                case \XMLReader::TEXT:

                case \XMLReader::CDATA:
                    $this->cdata(null, $xml->value);
                    break;
            }
        }
        if ($error = libxml_get_last_error()) {
            $this->error_code = $error->code;
            $this->error_string = $error->message;
            $this->current_line = $error->line;
            $this->current_column = $error->column;
            return false;
        }

        return true;
    }

    /**
     * @return int
     */
    public function get_error_code()
    {
        return $this->error_code;
    }

    /**
     * @return string
     */
    public function get_error_string()
    {
        return $this->error_string;
    }

    /**
     * @return int
     */
    public function get_current_line()
    {
        return $this->current_line;
    }

    /**
     * @return int
     */
    public function get_current_column()
    {
        return $this->current_column;
    }

    /**
     * @return int
     */
    public function get_current_byte()
    {
        return $this->current_byte;
    }

    /**
     * @return array<string, mixed>
     */
    public function get_data()
    {
        return $this->data;
    }

    /**
     * @param XMLParser|resource|null $parser
     * @param array<string, string> $attributes
     * @return void
     */
    public function tag_open($parser, string $tag, array $attributes)
    {
        [$this->namespace[], $this->element[]] = $this->split_ns($tag);

        $attribs = [];
        foreach ($attributes as $name => $value) {
            [$attrib_namespace, $attribute] = $this->split_ns($name);
            $attribs[$attrib_namespace][$attribute] = $value;
        }

        if (isset($attribs[\SimplePie\SimplePie::NAMESPACE_XML]['base'])) {
            $base = $this->registry->call(Misc::class, 'absolutize_url', [$attribs[\SimplePie\SimplePie::NAMESPACE_XML]['base'], end($this->xml_base)]);
            if ($base !== false) {
                $this->xml_base[] = $base;
                $this->xml_base_explicit[] = true;
            }
        } else {
            $this->xml_base[] = end($this->xml_base) ?: '';
            $this->xml_base_explicit[] = end($this->xml_base_explicit);
        }

        if (isset($attribs[\SimplePie\SimplePie::NAMESPACE_XML]['lang'])) {
            $this->xml_lang[] = $attribs[\SimplePie\SimplePie::NAMESPACE_XML]['lang'];
        } else {
            $this->xml_lang[] = end($this->xml_lang) ?: '';
        }

        if ($this->current_xhtml_construct >= 0) {
            $this->current_xhtml_construct++;
            if (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_XHTML) {
                $this->data['data'] .= '<' . end($this->element);
                if (isset($attribs[''])) {
                    foreach ($attribs[''] as $name => $value) {
                        $this->data['data'] .= ' ' . $name . '="' . htmlspecialchars($value, ENT_COMPAT, $this->encoding) . '"';
                    }
                }
                $this->data['data'] .= '>';
            }
        } else {
            $this->datas[] = &$this->data;
            $this->data = &$this->data['child'][end($this->namespace)][end($this->element)][];
            $this->data = ['data' => '', 'attribs' => $attribs, 'xml_base' => end($this->xml_base), 'xml_base_explicit' => end($this->xml_base_explicit), 'xml_lang' => end($this->xml_lang)];
            if ((end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_ATOM_03 && in_array(end($this->element), ['title', 'tagline', 'copyright', 'info', 'summary', 'content']) && isset($attribs['']['mode']) && $attribs['']['mode'] === 'xml')
            || (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_ATOM_10 && in_array(end($this->element), ['rights', 'subtitle', 'summary', 'info', 'title', 'content']) && isset($attribs['']['type']) && $attribs['']['type'] === 'xhtml')
            || (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_RSS_20 && in_array(end($this->element), ['title']))
            || (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_RSS_090 && in_array(end($this->element), ['title']))
            || (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_RSS_10 && in_array(end($this->element), ['title']))) {
                $this->current_xhtml_construct = 0;
            }
        }
    }

    /**
     * @param XMLParser|resource|null $parser
     * @return void
     */
    public function cdata($parser, string $cdata)
    {
        if ($this->current_xhtml_construct >= 0) {
            $this->data['data'] .= htmlspecialchars($cdata, ENT_QUOTES, $this->encoding);
        } else {
            $this->data['data'] .= $cdata;
        }
    }

    /**
     * @param XMLParser|resource|null $parser
     * @return void
     */
    public function tag_close($parser, string $tag)
    {
        if ($this->current_xhtml_construct >= 0) {
            $this->current_xhtml_construct--;
            if (end($this->namespace) === \SimplePie\SimplePie::NAMESPACE_XHTML && !in_array(end($this->element), ['area', 'base', 'basefont', 'br', 'col', 'frame', 'hr', 'img', 'input', 'isindex', 'link', 'meta', 'param'])) {
                $this->data['data'] .= '</' . end($this->element) . '>';
            }
        }
        if ($this->current_xhtml_construct === -1) {
            $this->data = &$this->datas[count($this->datas) - 1];
            array_pop($this->datas);
        }

        array_pop($this->element);
        array_pop($this->namespace);
        array_pop($this->xml_base);
        array_pop($this->xml_base_explicit);
        array_pop($this->xml_lang);
    }

    /**
     * @return array{string, string}
     */
    public function split_ns(string $string)
    {
        static $cache = [];
        if (!isset($cache[$string])) {
            if ($pos = strpos($string, $this->separator)) {
                static $separator_length;
                if (!$separator_length) {
                    $separator_length = strlen($this->separator);
                }
                $namespace = substr($string, 0, $pos);
                $local_name = substr($string, $pos + $separator_length);
                if (strtolower($namespace) === \SimplePie\SimplePie::NAMESPACE_ITUNES) {
                    $namespace = \SimplePie\SimplePie::NAMESPACE_ITUNES;
                }

                // Normalize the Media RSS namespaces
                if ($namespace === \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG ||
                    $namespace === \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG2 ||
                    $namespace === \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG3 ||
                    $namespace === \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG4 ||
                    $namespace === \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG5) {
                    $namespace = \SimplePie\SimplePie::NAMESPACE_MEDIARSS;
                }
                $cache[$string] = [$namespace, $local_name];
            } else {
                $cache[$string] = ['', $string];
            }
        }
        return $cache[$string];
    }

    /**
     * @param array<string, mixed> $data
     */
    private function parse_hcard(array $data, bool $category = false): string
    {
        $name = '';
        $link = '';
        // Check if h-card is set and pass that information on in the link.
        if (isset($data['type']) && in_array('h-card', $data['type'])) {
            if (isset($data['properties']['name'][0])) {
                $name = $data['properties']['name'][0];
            }
            if (isset($data['properties']['url'][0])) {
                $link = $data['properties']['url'][0];
                if ($name === '') {
                    $name = $link;
                } else {
                    // can't have commas in categories.
                    $name = str_replace(',', '', $name);
                }
                $person_tag = $category ? '<span class="person-tag"></span>' : '';
                return '<a class="h-card" href="'.$link.'">'.$person_tag.$name.'</a>';
            }
        }
        return $data['value'] ?? '';
    }

    /**
     * @return true
     */
    private function parse_microformats(string &$data, string $url): bool
    {
        // For PHPStan, we already check that in call site.
        \assert(function_exists('Mf2\parse'));
        \assert(function_exists('Mf2\fetch'));
        $feed_title = '';
        $feed_author = null;
        $author_cache = [];
        $items = [];
        $entries = [];
        $mf = \Mf2\parse($data, $url);
        // First look for an h-feed.
        $h_feed = [];
        foreach ($mf['items'] as $mf_item) {
            if (in_array('h-feed', $mf_item['type'])) {
                $h_feed = $mf_item;
                break;
            }
            // Also look for h-feed or h-entry in the children of each top level item.
            if (!isset($mf_item['children'][0]['type'])) {
                continue;
            }
            if (in_array('h-feed', $mf_item['children'][0]['type'])) {
                $h_feed = $mf_item['children'][0];
                // In this case the parent of the h-feed may be an h-card, so use it as
                // the feed_author.
                if (in_array('h-card', $mf_item['type'])) {
                    $feed_author = $mf_item;
                }
                break;
            } elseif (in_array('h-entry', $mf_item['children'][0]['type'])) {
                $entries = $mf_item['children'];
                // In this case the parent of the h-entry list may be an h-card, so use
                // it as the feed_author.
                if (in_array('h-card', $mf_item['type'])) {
                    $feed_author = $mf_item;
                }
                break;
            }
        }
        if (isset($h_feed['children'])) {
            $entries = $h_feed['children'];
            // Also set the feed title and store author from the h-feed if available.
            if (isset($mf['items'][0]['properties']['name'][0])) {
                $feed_title = $mf['items'][0]['properties']['name'][0];
            }
            if (isset($mf['items'][0]['properties']['author'][0])) {
                $feed_author = $mf['items'][0]['properties']['author'][0];
            }
        } elseif (count($entries) === 0) {
            $entries = $mf['items'];
        }
        for ($i = 0; $i < count($entries); $i++) {
            $entry = $entries[$i];
            if (in_array('h-entry', $entry['type'])) {
                $item = [];
                $title = '';
                $description = '';
                if (isset($entry['properties']['url'][0])) {
                    $link = $entry['properties']['url'][0];
                    if (isset($link['value'])) {
                        $link = $link['value'];
                    }
                    $item['link'] = [['data' => $link]];
                }
                if (isset($entry['properties']['uid'][0])) {
                    $guid = $entry['properties']['uid'][0];
                    if (isset($guid['value'])) {
                        $guid = $guid['value'];
                    }
                    $item['guid'] = [['data' => $guid]];
                }
                if (isset($entry['properties']['name'][0])) {
                    $title = $entry['properties']['name'][0];
                    if (isset($title['value'])) {
                        $title = $title['value'];
                    }
                    $item['title'] = [['data' => $title]];
                }
                if (isset($entry['properties']['author'][0]) || isset($feed_author)) {
                    // author is a special case, it can be plain text or an h-card array.
                    // If it's plain text it can also be a url that should be followed to
                    // get the actual h-card.
                    $author = $entry['properties']['author'][0] ?? $feed_author;
                    if (!is_string($author)) {
                        $author = $this->parse_hcard($author);
                    } elseif (strpos($author, 'http') === 0) {
                        if (isset($author_cache[$author])) {
                            $author = $author_cache[$author];
                        } else {
                            if ($mf = \Mf2\fetch($author)) {
                                foreach ($mf['items'] as $hcard) {
                                    // Only interested in an h-card by itself in this case.
                                    if (!in_array('h-card', $hcard['type'])) {
                                        continue;
                                    }
                                    // It must have a url property matching what we fetched.
                                    if (!isset($hcard['properties']['url']) ||
                                            !(in_array($author, $hcard['properties']['url']))) {
                                        continue;
                                    }
                                    // Save parse_hcard the trouble of finding the correct url.
                                    $hcard['properties']['url'][0] = $author;
                                    // Cache this h-card for the next h-entry to check.
                                    $author_cache[$author] = $this->parse_hcard($hcard);
                                    $author = $author_cache[$author];
                                    break;
                                }
                            }
                        }
                    }
                    $item['author'] = [['data' => $author]];
                }
                if (isset($entry['properties']['photo'][0])) {
                    // If a photo is also in content, don't need to add it again here.
                    $content = '';
                    if (isset($entry['properties']['content'][0]['html'])) {
                        $content = $entry['properties']['content'][0]['html'];
                    }
                    $photo_list = [];
                    for ($j = 0; $j < count($entry['properties']['photo']); $j++) {
                        $photo = $entry['properties']['photo'][$j];
                        if (!empty($photo) && strpos($content, $photo) === false) {
                            $photo_list[] = $photo;
                        }
                    }
                    // When there's more than one photo show the first and use a lightbox.
                    // Need a permanent, unique name for the image set, but don't have
                    // anything unique except for the content itself, so use that.
                    $count = count($photo_list);
                    if ($count > 1) {
                        $image_set_id = preg_replace('/[[:^alnum:]]/', '', $photo_list[0]);
                        $description = '<p>';
                        for ($j = 0; $j < $count; $j++) {
                            $hidden = $j === 0 ? '' : 'class="hidden" ';
                            $description .= '<a href="'.$photo_list[$j].'" '.$hidden.
                                'data-lightbox="image-set-'.$image_set_id.'">'.
                                '<img src="'.$photo_list[$j].'"></a>';
                        }
                        $description .= '<br><b>'.$count.' photos</b></p>';
                    } elseif ($count == 1) {
                        $description = '<p><img src="'.$photo_list[0].'"></p>';
                    }
                }
                if (isset($entry['properties']['content'][0]['html'])) {
                    // e-content['value'] is the same as p-name when they are on the same
                    // element. Use this to replace title with a strip_tags version so
                    // that alt text from images is not included in the title.
                    if ($entry['properties']['content'][0]['value'] === $title) {
                        $title = strip_tags($entry['properties']['content'][0]['html']);
                        $item['title'] = [['data' => $title]];
                    }
                    $description .= $entry['properties']['content'][0]['html'];
                    if (isset($entry['properties']['in-reply-to'][0])) {
                        $in_reply_to = '';
                        if (is_string($entry['properties']['in-reply-to'][0])) {
                            $in_reply_to = $entry['properties']['in-reply-to'][0];
                        } elseif (isset($entry['properties']['in-reply-to'][0]['value'])) {
                            $in_reply_to = $entry['properties']['in-reply-to'][0]['value'];
                        }
                        if ($in_reply_to !== '') {
                            $description .= '<p><span class="in-reply-to"></span> '.
                                '<a href="'.$in_reply_to.'">'.$in_reply_to.'</a><p>';
                        }
                    }
                    $item['description'] = [['data' => $description]];
                }
                if (isset($entry['properties']['category'])) {
                    $category_csv = '';
                    // Categories can also contain h-cards.
                    foreach ($entry['properties']['category'] as $category) {
                        if ($category_csv !== '') {
                            $category_csv .= ', ';
                        }
                        if (is_string($category)) {
                            // Can't have commas in categories.
                            $category_csv .= str_replace(',', '', $category);
                        } else {
                            $category_csv .= $this->parse_hcard($category, true);
                        }
                    }
                    $item['category'] = [['data' => $category_csv]];
                }
                if (isset($entry['properties']['published'][0])) {
                    $timestamp = strtotime($entry['properties']['published'][0]);
                    $pub_date = date('F j Y g:ia', $timestamp).' GMT';
                    $item['pubDate'] = [['data' => $pub_date]];
                }
                // The title and description are set to the empty string to represent
                // a deleted item (which also makes it an invalid rss item).
                if (isset($entry['properties']['deleted'][0])) {
                    $item['title'] = [['data' => '']];
                    $item['description'] = [['data' => '']];
                }
                $items[] = ['child' => ['' => $item]];
            }
        }
        // Mimic RSS data format when storing microformats.
        $link = [['data' => $url]];
        $image = '';
        if (!is_string($feed_author) &&
                isset($feed_author['properties']['photo'][0])) {
            $image = [['child' => ['' => ['url' =>
                [['data' => $feed_author['properties']['photo'][0]]]]]]];
        }
        // Use the name given for the h-feed, or get the title from the html.
        if ($feed_title !== '') {
            $feed_title = [['data' => htmlspecialchars($feed_title)]];
        } elseif ($position = strpos($data, '<title>')) {
            $start = $position < 200 ? 0 : $position - 200;
            $check = substr($data, $start, 400);
            $matches = [];
            if (preg_match('/<title>(.+)<\/title>/', $check, $matches)) {
                $feed_title = [['data' => htmlspecialchars($matches[1])]];
            }
        }
        $channel = ['channel' => [['child' => ['' =>
            ['link' => $link, 'image' => $image, 'title' => $feed_title,
                  'item' => $items]]]]];
        $rss = [['attribs' => ['' => ['version' => '2.0']],
                           'child' => ['' => $channel]]];
        $this->data = ['child' => ['' => ['rss' => $rss]]];
        return true;
    }

    private static function set_doctype(string $data): string
    {
        // Strip DOCTYPE except if containing an [internal subset]
        $data = preg_replace('/^\\s*<!DOCTYPE\\s[^>\\[\\]]*>\s*/', '', $data) ?? $data;
        // Declare HTML entities only if no remaining DOCTYPE
        $doctype = preg_match('/^\\s*<!DOCTYPE\\s/', $data) ? '' : self::declare_html_entities();
        return $doctype . $data;
    }

    private static function declare_html_entities(): string
    {
        // This is required because the RSS specification says that entity-encoded
        // html is allowed, but the xml specification says they must be declared.
        return '<!DOCTYPE rss [ <!ENTITY nbsp "&#x00A0;"> <!ENTITY iexcl "&#x00A1;"> <!ENTITY cent "&#x00A2;"> <!ENTITY pound "&#x00A3;"> <!ENTITY curren "&#x00A4;"> <!ENTITY yen "&#x00A5;"> <!ENTITY brvbar "&#x00A6;"> <!ENTITY sect "&#x00A7;"> <!ENTITY uml "&#x00A8;"> <!ENTITY copy "&#x00A9;"> <!ENTITY ordf "&#x00AA;"> <!ENTITY laquo "&#x00AB;"> <!ENTITY not "&#x00AC;"> <!ENTITY shy "&#x00AD;"> <!ENTITY reg "&#x00AE;"> <!ENTITY macr "&#x00AF;"> <!ENTITY deg "&#x00B0;"> <!ENTITY plusmn "&#x00B1;"> <!ENTITY sup2 "&#x00B2;"> <!ENTITY sup3 "&#x00B3;"> <!ENTITY acute "&#x00B4;"> <!ENTITY micro "&#x00B5;"> <!ENTITY para "&#x00B6;"> <!ENTITY middot "&#x00B7;"> <!ENTITY cedil "&#x00B8;"> <!ENTITY sup1 "&#x00B9;"> <!ENTITY ordm "&#x00BA;"> <!ENTITY raquo "&#x00BB;"> <!ENTITY frac14 "&#x00BC;"> <!ENTITY frac12 "&#x00BD;"> <!ENTITY frac34 "&#x00BE;"> <!ENTITY iquest "&#x00BF;"> <!ENTITY Agrave "&#x00C0;"> <!ENTITY Aacute "&#x00C1;"> <!ENTITY Acirc "&#x00C2;"> <!ENTITY Atilde "&#x00C3;"> <!ENTITY Auml "&#x00C4;"> <!ENTITY Aring "&#x00C5;"> <!ENTITY AElig "&#x00C6;"> <!ENTITY Ccedil "&#x00C7;"> <!ENTITY Egrave "&#x00C8;"> <!ENTITY Eacute "&#x00C9;"> <!ENTITY Ecirc "&#x00CA;"> <!ENTITY Euml "&#x00CB;"> <!ENTITY Igrave "&#x00CC;"> <!ENTITY Iacute "&#x00CD;"> <!ENTITY Icirc "&#x00CE;"> <!ENTITY Iuml "&#x00CF;"> <!ENTITY ETH "&#x00D0;"> <!ENTITY Ntilde "&#x00D1;"> <!ENTITY Ograve "&#x00D2;"> <!ENTITY Oacute "&#x00D3;"> <!ENTITY Ocirc "&#x00D4;"> <!ENTITY Otilde "&#x00D5;"> <!ENTITY Ouml "&#x00D6;"> <!ENTITY times "&#x00D7;"> <!ENTITY Oslash "&#x00D8;"> <!ENTITY Ugrave "&#x00D9;"> <!ENTITY Uacute "&#x00DA;"> <!ENTITY Ucirc "&#x00DB;"> <!ENTITY Uuml "&#x00DC;"> <!ENTITY Yacute "&#x00DD;"> <!ENTITY THORN "&#x00DE;"> <!ENTITY szlig "&#x00DF;"> <!ENTITY agrave "&#x00E0;"> <!ENTITY aacute "&#x00E1;"> <!ENTITY acirc "&#x00E2;"> <!ENTITY atilde "&#x00E3;"> <!ENTITY auml "&#x00E4;"> <!ENTITY aring "&#x00E5;"> <!ENTITY aelig "&#x00E6;"> <!ENTITY ccedil "&#x00E7;"> <!ENTITY egrave "&#x00E8;"> <!ENTITY eacute "&#x00E9;"> <!ENTITY ecirc "&#x00EA;"> <!ENTITY euml "&#x00EB;"> <!ENTITY igrave "&#x00EC;"> <!ENTITY iacute "&#x00ED;"> <!ENTITY icirc "&#x00EE;"> <!ENTITY iuml "&#x00EF;"> <!ENTITY eth "&#x00F0;"> <!ENTITY ntilde "&#x00F1;"> <!ENTITY ograve "&#x00F2;"> <!ENTITY oacute "&#x00F3;"> <!ENTITY ocirc "&#x00F4;"> <!ENTITY otilde "&#x00F5;"> <!ENTITY ouml "&#x00F6;"> <!ENTITY divide "&#x00F7;"> <!ENTITY oslash "&#x00F8;"> <!ENTITY ugrave "&#x00F9;"> <!ENTITY uacute "&#x00FA;"> <!ENTITY ucirc "&#x00FB;"> <!ENTITY uuml "&#x00FC;"> <!ENTITY yacute "&#x00FD;"> <!ENTITY thorn "&#x00FE;"> <!ENTITY yuml "&#x00FF;"> <!ENTITY OElig "&#x0152;"> <!ENTITY oelig "&#x0153;"> <!ENTITY Scaron "&#x0160;"> <!ENTITY scaron "&#x0161;"> <!ENTITY Yuml "&#x0178;"> <!ENTITY fnof "&#x0192;"> <!ENTITY circ "&#x02C6;"> <!ENTITY tilde "&#x02DC;"> <!ENTITY Alpha "&#x0391;"> <!ENTITY Beta "&#x0392;"> <!ENTITY Gamma "&#x0393;"> <!ENTITY Epsilon "&#x0395;"> <!ENTITY Zeta "&#x0396;"> <!ENTITY Eta "&#x0397;"> <!ENTITY Theta "&#x0398;"> <!ENTITY Iota "&#x0399;"> <!ENTITY Kappa "&#x039A;"> <!ENTITY Lambda "&#x039B;"> <!ENTITY Mu "&#x039C;"> <!ENTITY Nu "&#x039D;"> <!ENTITY Xi "&#x039E;"> <!ENTITY Omicron "&#x039F;"> <!ENTITY Pi "&#x03A0;"> <!ENTITY Rho "&#x03A1;"> <!ENTITY Sigma "&#x03A3;"> <!ENTITY Tau "&#x03A4;"> <!ENTITY Upsilon "&#x03A5;"> <!ENTITY Phi "&#x03A6;"> <!ENTITY Chi "&#x03A7;"> <!ENTITY Psi "&#x03A8;"> <!ENTITY Omega "&#x03A9;"> <!ENTITY alpha "&#x03B1;"> <!ENTITY beta "&#x03B2;"> <!ENTITY gamma "&#x03B3;"> <!ENTITY delta "&#x03B4;"> <!ENTITY epsilon "&#x03B5;"> <!ENTITY zeta "&#x03B6;"> <!ENTITY eta "&#x03B7;"> <!ENTITY theta "&#x03B8;"> <!ENTITY iota "&#x03B9;"> <!ENTITY kappa "&#x03BA;"> <!ENTITY lambda "&#x03BB;"> <!ENTITY mu "&#x03BC;"> <!ENTITY nu "&#x03BD;"> <!ENTITY xi "&#x03BE;"> <!ENTITY omicron "&#x03BF;"> <!ENTITY pi "&#x03C0;"> <!ENTITY rho "&#x03C1;"> <!ENTITY sigmaf "&#x03C2;"> <!ENTITY sigma "&#x03C3;"> <!ENTITY tau "&#x03C4;"> <!ENTITY upsilon "&#x03C5;"> <!ENTITY phi "&#x03C6;"> <!ENTITY chi "&#x03C7;"> <!ENTITY psi "&#x03C8;"> <!ENTITY omega "&#x03C9;"> <!ENTITY thetasym "&#x03D1;"> <!ENTITY upsih "&#x03D2;"> <!ENTITY piv "&#x03D6;"> <!ENTITY ensp "&#x2002;"> <!ENTITY emsp "&#x2003;"> <!ENTITY thinsp "&#x2009;"> <!ENTITY zwnj "&#x200C;"> <!ENTITY zwj "&#x200D;"> <!ENTITY lrm "&#x200E;"> <!ENTITY rlm "&#x200F;"> <!ENTITY ndash "&#x2013;"> <!ENTITY mdash "&#x2014;"> <!ENTITY lsquo "&#x2018;"> <!ENTITY rsquo "&#x2019;"> <!ENTITY sbquo "&#x201A;"> <!ENTITY ldquo "&#x201C;"> <!ENTITY rdquo "&#x201D;"> <!ENTITY bdquo "&#x201E;"> <!ENTITY dagger "&#x2020;"> <!ENTITY Dagger "&#x2021;"> <!ENTITY bull "&#x2022;"> <!ENTITY hellip "&#x2026;"> <!ENTITY permil "&#x2030;"> <!ENTITY prime "&#x2032;"> <!ENTITY Prime "&#x2033;"> <!ENTITY lsaquo "&#x2039;"> <!ENTITY rsaquo "&#x203A;"> <!ENTITY oline "&#x203E;"> <!ENTITY frasl "&#x2044;"> <!ENTITY euro "&#x20AC;"> <!ENTITY image "&#x2111;"> <!ENTITY weierp "&#x2118;"> <!ENTITY real "&#x211C;"> <!ENTITY trade "&#x2122;"> <!ENTITY alefsym "&#x2135;"> <!ENTITY larr "&#x2190;"> <!ENTITY uarr "&#x2191;"> <!ENTITY rarr "&#x2192;"> <!ENTITY darr "&#x2193;"> <!ENTITY harr "&#x2194;"> <!ENTITY crarr "&#x21B5;"> <!ENTITY lArr "&#x21D0;"> <!ENTITY uArr "&#x21D1;"> <!ENTITY rArr "&#x21D2;"> <!ENTITY dArr "&#x21D3;"> <!ENTITY hArr "&#x21D4;"> <!ENTITY forall "&#x2200;"> <!ENTITY part "&#x2202;"> <!ENTITY exist "&#x2203;"> <!ENTITY empty "&#x2205;"> <!ENTITY nabla "&#x2207;"> <!ENTITY isin "&#x2208;"> <!ENTITY notin "&#x2209;"> <!ENTITY ni "&#x220B;"> <!ENTITY prod "&#x220F;"> <!ENTITY sum "&#x2211;"> <!ENTITY minus "&#x2212;"> <!ENTITY lowast "&#x2217;"> <!ENTITY radic "&#x221A;"> <!ENTITY prop "&#x221D;"> <!ENTITY infin "&#x221E;"> <!ENTITY ang "&#x2220;"> <!ENTITY and "&#x2227;"> <!ENTITY or "&#x2228;"> <!ENTITY cap "&#x2229;"> <!ENTITY cup "&#x222A;"> <!ENTITY int "&#x222B;"> <!ENTITY there4 "&#x2234;"> <!ENTITY sim "&#x223C;"> <!ENTITY cong "&#x2245;"> <!ENTITY asymp "&#x2248;"> <!ENTITY ne "&#x2260;"> <!ENTITY equiv "&#x2261;"> <!ENTITY le "&#x2264;"> <!ENTITY ge "&#x2265;"> <!ENTITY sub "&#x2282;"> <!ENTITY sup "&#x2283;"> <!ENTITY nsub "&#x2284;"> <!ENTITY sube "&#x2286;"> <!ENTITY supe "&#x2287;"> <!ENTITY oplus "&#x2295;"> <!ENTITY otimes "&#x2297;"> <!ENTITY perp "&#x22A5;"> <!ENTITY sdot "&#x22C5;"> <!ENTITY lceil "&#x2308;"> <!ENTITY rceil "&#x2309;"> <!ENTITY lfloor "&#x230A;"> <!ENTITY rfloor "&#x230B;"> <!ENTITY lang "&#x2329;"> <!ENTITY rang "&#x232A;"> <!ENTITY loz "&#x25CA;"> <!ENTITY spades "&#x2660;"> <!ENTITY clubs "&#x2663;"> <!ENTITY hearts "&#x2665;"> <!ENTITY diams "&#x2666;"> ]>';
    }
}

class_alias('SimplePie\Parser', 'SimplePie_Parser');

}

namespace SimplePie {

/**
 * Handles `<media:rating>` or `<itunes:explicit>` tags as defined in Media RSS and iTunes RSS respectively
 *
 * Used by {@see \SimplePie\Enclosure::get_rating()} and {@see \SimplePie\Enclosure::get_ratings()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_rating_class()}
 */
class Rating
{
    /**
     * Rating scheme
     *
     * @var ?string
     * @see get_scheme()
     */
    public $scheme;

    /**
     * Rating value
     *
     * @var ?string
     * @see get_value()
     */
    public $value;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     */
    public function __construct(
        ?string $scheme = null,
        ?string $value = null
    ) {
        $this->scheme = $scheme;
        $this->value = $value;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the organizational scheme for the rating
     *
     * @return string|null
     */
    public function get_scheme()
    {
        if ($this->scheme !== null) {
            return $this->scheme;
        }

        return null;
    }

    /**
     * Get the value of the rating
     *
     * @return string|null
     */
    public function get_value()
    {
        if ($this->value !== null) {
            return $this->value;
        }

        return null;
    }
}

class_alias('SimplePie\Rating', 'SimplePie_Rating');

}

namespace SimplePie {

use InvalidArgumentException;
use SimplePie\Content\Type\Sniffer;
use SimplePie\Parse\Date;
use SimplePie\XML\Declaration\Parser as DeclarationParser;

/**
 * Handles creating objects and calling methods
 *
 * Access this via {@see \SimplePie\SimplePie::get_registry()}
 */
class Registry
{
    /**
     * Default class mapping
     *
     * Overriding classes *must* subclass these.
     *
     * @var array<class-string, class-string>
     */
    protected $default = [
        Cache::class => Cache::class,
        Locator::class => Locator::class,
        Parser::class => Parser::class,
        File::class => File::class,
        Sanitize::class => Sanitize::class,
        Item::class => Item::class,
        Author::class => Author::class,
        Category::class => Category::class,
        Enclosure::class => Enclosure::class,
        Caption::class => Caption::class,
        Copyright::class => Copyright::class,
        Credit::class => Credit::class,
        Rating::class => Rating::class,
        Restriction::class => Restriction::class,
        Sniffer::class => Sniffer::class,
        Source::class => Source::class,
        Misc::class => Misc::class,
        DeclarationParser::class => DeclarationParser::class,
        Date::class => Date::class,
    ];

    /**
     * Class mapping
     *
     * @see register()
     * @var array<string, class-string>
     */
    protected $classes = [];

    /**
     * Legacy classes
     *
     * @see register()
     * @var array<class-string>
     */
    protected $legacy = [];

    /**
     * Legacy types
     *
     * @see register()
     * @var array<string, class-string>
     */
    private $legacyTypes = [
        'Cache' => Cache::class,
        'Locator' => Locator::class,
        'Parser' => Parser::class,
        'File' => File::class,
        'Sanitize' => Sanitize::class,
        'Item' => Item::class,
        'Author' => Author::class,
        'Category' => Category::class,
        'Enclosure' => Enclosure::class,
        'Caption' => Caption::class,
        'Copyright' => Copyright::class,
        'Credit' => Credit::class,
        'Rating' => Rating::class,
        'Restriction' => Restriction::class,
        'Content_Type_Sniffer' => Sniffer::class,
        'Source' => Source::class,
        'Misc' => Misc::class,
        'XML_Declaration_Parser' => DeclarationParser::class,
        'Parse_Date' => Date::class,
    ];

    /**
     * Constructor
     *
     * No-op
     */
    public function __construct()
    {
    }

    /**
     * Register a class
     *
     * @param string $type See {@see $default} for names
     * @param class-string $class Class name, must subclass the corresponding default
     * @param bool $legacy Whether to enable legacy support for this class
     * @return bool Successfulness
     */
    public function register(string $type, $class, bool $legacy = false)
    {
        if (array_key_exists($type, $this->legacyTypes)) {
            // trigger_error(sprintf('"%s"(): Using argument #1 ($type) with value "%s" is deprecated since SimplePie 1.8.0, use class-string "%s" instead.', __METHOD__, $type, $this->legacyTypes[$type]), \E_USER_DEPRECATED);

            $type = $this->legacyTypes[$type];
        }

        if (!array_key_exists($type, $this->default)) {
            return false;
        }

        if (!class_exists($class)) {
            return false;
        }

        /** @var string */
        $base_class = $this->default[$type];

        if (!is_subclass_of($class, $base_class)) {
            return false;
        }

        $this->classes[$type] = $class;

        if ($legacy) {
            $this->legacy[] = $class;
        }

        return true;
    }

    /**
     * Get the class registered for a type
     *
     * Where possible, use {@see create()} or {@see call()} instead
     *
     * @template T
     * @param class-string<T> $type
     * @return class-string<T>|null
     */
    public function get_class($type)
    {
        if (array_key_exists($type, $this->legacyTypes)) {
            // trigger_error(sprintf('"%s"(): Using argument #1 ($type) with value "%s" is deprecated since SimplePie 1.8.0, use class-string "%s" instead.', __METHOD__, $type, $this->legacyTypes[$type]), \E_USER_DEPRECATED);

            $type = $this->legacyTypes[$type];
        }

        if (!array_key_exists($type, $this->default)) {
            return null;
        }

        // For PHPStan: values in $default should be subtypes of keys.
        /** @var class-string<T> */
        $class = $this->default[$type];

        if (array_key_exists($type, $this->classes)) {
            // For PHPStan: values in $classes should be subtypes of keys.
            /** @var class-string<T> */
            $class = $this->classes[$type];
        }

        return $class;
    }

    /**
     * Create a new instance of a given type
     *
     * @template T class-string $type
     * @param class-string<T> $type
     * @param array<mixed> $parameters Parameters to pass to the constructor
     * @return T Instance of class
     */
    public function &create($type, array $parameters = [])
    {
        $class = $this->get_class($type);
        if ($class === null) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($type) "%s" not found in class list.',
                __METHOD__,
                $type
            ), 1);
        }

        if (!method_exists($class, '__construct')) {
            $instance = new $class();
        } else {
            $reflector = new \ReflectionClass($class);
            // For PHPStan: $class is T.
            /** @var T */
            $instance = $reflector->newInstanceArgs($parameters);
        }

        if ($instance instanceof RegistryAware) {
            $instance->set_registry($this);
        } elseif (method_exists($instance, 'set_registry')) {
            trigger_error(sprintf('Using the method "set_registry()" without implementing "%s" is deprecated since SimplePie 1.8.0, implement "%s" in "%s".', RegistryAware::class, RegistryAware::class, $class), \E_USER_DEPRECATED);
            $instance->set_registry($this);
        }

        return $instance;
    }

    /**
     * Call a static method for a type
     *
     * @param class-string $type
     * @param string $method
     * @param array<mixed> $parameters
     * @return mixed
     */
    public function &call($type, string $method, array $parameters = [])
    {
        $class = $this->get_class($type);
        if ($class === null) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #1 ($type) "%s" not found in class list.',
                __METHOD__,
                $type
            ), 1);
        }

        if (in_array($class, $this->legacy)) {
            switch ($type) {
                case Cache::class:
                    // For backwards compatibility with old non-static
                    // Cache::create() methods in PHP < 8.0.
                    // No longer supported as of PHP 8.0.
                    if ($method === 'get_handler') {
                        // Fixing this PHPStan error breaks CacheTest::testDirectOverrideLegacy()
                        /** @phpstan-ignore argument.type */
                        $result = @call_user_func_array([$class, 'create'], $parameters);
                        return $result;
                    }
                    break;
            }
        }

        $callable = [$class, $method];
        assert(is_callable($callable), 'For PHPstan');
        $result = call_user_func_array($callable, $parameters);
        return $result;
    }
}

class_alias('SimplePie\Registry', 'SimplePie_Registry');

}

namespace SimplePie {

/**
 * Handles the injection of Registry into other class
 *
 * {@see \SimplePie\SimplePie::get_registry()}
 */
interface RegistryAware
{
    /**
     * Set the Registry into the class
     *
     * @return void
     */
    public function set_registry(Registry $registry);
}

}

namespace SimplePie {

/**
 * Handles `<media:restriction>` as defined in Media RSS
 *
 * Used by {@see \SimplePie\Enclosure::get_restriction()} and {@see \SimplePie\Enclosure::get_restrictions()}
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_restriction_class()}
 */
class Restriction
{
    public const RELATIONSHIP_ALLOW = 'allow';
    public const RELATIONSHIP_DENY = 'deny';

    /**
     * Relationship ('allow'/'deny')
     *
     * @var self::RELATIONSHIP_*|null
     * @see get_relationship()
     */
    public $relationship;

    /**
     * Type of restriction
     *
     * @var string|null
     * @see get_type()
     */
    public $type;

    /**
     * Restricted values
     *
     * @var string|null
     * @see get_value()
     */
    public $value;

    /**
     * Constructor, used to input the data
     *
     * For documentation on all the parameters, see the corresponding
     * properties and their accessors
     *
     * @param ?self::RELATIONSHIP_* $relationship
     */
    public function __construct(?string $relationship = null, ?string $type = null, ?string $value = null)
    {
        $this->relationship = $relationship;
        $this->type = $type;
        $this->value = $value;
    }

    /**
     * String-ified version
     *
     * @return string
     */
    public function __toString()
    {
        // There is no $this->data here
        return md5(serialize($this));
    }

    /**
     * Get the relationship
     *
     * @return ?self::RELATIONSHIP_*
     */
    public function get_relationship()
    {
        if ($this->relationship !== null) {
            return $this->relationship;
        }

        return null;
    }

    /**
     * Get the type
     *
     * @return string|null
     */
    public function get_type()
    {
        if ($this->type !== null) {
            return $this->type;
        }

        return null;
    }

    /**
     * Get the list of restricted things
     *
     * @return string|null
     */
    public function get_value()
    {
        if ($this->value !== null) {
            return $this->value;
        }

        return null;
    }
}

class_alias('SimplePie\Restriction', 'SimplePie_Restriction');

}

namespace SimplePie {

use DOMDocument;
use DOMXPath;
use InvalidArgumentException;
use Psr\Http\Client\ClientInterface;
use Psr\Http\Message\RequestFactoryInterface;
use Psr\Http\Message\UriFactoryInterface;
use SimplePie\Cache\Base;
use SimplePie\Cache\BaseDataCache;
use SimplePie\Cache\CallableNameFilter;
use SimplePie\Cache\DataCache;
use SimplePie\Cache\NameFilter;
use SimplePie\HTTP\Client;
use SimplePie\HTTP\ClientException;
use SimplePie\HTTP\FileClient;
use SimplePie\HTTP\Psr18Client;

/**
 * Used for data cleanup and post-processing
 *
 *
 * This class can be overloaded with {@see \SimplePie\SimplePie::set_sanitize_class()}
 *
 * @todo Move to using an actual HTML parser (this will allow tags to be properly stripped, and to switch between HTML and XHTML), this will also make it easier to shorten a string while preserving HTML tags
 */
class Sanitize implements RegistryAware
{
    // Private vars
    /** @var string */
    public $base = '';

    // Options
    /** @var bool */
    public $remove_div = true;
    /** @var string */
    public $image_handler = '';
    /** @var string[] */
    public $strip_htmltags = ['base', 'blink', 'body', 'doctype', 'embed', 'font', 'form', 'frame', 'frameset', 'html', 'iframe', 'input', 'marquee', 'meta', 'noscript', 'object', 'param', 'script', 'style'];
    /** @var bool */
    public $encode_instead_of_strip = false;
    /** @var string[] */
    public $strip_attributes = ['bgsound', 'expr', 'id', 'style', 'onclick', 'onerror', 'onfinish', 'onmouseover', 'onmouseout', 'onfocus', 'onblur', 'lowsrc', 'dynsrc'];
    /** @var string[] */
    public $rename_attributes = [];
    /** @var array<string, array<string, string>> */
    public $add_attributes = ['audio' => ['preload' => 'none'], 'iframe' => ['sandbox' => 'allow-scripts allow-same-origin'], 'video' => ['preload' => 'none']];
    /** @var bool */
    public $strip_comments = false;
    /** @var string */
    public $output_encoding = 'UTF-8';
    /** @var bool */
    public $enable_cache = true;
    /** @var string */
    public $cache_location = './cache';
    /** @var string&(callable(string): string) */
    public $cache_name_function = 'md5';

    /**
     * @var NameFilter
     */
    private $cache_namefilter;
    /** @var int */
    public $timeout = 10;
    /** @var string */
    public $useragent = '';
    /** @var bool */
    public $force_fsockopen = false;
    /** @var array<string, string|string[]> */
    public $replace_url_attributes = [];
    /**
     * @var array<int, mixed> Custom curl options
     * @see SimplePie::set_curl_options()
     */
    private $curl_options = [];

    /** @var Registry */
    public $registry;

    /**
     * @var DataCache|null
     */
    private $cache = null;

    /**
     * @var int Cache duration (in seconds)
     */
    private $cache_duration = 3600;

    /**
     * List of domains for which to force HTTPS.
     * @see \SimplePie\Sanitize::set_https_domains()
     * Array is a tree split at DNS levels. Example:
     * array('biz' => true, 'com' => array('example' => true), 'net' => array('example' => array('www' => true)))
     * @var true|array<string, true|array<string, true|array<string, array<string, true|array<string, true|array<string, true>>>>>>
     */
    public $https_domains = [];

    /**
     * @var Client|null
     */
    private $http_client = null;

    public function __construct()
    {
        // Set defaults
        $this->set_url_replacements(null);
    }

    /**
     * @return void
     */
    public function remove_div(bool $enable = true)
    {
        $this->remove_div = (bool) $enable;
    }

    /**
     * @param string|false $page
     * @return void
     */
    public function set_image_handler($page = false)
    {
        if ($page) {
            $this->image_handler = (string) $page;
        } else {
            $this->image_handler = '';
        }
    }

    /**
     * @return void
     */
    public function set_registry(\SimplePie\Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * @param (string&(callable(string): string))|NameFilter $cache_name_function
     * @param class-string<Cache> $cache_class
     * @return void
     */
    public function pass_cache_data(bool $enable_cache = true, string $cache_location = './cache', $cache_name_function = 'md5', string $cache_class = Cache::class, ?DataCache $cache = null)
    {
        $this->enable_cache = $enable_cache;

        if ($cache_location) {
            $this->cache_location = $cache_location;
        }

        // @phpstan-ignore-next-line Enforce PHPDoc type.
        if (!is_string($cache_name_function) && !$cache_name_function instanceof NameFilter) {
            throw new InvalidArgumentException(sprintf(
                '%s(): Argument #3 ($cache_name_function) must be of type %s',
                __METHOD__,
                NameFilter::class
            ), 1);
        }

        // BC: $cache_name_function could be a callable as string
        if (is_string($cache_name_function)) {
            // trigger_error(sprintf('Providing $cache_name_function as string in "%s()" is deprecated since SimplePie 1.8.0, provide as "%s" instead.', __METHOD__, NameFilter::class), \E_USER_DEPRECATED);
            $this->cache_name_function = $cache_name_function;

            $cache_name_function = new CallableNameFilter($cache_name_function);
        }

        $this->cache_namefilter = $cache_name_function;

        if ($cache !== null) {
            $this->cache = $cache;
        }
    }

    /**
     * Set a PSR-18 client and PSR-17 factories
     *
     * Allows you to use your own HTTP client implementations.
     */
    final public function set_http_client(
        ClientInterface $http_client,
        RequestFactoryInterface $request_factory,
        UriFactoryInterface $uri_factory
    ): void {
        $this->http_client = new Psr18Client($http_client, $request_factory, $uri_factory);
    }

    /**
     * @deprecated since SimplePie 1.9.0, use \SimplePie\Sanitize::set_http_client() instead.
     * @param class-string<File> $file_class
     * @param array<int, mixed> $curl_options
     * @return void
     */
    public function pass_file_data(string $file_class = File::class, int $timeout = 10, string $useragent = '', bool $force_fsockopen = false, array $curl_options = [])
    {
        // trigger_error(sprintf('SimplePie\Sanitize::pass_file_data() is deprecated since SimplePie 1.9.0, please use "SimplePie\Sanitize::set_http_client()" instead.'), \E_USER_DEPRECATED);
        if ($timeout) {
            $this->timeout = $timeout;
        }

        if ($useragent) {
            $this->useragent = $useragent;
        }

        if ($force_fsockopen) {
            $this->force_fsockopen = $force_fsockopen;
        }

        $this->curl_options = $curl_options;
        // Invalidate the registered client.
        $this->http_client = null;
    }

    /**
     * @param string[]|string|false $tags Set a list of tags to strip, or set empty string to use default tags, or false to strip nothing.
     * @return void
     */
    public function strip_htmltags($tags = ['base', 'blink', 'body', 'doctype', 'embed', 'font', 'form', 'frame', 'frameset', 'html', 'iframe', 'input', 'marquee', 'meta', 'noscript', 'object', 'param', 'script', 'style'])
    {
        if ($tags) {
            if (is_array($tags)) {
                $this->strip_htmltags = $tags;
            } else {
                $this->strip_htmltags = explode(',', $tags);
            }
        } else {
            $this->strip_htmltags = [];
        }
    }

    /**
     * @return void
     */
    public function encode_instead_of_strip(bool $encode = false)
    {
        $this->encode_instead_of_strip = $encode;
    }

    /**
     * @param string[]|string $attribs
     * @return void
     */
    public function rename_attributes($attribs = [])
    {
        if ($attribs) {
            if (is_array($attribs)) {
                $this->rename_attributes = $attribs;
            } else {
                $this->rename_attributes = explode(',', $attribs);
            }
        } else {
            $this->rename_attributes = [];
        }
    }

    /**
     * @param string[]|string $attribs
     * @return void
     */
    public function strip_attributes($attribs = ['bgsound', 'expr', 'id', 'style', 'onclick', 'onerror', 'onfinish', 'onmouseover', 'onmouseout', 'onfocus', 'onblur', 'lowsrc', 'dynsrc'])
    {
        if ($attribs) {
            if (is_array($attribs)) {
                $this->strip_attributes = $attribs;
            } else {
                $this->strip_attributes = explode(',', $attribs);
            }
        } else {
            $this->strip_attributes = [];
        }
    }

    /**
     * @param array<string, array<string, string>> $attribs
     * @return void
     */
    public function add_attributes(array $attribs = ['audio' => ['preload' => 'none'], 'iframe' => ['sandbox' => 'allow-scripts allow-same-origin'], 'video' => ['preload' => 'none']])
    {
        $this->add_attributes = $attribs;
    }

    /**
     * @return void
     */
    public function strip_comments(bool $strip = false)
    {
        $this->strip_comments = $strip;
    }

    /**
     * @return void
     */
    public function set_output_encoding(string $encoding = 'UTF-8')
    {
        $this->output_encoding = $encoding;
    }

    /**
     * Set element/attribute key/value pairs of HTML attributes
     * containing URLs that need to be resolved relative to the feed
     *
     * Defaults to |a|@href, |area|@href, |audio|@src, |blockquote|@cite,
     * |del|@cite, |form|@action, |img|@longdesc, |img|@src, |input|@src,
     * |ins|@cite, |q|@cite, |source|@src, |video|@src
     *
     * @since 1.0
     * @param array<string, string|string[]>|null $element_attribute Element/attribute key/value pairs, null for default
     * @return void
     */
    public function set_url_replacements(?array $element_attribute = null)
    {
        if ($element_attribute === null) {
            $element_attribute = [
                'a' => 'href',
                'area' => 'href',
                'audio' => 'src',
                'blockquote' => 'cite',
                'del' => 'cite',
                'form' => 'action',
                'img' => [
                    'longdesc',
                    'src'
                ],
                'input' => 'src',
                'ins' => 'cite',
                'q' => 'cite',
                'source' => 'src',
                'video' => [
                    'poster',
                    'src'
                ]
            ];
        }
        $this->replace_url_attributes = $element_attribute;
    }

    /**
     * Set the list of domains for which to force HTTPS.
     * @see \SimplePie\Misc::https_url()
     * Example array('biz', 'example.com', 'example.org', 'www.example.net');
     *
     * @param string[] $domains list of domain names ['biz', 'example.com', 'example.org', 'www.example.net']
     *
     * @return void
     */
    public function set_https_domains(array $domains)
    {
        $this->https_domains = [];
        foreach ($domains as $domain) {
            $domain = trim($domain, ". \t\n\r\0\x0B");
            $segments = array_reverse(explode('.', $domain));
            /** @var true|array<string, true|array<string, true|array<string, array<string, true|array<string, true|array<string, true>>>>>> */ // Needed for PHPStan.
            $node = &$this->https_domains;
            foreach ($segments as $segment) {//Build a tree
                if ($node === true) {
                    break;
                }
                if (!isset($node[$segment])) {
                    $node[$segment] = [];
                }
                $node = &$node[$segment];
            }
            $node = true;
        }
    }

    /**
     * Check if the domain is in the list of forced HTTPS.
     *
     * @return bool
     */
    protected function is_https_domain(string $domain)
    {
        $domain = trim($domain, '. ');
        $segments = array_reverse(explode('.', $domain));
        $node = &$this->https_domains;
        foreach ($segments as $segment) {//Explore the tree
            if (isset($node[$segment])) {
                $node = &$node[$segment];
            } else {
                break;
            }
        }
        return $node === true;
    }

    /**
     * Force HTTPS for selected Web sites.
     *
     * @return string
     */
    public function https_url(string $url)
    {
        return (
            strtolower(substr($url, 0, 7)) === 'http://'
            && ($parsed = parse_url($url, PHP_URL_HOST)) !== false // Malformed URL
            && $parsed !== null // Missing host
            && $this->is_https_domain($parsed) // Should be forced?
        ) ? substr_replace($url, 's', 4, 0) // Add the 's' to HTTPS
        : $url;
    }

    /**
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @param string $base
     * @return string Sanitized data; false if output encoding is changed to something other than UTF-8 and conversion fails
     */
    public function sanitize(string $data, int $type, string $base = '')
    {
        $data = trim($data);
        if ($data !== '' || $type & \SimplePie\SimplePie::CONSTRUCT_IRI) {
            if ($type & \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML) {
                if (preg_match('/(&(#(x[0-9a-fA-F]+|[0-9]+)|[a-zA-Z0-9]+)|<\/[A-Za-z][^\x09\x0A\x0B\x0C\x0D\x20\x2F\x3E]*' . \SimplePie\SimplePie::PCRE_HTML_ATTRIBUTE . '>)/', $data)) {
                    $type |= \SimplePie\SimplePie::CONSTRUCT_HTML;
                } else {
                    $type |= \SimplePie\SimplePie::CONSTRUCT_TEXT;
                }
            }

            if ($type & \SimplePie\SimplePie::CONSTRUCT_BASE64) {
                $data = base64_decode($data);
            }

            if ($type & (\SimplePie\SimplePie::CONSTRUCT_HTML | \SimplePie\SimplePie::CONSTRUCT_XHTML)) {
                if (!class_exists('DOMDocument')) {
                    throw new \SimplePie\Exception('DOMDocument not found, unable to use sanitizer');
                }
                $document = new \DOMDocument();
                $document->encoding = 'UTF-8';

                // PHPStan seems to have trouble resolving int-mask because bitwise
                // operators are used when operators are used when passing this parameter.
                // https://github.com/phpstan/phpstan/issues/9384
                /** @var int-mask-of<SimplePie::CONSTRUCT_*> $type */
                $data = $this->preprocess($data, $type);

                set_error_handler([Misc::class, 'silence_errors']);
                $document->loadHTML($data);
                restore_error_handler();

                $xpath = new \DOMXPath($document);

                // Strip comments
                if ($this->strip_comments) {
                    /** @var \DOMNodeList<\DOMComment> */
                    $comments = $xpath->query('//comment()');

                    foreach ($comments as $comment) {
                        $parentNode = $comment->parentNode;
                        assert($parentNode !== null, 'For PHPStan, comment must have a parent');
                        $parentNode->removeChild($comment);
                    }
                }

                // Strip out HTML tags and attributes that might cause various security problems.
                // Based on recommendations by Mark Pilgrim at:
                // https://web.archive.org/web/20110902041826/http://diveintomark.org:80/archives/2003/06/12/how_to_consume_rss_safely
                if ($this->strip_htmltags) {
                    foreach ($this->strip_htmltags as $tag) {
                        $this->strip_tag($tag, $document, $xpath, $type);
                    }
                }

                if ($this->rename_attributes) {
                    foreach ($this->rename_attributes as $attrib) {
                        $this->rename_attr($attrib, $xpath);
                    }
                }

                if ($this->strip_attributes) {
                    foreach ($this->strip_attributes as $attrib) {
                        $this->strip_attr($attrib, $xpath);
                    }
                }

                if ($this->add_attributes) {
                    foreach ($this->add_attributes as $tag => $valuePairs) {
                        $this->add_attr($tag, $valuePairs, $document);
                    }
                }

                // Replace relative URLs
                $this->base = $base;
                foreach ($this->replace_url_attributes as $element => $attributes) {
                    $this->replace_urls($document, $element, $attributes);
                }

                // If image handling (caching, etc.) is enabled, cache and rewrite all the image tags.
                if ($this->image_handler !== '' && $this->enable_cache) {
                    $images = $document->getElementsByTagName('img');

                    foreach ($images as $img) {
                        if ($img->hasAttribute('src')) {
                            $image_url = $this->cache_namefilter->filter($img->getAttribute('src'));
                            $cache = $this->get_cache($image_url);

                            if ($cache->get_data($image_url, false)) {
                                $img->setAttribute('src', $this->image_handler . $image_url);
                            } else {
                                try {
                                    $file = $this->get_http_client()->request(
                                        Client::METHOD_GET,
                                        $img->getAttribute('src'),
                                        ['X-FORWARDED-FOR' => $_SERVER['REMOTE_ADDR']]
                                    );
                                } catch (ClientException $th) {
                                    continue;
                                }

                                if ((!Misc::is_remote_uri($file->get_final_requested_uri()) || ($file->get_status_code() === 200 || $file->get_status_code() > 206 && $file->get_status_code() < 300))) {
                                    if ($cache->set_data($image_url, ['headers' => $file->get_headers(), 'body' => $file->get_body_content()], $this->cache_duration)) {
                                        $img->setAttribute('src', $this->image_handler . $image_url);
                                    } else {
                                        trigger_error("$this->cache_location is not writable. Make sure you've set the correct relative or absolute path, and that the location is server-writable.", E_USER_WARNING);
                                    }
                                }
                            }
                        }
                    }
                }

                // Get content node
                $div = null;
                if (($item = $document->getElementsByTagName('body')->item(0)) !== null) {
                    $div = $item->firstChild;
                }
                // Finally, convert to a HTML string
                $data = trim((string) $document->saveHTML($div));

                if ($this->remove_div) {
                    $data = preg_replace('/^<div' . \SimplePie\SimplePie::PCRE_XML_ATTRIBUTE . '>/', '', $data);
                    // Cast for PHPStan, it is unable to validate a non-literal regex above.
                    $data = preg_replace('/<\/div>$/', '', (string) $data);
                } else {
                    $data = preg_replace('/^<div' . \SimplePie\SimplePie::PCRE_XML_ATTRIBUTE . '>/', '<div>', $data);
                }

                // Cast for PHPStan, it is unable to validate a non-literal regex above.
                $data = str_replace('</source>', '', (string) $data);
            }

            if ($type & \SimplePie\SimplePie::CONSTRUCT_IRI) {
                $absolute = $this->registry->call(Misc::class, 'absolutize_url', [$data, $base]);
                if ($absolute !== false) {
                    $data = $absolute;
                }
            }

            if ($type & (\SimplePie\SimplePie::CONSTRUCT_TEXT | \SimplePie\SimplePie::CONSTRUCT_IRI)) {
                $data = htmlspecialchars($data, ENT_COMPAT, 'UTF-8');
            }

            if ($this->output_encoding !== 'UTF-8') {
                // This really returns string|false but changing encoding is uncommon and we are going to deprecate it, so let’s just lie to PHPStan in the interest of cleaner annotations.
                /** @var string */
                $data = $this->registry->call(Misc::class, 'change_encoding', [$data, 'UTF-8', $this->output_encoding]);
            }
        }
        return $data;
    }

    /**
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @return string
     */
    protected function preprocess(string $html, int $type)
    {
        $ret = '';
        $html = preg_replace('%</?(?:html|body)[^>]*?'.'>%is', '', $html);
        if ($type & ~\SimplePie\SimplePie::CONSTRUCT_XHTML) {
            // Atom XHTML constructs are wrapped with a div by default
            // Note: No protection if $html contains a stray </div>!
            $html = '<div>' . $html . '</div>';
            $ret .= '<!DOCTYPE html>';
            $content_type = 'text/html';
        } else {
            $ret .= '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Strict//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd">';
            $content_type = 'application/xhtml+xml';
        }

        $ret .= '<html><head>';
        $ret .= '<meta http-equiv="Content-Type" content="' . $content_type . '; charset=utf-8" />';
        $ret .= '</head><body>' . $html . '</body></html>';
        return $ret;
    }

    /**
     * @param array<string>|string $attributes
     * @return void
     */
    public function replace_urls(DOMDocument $document, string $tag, $attributes)
    {
        if (!is_array($attributes)) {
            $attributes = [$attributes];
        }

        if (!is_array($this->strip_htmltags) || !in_array($tag, $this->strip_htmltags)) {
            $elements = $document->getElementsByTagName($tag);
            foreach ($elements as $element) {
                foreach ($attributes as $attribute) {
                    if ($element->hasAttribute($attribute)) {
                        $value = $this->registry->call(Misc::class, 'absolutize_url', [$element->getAttribute($attribute), $this->base]);
                        if ($value !== false) {
                            $value = $this->https_url($value);
                            $element->setAttribute($attribute, $value);
                        }
                    }
                }
            }
        }
    }

    /**
     * @param array<int, string> $match
     * @return string
     */
    public function do_strip_htmltags(array $match)
    {
        if ($this->encode_instead_of_strip) {
            if (isset($match[4]) && !in_array(strtolower($match[1]), ['script', 'style'])) {
                $match[1] = htmlspecialchars($match[1], ENT_COMPAT, 'UTF-8');
                $match[2] = htmlspecialchars($match[2], ENT_COMPAT, 'UTF-8');
                return "&lt;$match[1]$match[2]&gt;$match[3]&lt;/$match[1]&gt;";
            } else {
                return htmlspecialchars($match[0], ENT_COMPAT, 'UTF-8');
            }
        } elseif (isset($match[4]) && !in_array(strtolower($match[1]), ['script', 'style'])) {
            return $match[4];
        } else {
            return '';
        }
    }

    /**
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @return void
     */
    protected function strip_tag(string $tag, DOMDocument $document, DOMXPath $xpath, int $type)
    {
        $elements = $xpath->query('body//' . $tag);

        if ($elements === false) {
            throw new \SimplePie\Exception(sprintf(
                '%s(): Possibly malformed expression, check argument #1 ($tag)',
                __METHOD__
            ), 1);
        }

        if ($this->encode_instead_of_strip) {
            foreach ($elements as $element) {
                $fragment = $document->createDocumentFragment();

                // For elements which aren't script or style, include the tag itself
                if (!in_array($tag, ['script', 'style'])) {
                    $text = '<' . $tag;
                    if ($element->attributes !== null) {
                        $attrs = [];
                        foreach ($element->attributes as $name => $attr) {
                            $value = $attr->value;

                            // In XHTML, empty values should never exist, so we repeat the value
                            if (empty($value) && ($type & \SimplePie\SimplePie::CONSTRUCT_XHTML)) {
                                $value = $name;
                            }
                            // For HTML, empty is fine
                            elseif (empty($value) && ($type & \SimplePie\SimplePie::CONSTRUCT_HTML)) {
                                $attrs[] = $name;
                                continue;
                            }

                            // Standard attribute text
                            $attrs[] = $name . '="' . $attr->value . '"';
                        }
                        $text .= ' ' . implode(' ', $attrs);
                    }
                    $text .= '>';
                    $fragment->appendChild(new \DOMText($text));
                }

                $number = $element->childNodes->length;
                for ($i = $number; $i > 0; $i--) {
                    if (($child = $element->childNodes->item(0)) !== null) {
                        $fragment->appendChild($child);
                    }
                }

                if (!in_array($tag, ['script', 'style'])) {
                    $fragment->appendChild(new \DOMText('</' . $tag . '>'));
                }

                if (($parentNode = $element->parentNode) !== null) {
                    $parentNode->replaceChild($fragment, $element);
                }
            }

            return;
        } elseif (in_array($tag, ['script', 'style'])) {
            foreach ($elements as $element) {
                if (($parentNode = $element->parentNode) !== null) {
                    $parentNode->removeChild($element);
                }
            }

            return;
        } else {
            foreach ($elements as $element) {
                $fragment = $document->createDocumentFragment();
                $number = $element->childNodes->length;
                for ($i = $number; $i > 0; $i--) {
                    if (($child = $element->childNodes->item(0)) !== null) {
                        $fragment->appendChild($child);
                    }
                }

                if (($parentNode = $element->parentNode) !== null) {
                    $parentNode->replaceChild($fragment, $element);
                }
            }
        }
    }

    /**
     * @return void
     */
    protected function strip_attr(string $attrib, DOMXPath $xpath)
    {
        $elements = $xpath->query('//*[@' . $attrib . ']');

        if ($elements === false) {
            throw new \SimplePie\Exception(sprintf(
                '%s(): Possibly malformed expression, check argument #1 ($attrib)',
                __METHOD__
            ), 1);
        }

        /** @var \DOMElement $element */
        foreach ($elements as $element) {
            $element->removeAttribute($attrib);
        }
    }

    /**
     * @return void
     */
    protected function rename_attr(string $attrib, DOMXPath $xpath)
    {
        $elements = $xpath->query('//*[@' . $attrib . ']');

        if ($elements === false) {
            throw new \SimplePie\Exception(sprintf(
                '%s(): Possibly malformed expression, check argument #1 ($attrib)',
                __METHOD__
            ), 1);
        }

        /** @var \DOMElement $element */
        foreach ($elements as $element) {
            $element->setAttribute('data-sanitized-' . $attrib, $element->getAttribute($attrib));
            $element->removeAttribute($attrib);
        }
    }

    /**
     * @param array<string, string> $valuePairs
     * @return void
     */
    protected function add_attr(string $tag, array $valuePairs, DOMDocument $document)
    {
        $elements = $document->getElementsByTagName($tag);
        /** @var \DOMElement $element */
        foreach ($elements as $element) {
            foreach ($valuePairs as $attrib => $value) {
                $element->setAttribute($attrib, $value);
            }
        }
    }

    /**
     * Get a DataCache
     *
     * @param string $image_url Only needed for BC, can be removed in SimplePie 2.0.0
     *
     * @return DataCache
     */
    private function get_cache(string $image_url = ''): DataCache
    {
        if ($this->cache === null) {
            // @trigger_error(sprintf('Not providing as PSR-16 cache implementation is deprecated since SimplePie 1.8.0, please use "SimplePie\SimplePie::set_cache()".'), \E_USER_DEPRECATED);
            $cache = $this->registry->call(Cache::class, 'get_handler', [
                $this->cache_location,
                $image_url,
                Base::TYPE_IMAGE
            ]);

            return new BaseDataCache($cache);
        }

        return $this->cache;
    }

    /**
     * Get a HTTP client
     */
    private function get_http_client(): Client
    {
        if ($this->http_client === null) {
            $this->http_client = new FileClient(
                $this->registry,
                [
                    'timeout' => $this->timeout,
                    'redirects' => 5,
                    'useragent' => $this->useragent,
                    'force_fsockopen' => $this->force_fsockopen,
                    'curl_options' => $this->curl_options,
                ]
            );
        }

        return $this->http_client;
    }
}

class_alias('SimplePie\Sanitize', 'SimplePie_Sanitize');

}

namespace SimplePie {

/**
 * Handles `<atom:source>`
 *
 * Used by {@see \SimplePie\Item::get_source()}
 *
 * This class can be overloaded with {@see \SimplePie::set_source_class()}
 */
class Source implements RegistryAware
{
    /** @var Item */
    public $item;
    /** @var array<string, mixed> */
    public $data = [];
    /** @var Registry */
    protected $registry;

    /**
     * @param array<string, mixed> $data
     */
    public function __construct(Item $item, array $data)
    {
        $this->item = $item;
        $this->data = $data;
    }

    /**
     * @return void
     */
    public function set_registry(\SimplePie\Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * @return string
     */
    public function __toString()
    {
        return md5(serialize($this->data));
    }

    /**
     * @param string $namespace
     * @param string $tag
     * @return array<array<string, mixed>>|null
     */
    public function get_source_tags(string $namespace, string $tag)
    {
        if (isset($this->data['child'][$namespace][$tag])) {
            return $this->data['child'][$namespace][$tag];
        }

        return null;
    }

    /**
     * @param array<string, mixed> $element
     * @return string
     */
    public function get_base(array $element = [])
    {
        return $this->item->get_base($element);
    }

    /**
     * @param string $data
     * @param int-mask-of<SimplePie::CONSTRUCT_*> $type
     * @param string $base
     * @return string
     */
    public function sanitize(string $data, $type, string $base = '')
    {
        return $this->item->sanitize($data, $type, $base);
    }

    /**
     * @return Item
     */
    public function get_item()
    {
        return $this->item;
    }

    /**
     * @return string|null
     */
    public function get_title()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'title')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'title')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'title')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'title')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'title')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'title')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'title')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * @param int $key
     * @return Category|null
     */
    public function get_category(int $key = 0)
    {
        $categories = $this->get_categories();
        if (isset($categories[$key])) {
            return $categories[$key];
        }

        return null;
    }

    /**
     * @return array<Category>|null
     */
    public function get_categories()
    {
        $categories = [];

        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'category') as $category) {
            $term = null;
            $scheme = null;
            $label = null;
            if (isset($category['attribs']['']['term'])) {
                $term = $this->sanitize($category['attribs']['']['term'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['scheme'])) {
                $scheme = $this->sanitize($category['attribs']['']['scheme'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($category['attribs']['']['label'])) {
                $label = $this->sanitize($category['attribs']['']['label'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, $label]);
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'category') as $category) {
            // This is really the label, but keep this as the term also for BC.
            // Label will also work on retrieving because that falls back to term.
            $term = $this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            if (isset($category['attribs']['']['domain'])) {
                $scheme = $this->sanitize($category['attribs']['']['domain'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            } else {
                $scheme = null;
            }
            $categories[] = $this->registry->create(Category::class, [$term, $scheme, null]);
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'subject') as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'subject') as $category) {
            $categories[] = $this->registry->create(Category::class, [$this->sanitize($category['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }

        if (!empty($categories)) {
            return array_unique($categories);
        }

        return null;
    }

    /**
     * @param int $key
     * @return Author|null
     */
    public function get_author(int $key = 0)
    {
        $authors = $this->get_authors();
        if (isset($authors[$key])) {
            return $authors[$key];
        }

        return null;
    }

    /**
     * @return array<Author>|null
     */
    public function get_authors()
    {
        $authors = [];
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'author') as $author) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($author['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        if ($author = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'author')) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($author[0]['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $authors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'creator') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'author') as $author) {
            $authors[] = $this->registry->create(Author::class, [$this->sanitize($author['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT), null, null]);
        }

        if (!empty($authors)) {
            return array_unique($authors);
        }

        return null;
    }

    /**
     * @param int $key
     * @return Author|null
     */
    public function get_contributor(int $key = 0)
    {
        $contributors = $this->get_contributors();
        if (isset($contributors[$key])) {
            return $contributors[$key];
        }

        return null;
    }

    /**
     * @return array<Author>|null
     */
    public function get_contributors()
    {
        $contributors = [];
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'contributor') as $contributor) {
            $name = null;
            $uri = null;
            $email = null;
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0]['data'])) {
                $uri = $contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['uri'][0];
                $uri = $this->sanitize($uri['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($uri));
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_10]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $uri !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $uri, $email]);
            }
        }
        foreach ((array) $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'contributor') as $contributor) {
            $name = null;
            $url = null;
            $email = null;
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'])) {
                $name = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['name'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0]['data'])) {
                $url = $contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['url'][0];
                $url = $this->sanitize($url['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($url));
            }
            if (isset($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'])) {
                $email = $this->sanitize($contributor['child'][\SimplePie\SimplePie::NAMESPACE_ATOM_03]['email'][0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
            }
            if ($name !== null || $email !== null || $url !== null) {
                $contributors[] = $this->registry->create(Author::class, [$name, $url, $email]);
            }
        }

        if (!empty($contributors)) {
            return array_unique($contributors);
        }

        return null;
    }

    /**
     * @param int $key
     * @param string $rel
     * @return string|null
     */
    public function get_link(int $key = 0, string $rel = 'alternate')
    {
        $links = $this->get_links($rel);
        if (isset($links[$key])) {
            return $links[$key];
        }

        return null;
    }

    /**
     * Added for parity between the parent-level and the item/entry-level.
     *
     * @return string|null
     */
    public function get_permalink()
    {
        return $this->get_link(0);
    }

    /**
     * @param string $rel
     * @return array<string>|null
     */
    public function get_links(string $rel = 'alternate')
    {
        if (!isset($this->data['links'])) {
            $this->data['links'] = [];
            if ($links = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'link')) {
                foreach ($links as $link) {
                    if (isset($link['attribs']['']['href'])) {
                        $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                        $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($link));
                    }
                }
            }
            if ($links = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'link')) {
                foreach ($links as $link) {
                    if (isset($link['attribs']['']['href'])) {
                        $link_rel = (isset($link['attribs']['']['rel'])) ? $link['attribs']['']['rel'] : 'alternate';
                        $this->data['links'][$link_rel][] = $this->sanitize($link['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($link));
                    }
                }
            }
            if ($links = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($links[0]));
            }
            if ($links = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($links[0]));
            }
            if ($links = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'link')) {
                $this->data['links']['alternate'][] = $this->sanitize($links[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($links[0]));
            }

            $keys = array_keys($this->data['links']);
            foreach ($keys as $key) {
                $key = (string) $key;

                if ($this->registry->call(Misc::class, 'is_isegment_nz_nc', [$key])) {
                    if (isset($this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key])) {
                        $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key] = array_merge($this->data['links'][$key], $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key]);
                        $this->data['links'][$key] = &$this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key];
                    } else {
                        $this->data['links'][\SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY . $key] = &$this->data['links'][$key];
                    }
                } elseif (substr($key, 0, 41) === \SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY) {
                    $this->data['links'][substr($key, 41)] = &$this->data['links'][$key];
                }
                $this->data['links'][$key] = array_unique($this->data['links'][$key]);
            }
        }

        if (isset($this->data['links'][$rel])) {
            return $this->data['links'][$rel];
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function get_description()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'subtitle')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'tagline')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_10, 'description')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_090, 'description')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'description')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'description')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'description')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'summary')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'subtitle')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_HTML, $this->get_base($return[0]));
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function get_copyright()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'rights')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_10_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_03, 'copyright')) {
            return $this->sanitize($return[0]['data'], $this->registry->call(Misc::class, 'atom_03_construct_type', [$return[0]['attribs']]), $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'copyright')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'rights')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'rights')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function get_language()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_RSS_20, 'language')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_11, 'language')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_DC_10, 'language')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        } elseif (isset($this->data['xml_lang'])) {
            return $this->sanitize($this->data['xml_lang'], \SimplePie\SimplePie::CONSTRUCT_TEXT);
        }

        return null;
    }

    /**
     * @return float|null
     */
    public function get_latitude()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'lat')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[1];
        }

        return null;
    }

    /**
     * @return float|null
     */
    public function get_longitude()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'long')) {
            return (float) $return[0]['data'];
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO, 'lon')) {
            return (float) $return[0]['data'];
        } elseif (($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_GEORSS, 'point')) && preg_match('/^((?:-)?[0-9]+(?:\.[0-9]+)) ((?:-)?[0-9]+(?:\.[0-9]+))$/', trim($return[0]['data']), $match)) {
            return (float) $match[2];
        }

        return null;
    }

    /**
     * @return string|null
     */
    public function get_image_url()
    {
        if ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ITUNES, 'image')) {
            return $this->sanitize($return[0]['attribs']['']['href'], \SimplePie\SimplePie::CONSTRUCT_IRI);
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'logo')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($return[0]));
        } elseif ($return = $this->get_source_tags(\SimplePie\SimplePie::NAMESPACE_ATOM_10, 'icon')) {
            return $this->sanitize($return[0]['data'], \SimplePie\SimplePie::CONSTRUCT_IRI, $this->get_base($return[0]));
        }

        return null;
    }
}

class_alias('SimplePie\Source', 'SimplePie_Source');

}

namespace SimplePie\XML\Declaration {

/**
 * Parses the XML Declaration
 */
class Parser
{
    /**
     * XML Version
     *
     * @access public
     * @var string
     */
    public $version = '1.0';

    /**
     * Encoding
     *
     * @access public
     * @var string
     */
    public $encoding = 'UTF-8';

    /**
     * Standalone
     *
     * @access public
     * @var bool
     */
    public $standalone = false;

    private const STATE_BEFORE_VERSION_NAME = 'before_version_name';

    private const STATE_VERSION_NAME = 'version_name';

    private const STATE_VERSION_EQUALS = 'version_equals';

    private const STATE_VERSION_VALUE = 'version_value';

    private const STATE_ENCODING_NAME = 'encoding_name';

    private const STATE_EMIT = 'emit';

    private const STATE_ENCODING_EQUALS = 'encoding_equals';

    private const STATE_STANDALONE_NAME = 'standalone_name';

    private const STATE_ENCODING_VALUE = 'encoding_value';

    private const STATE_STANDALONE_EQUALS = 'standalone_equals';

    private const STATE_STANDALONE_VALUE = 'standalone_value';

    private const STATE_ERROR = false;

    /**
     * Current state of the state machine
     *
     * @access private
     * @var self::STATE_*
     */
    public $state = self::STATE_BEFORE_VERSION_NAME;

    /**
     * Input data
     *
     * @access private
     * @var string
     */
    public $data = '';

    /**
     * Input data length (to avoid calling strlen() everytime this is needed)
     *
     * @access private
     * @var int
     */
    public $data_length = 0;

    /**
     * Current position of the pointer
     *
     * @var int
     * @access private
     */
    public $position = 0;

    /**
     * Create an instance of the class with the input data
     *
     * @access public
     * @param string $data Input data
     */
    public function __construct(string $data)
    {
        $this->data = $data;
        $this->data_length = strlen($this->data);
    }

    /**
     * Parse the input data
     *
     * @access public
     * @return bool true on success, false on failure
     */
    public function parse(): bool
    {
        while ($this->state && $this->state !== self::STATE_EMIT && $this->has_data()) {
            $state = $this->state;
            $this->$state();
        }
        $this->data = '';
        if ($this->state === self::STATE_EMIT) {
            return true;
        }

        // Reset the parser state.
        $this->version = '1.0';
        $this->encoding = 'UTF-8';
        $this->standalone = false;
        return false;
    }

    /**
     * Check whether there is data beyond the pointer
     *
     * @access private
     * @return bool true if there is further data, false if not
     */
    public function has_data(): bool
    {
        return (bool) ($this->position < $this->data_length);
    }

    /**
     * Advance past any whitespace
     *
     * @return int Number of whitespace characters passed
     */
    public function skip_whitespace()
    {
        $whitespace = strspn($this->data, "\x09\x0A\x0D\x20", $this->position);
        $this->position += $whitespace;
        return $whitespace;
    }

    /**
     * Read value
     *
     * @return string|false
     */
    public function get_value()
    {
        $quote = substr($this->data, $this->position, 1);
        if ($quote === '"' || $quote === "'") {
            $this->position++;
            $len = strcspn($this->data, $quote, $this->position);
            if ($this->has_data()) {
                $value = substr($this->data, $this->position, $len);
                $this->position += $len + 1;
                return $value;
            }
        }
        return false;
    }

    public function before_version_name(): void
    {
        if ($this->skip_whitespace()) {
            $this->state = self::STATE_VERSION_NAME;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function version_name(): void
    {
        if (substr($this->data, $this->position, 7) === 'version') {
            $this->position += 7;
            $this->skip_whitespace();
            $this->state = self::STATE_VERSION_EQUALS;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function version_equals(): void
    {
        if (substr($this->data, $this->position, 1) === '=') {
            $this->position++;
            $this->skip_whitespace();
            $this->state = self::STATE_VERSION_VALUE;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function version_value(): void
    {
        if ($version = $this->get_value()) {
            $this->version = $version;
            $this->skip_whitespace();
            if ($this->has_data()) {
                $this->state = self::STATE_ENCODING_NAME;
            } else {
                $this->state = self::STATE_EMIT;
            }
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function encoding_name(): void
    {
        if (substr($this->data, $this->position, 8) === 'encoding') {
            $this->position += 8;
            $this->skip_whitespace();
            $this->state = self::STATE_ENCODING_EQUALS;
        } else {
            $this->state = self::STATE_STANDALONE_NAME;
        }
    }

    public function encoding_equals(): void
    {
        if (substr($this->data, $this->position, 1) === '=') {
            $this->position++;
            $this->skip_whitespace();
            $this->state = self::STATE_ENCODING_VALUE;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function encoding_value(): void
    {
        if ($encoding = $this->get_value()) {
            $this->encoding = $encoding;
            $this->skip_whitespace();
            if ($this->has_data()) {
                $this->state = self::STATE_STANDALONE_NAME;
            } else {
                $this->state = self::STATE_EMIT;
            }
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function standalone_name(): void
    {
        if (substr($this->data, $this->position, 10) === 'standalone') {
            $this->position += 10;
            $this->skip_whitespace();
            $this->state = self::STATE_STANDALONE_EQUALS;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function standalone_equals(): void
    {
        if (substr($this->data, $this->position, 1) === '=') {
            $this->position++;
            $this->skip_whitespace();
            $this->state = self::STATE_STANDALONE_VALUE;
        } else {
            $this->state = self::STATE_ERROR;
        }
    }

    public function standalone_value(): void
    {
        if ($standalone = $this->get_value()) {
            switch ($standalone) {
                case 'yes':
                    $this->standalone = true;
                    break;

                case 'no':
                    $this->standalone = false;
                    break;

                default:
                    $this->state = self::STATE_ERROR;
                    return;
            }

            $this->skip_whitespace();
            if ($this->has_data()) {
                $this->state = self::STATE_ERROR;
            } else {
                $this->state = self::STATE_EMIT;
            }
        } else {
            $this->state = self::STATE_ERROR;
        }
    }
}

class_alias('SimplePie\XML\Declaration\Parser', 'SimplePie_XML_Declaration_Parser');

}

namespace {

use SimplePie\Misc;

class_exists('SimplePie\Misc');

// @trigger_error(sprintf('Using the "SimplePie_Misc" class is deprecated since SimplePie 1.7.0, use "SimplePie\Misc" instead.'), \E_USER_DEPRECATED);

/** @phpstan-ignore-next-line */
if (\false) {
    /** @deprecated since SimplePie 1.7.0, use "SimplePie\Misc" instead */
    class SimplePie_Misc extends Misc
    {
    }
}

}

namespace {

/**
 * SimplePie class.
 *
 * Class for backward compatibility.
 *
 * @deprecated Use {@see SimplePie} directly
 */
class SimplePie_Core extends SimplePie
{
}

}

namespace {

/**
 * Decode HTML Entities
 *
 * This implements HTML5 as of revision 967 (2007-06-28)
 *
 * @deprecated Use DOMDocument instead!
 */
class SimplePie_Decode_HTML_Entities
{
    /**
     * Data to be parsed
     *
     * @access private
     * @var string
     */
    public $data = '';

    /**
     * Currently consumed bytes
     *
     * @access private
     * @var string
     */
    public $consumed = '';

    /**
     * Position of the current byte being parsed
     *
     * @access private
     * @var int
     */
    public $position = 0;

    /**
     * Create an instance of the class with the input data
     *
     * @access public
     * @param string $data Input data
     */
    public function __construct(string $data)
    {
        $this->data = $data;
    }

    /**
     * Parse the input data
     *
     * @access public
     * @return string Output data
     */
    public function parse()
    {
        while (($position = strpos($this->data, '&', $this->position)) !== false) {
            $this->position = $position;
            $this->consume();
            $this->entity();
            $this->consumed = '';
        }
        return $this->data;
    }

    /**
     * Consume the next byte
     *
     * @access private
     * @return string|false The next byte, or false, if there is no more data
     */
    public function consume()
    {
        if (isset($this->data[$this->position])) {
            $this->consumed .= $this->data[$this->position];
            return $this->data[$this->position++];
        }

        return false;
    }

    /**
     * Consume a range of characters
     *
     * @access private
     * @param string $chars Characters to consume
     * @return string|false A series of characters that match the range, or false
     */
    public function consume_range(string $chars)
    {
        if ($len = strspn($this->data, $chars, $this->position)) {
            $data = substr($this->data, $this->position, $len);
            $this->consumed .= $data;
            $this->position += $len;
            return $data;
        }

        return false;
    }

    /**
     * Unconsume one byte
     *
     * @access private
     * @return void
     */
    public function unconsume()
    {
        $this->consumed = substr($this->consumed, 0, -1);
        $this->position--;
    }

    /**
     * Decode an entity
     *
     * @access private
     * @return void
     */
    public function entity()
    {
        switch ($this->consume()) {
            case "\x09":
            case "\x0A":
            case "\x0B":
            case "\x0C":
            case "\x20":
            case "\x3C":
            case "\x26":
            case false:
                break;

            case "\x23":
                switch ($this->consume()) {
                    case "\x78":
                    case "\x58":
                        $range = '0123456789ABCDEFabcdef';
                        $hex = true;
                        break;

                    default:
                        $range = '0123456789';
                        $hex = false;
                        $this->unconsume();
                        break;
                }

                if ($codepoint = $this->consume_range($range)) {
                    static $windows_1252_specials = [0x0D => "\x0A", 0x80 => "\xE2\x82\xAC", 0x81 => "\xEF\xBF\xBD", 0x82 => "\xE2\x80\x9A", 0x83 => "\xC6\x92", 0x84 => "\xE2\x80\x9E", 0x85 => "\xE2\x80\xA6", 0x86 => "\xE2\x80\xA0", 0x87 => "\xE2\x80\xA1", 0x88 => "\xCB\x86", 0x89 => "\xE2\x80\xB0", 0x8A => "\xC5\xA0", 0x8B => "\xE2\x80\xB9", 0x8C => "\xC5\x92", 0x8D => "\xEF\xBF\xBD", 0x8E => "\xC5\xBD", 0x8F => "\xEF\xBF\xBD", 0x90 => "\xEF\xBF\xBD", 0x91 => "\xE2\x80\x98", 0x92 => "\xE2\x80\x99", 0x93 => "\xE2\x80\x9C", 0x94 => "\xE2\x80\x9D", 0x95 => "\xE2\x80\xA2", 0x96 => "\xE2\x80\x93", 0x97 => "\xE2\x80\x94", 0x98 => "\xCB\x9C", 0x99 => "\xE2\x84\xA2", 0x9A => "\xC5\xA1", 0x9B => "\xE2\x80\xBA", 0x9C => "\xC5\x93", 0x9D => "\xEF\xBF\xBD", 0x9E => "\xC5\xBE", 0x9F => "\xC5\xB8"];

                    if ($hex) {
                        // Cap to PHP_INT_MAX to ensure consistent behaviour if $codepoint is so large
                        // it cannot fit into int – just casting float to int might return junk (e.g. a negative number).
                        // If it is so large, `Misc::codepoint_to_utf8` will just return a replacement character.
                        $codepoint = (int) min(hexdec($codepoint), \PHP_INT_MAX);
                    } else {
                        // Casting string to int caps at PHP_INT_MAX automatically.
                        $codepoint = (int) $codepoint;
                    }

                    if (isset($windows_1252_specials[$codepoint])) {
                        $replacement = $windows_1252_specials[$codepoint];
                    } else {
                        $replacement = SimplePie_Misc::codepoint_to_utf8($codepoint);
                    }

                    if (!in_array($this->consume(), [';', false], true)) {
                        $this->unconsume();
                    }

                    $consumed_length = strlen($this->consumed);
                    $this->data = substr_replace($this->data, $replacement, $this->position - $consumed_length, $consumed_length);
                    $this->position += strlen($replacement) - $consumed_length;
                }
                break;

            default:
                static $entities = [
                    'Aacute' => "\xC3\x81",
                    'aacute' => "\xC3\xA1",
                    'Aacute;' => "\xC3\x81",
                    'aacute;' => "\xC3\xA1",
                    'Acirc' => "\xC3\x82",
                    'acirc' => "\xC3\xA2",
                    'Acirc;' => "\xC3\x82",
                    'acirc;' => "\xC3\xA2",
                    'acute' => "\xC2\xB4",
                    'acute;' => "\xC2\xB4",
                    'AElig' => "\xC3\x86",
                    'aelig' => "\xC3\xA6",
                    'AElig;' => "\xC3\x86",
                    'aelig;' => "\xC3\xA6",
                    'Agrave' => "\xC3\x80",
                    'agrave' => "\xC3\xA0",
                    'Agrave;' => "\xC3\x80",
                    'agrave;' => "\xC3\xA0",
                    'alefsym;' => "\xE2\x84\xB5",
                    'Alpha;' => "\xCE\x91",
                    'alpha;' => "\xCE\xB1",
                    'AMP' => "\x26",
                    'amp' => "\x26",
                    'AMP;' => "\x26",
                    'amp;' => "\x26",
                    'and;' => "\xE2\x88\xA7",
                    'ang;' => "\xE2\x88\xA0",
                    'apos;' => "\x27",
                    'Aring' => "\xC3\x85",
                    'aring' => "\xC3\xA5",
                    'Aring;' => "\xC3\x85",
                    'aring;' => "\xC3\xA5",
                    'asymp;' => "\xE2\x89\x88",
                    'Atilde' => "\xC3\x83",
                    'atilde' => "\xC3\xA3",
                    'Atilde;' => "\xC3\x83",
                    'atilde;' => "\xC3\xA3",
                    'Auml' => "\xC3\x84",
                    'auml' => "\xC3\xA4",
                    'Auml;' => "\xC3\x84",
                    'auml;' => "\xC3\xA4",
                    'bdquo;' => "\xE2\x80\x9E",
                    'Beta;' => "\xCE\x92",
                    'beta;' => "\xCE\xB2",
                    'brvbar' => "\xC2\xA6",
                    'brvbar;' => "\xC2\xA6",
                    'bull;' => "\xE2\x80\xA2",
                    'cap;' => "\xE2\x88\xA9",
                    'Ccedil' => "\xC3\x87",
                    'ccedil' => "\xC3\xA7",
                    'Ccedil;' => "\xC3\x87",
                    'ccedil;' => "\xC3\xA7",
                    'cedil' => "\xC2\xB8",
                    'cedil;' => "\xC2\xB8",
                    'cent' => "\xC2\xA2",
                    'cent;' => "\xC2\xA2",
                    'Chi;' => "\xCE\xA7",
                    'chi;' => "\xCF\x87",
                    'circ;' => "\xCB\x86",
                    'clubs;' => "\xE2\x99\xA3",
                    'cong;' => "\xE2\x89\x85",
                    'COPY' => "\xC2\xA9",
                    'copy' => "\xC2\xA9",
                    'COPY;' => "\xC2\xA9",
                    'copy;' => "\xC2\xA9",
                    'crarr;' => "\xE2\x86\xB5",
                    'cup;' => "\xE2\x88\xAA",
                    'curren' => "\xC2\xA4",
                    'curren;' => "\xC2\xA4",
                    'Dagger;' => "\xE2\x80\xA1",
                    'dagger;' => "\xE2\x80\xA0",
                    'dArr;' => "\xE2\x87\x93",
                    'darr;' => "\xE2\x86\x93",
                    'deg' => "\xC2\xB0",
                    'deg;' => "\xC2\xB0",
                    'Delta;' => "\xCE\x94",
                    'delta;' => "\xCE\xB4",
                    'diams;' => "\xE2\x99\xA6",
                    'divide' => "\xC3\xB7",
                    'divide;' => "\xC3\xB7",
                    'Eacute' => "\xC3\x89",
                    'eacute' => "\xC3\xA9",
                    'Eacute;' => "\xC3\x89",
                    'eacute;' => "\xC3\xA9",
                    'Ecirc' => "\xC3\x8A",
                    'ecirc' => "\xC3\xAA",
                    'Ecirc;' => "\xC3\x8A",
                    'ecirc;' => "\xC3\xAA",
                    'Egrave' => "\xC3\x88",
                    'egrave' => "\xC3\xA8",
                    'Egrave;' => "\xC3\x88",
                    'egrave;' => "\xC3\xA8",
                    'empty;' => "\xE2\x88\x85",
                    'emsp;' => "\xE2\x80\x83",
                    'ensp;' => "\xE2\x80\x82",
                    'Epsilon;' => "\xCE\x95",
                    'epsilon;' => "\xCE\xB5",
                    'equiv;' => "\xE2\x89\xA1",
                    'Eta;' => "\xCE\x97",
                    'eta;' => "\xCE\xB7",
                    'ETH' => "\xC3\x90",
                    'eth' => "\xC3\xB0",
                    'ETH;' => "\xC3\x90",
                    'eth;' => "\xC3\xB0",
                    'Euml' => "\xC3\x8B",
                    'euml' => "\xC3\xAB",
                    'Euml;' => "\xC3\x8B",
                    'euml;' => "\xC3\xAB",
                    'euro;' => "\xE2\x82\xAC",
                    'exist;' => "\xE2\x88\x83",
                    'fnof;' => "\xC6\x92",
                    'forall;' => "\xE2\x88\x80",
                    'frac12' => "\xC2\xBD",
                    'frac12;' => "\xC2\xBD",
                    'frac14' => "\xC2\xBC",
                    'frac14;' => "\xC2\xBC",
                    'frac34' => "\xC2\xBE",
                    'frac34;' => "\xC2\xBE",
                    'frasl;' => "\xE2\x81\x84",
                    'Gamma;' => "\xCE\x93",
                    'gamma;' => "\xCE\xB3",
                    'ge;' => "\xE2\x89\xA5",
                    'GT' => "\x3E",
                    'gt' => "\x3E",
                    'GT;' => "\x3E",
                    'gt;' => "\x3E",
                    'hArr;' => "\xE2\x87\x94",
                    'harr;' => "\xE2\x86\x94",
                    'hearts;' => "\xE2\x99\xA5",
                    'hellip;' => "\xE2\x80\xA6",
                    'Iacute' => "\xC3\x8D",
                    'iacute' => "\xC3\xAD",
                    'Iacute;' => "\xC3\x8D",
                    'iacute;' => "\xC3\xAD",
                    'Icirc' => "\xC3\x8E",
                    'icirc' => "\xC3\xAE",
                    'Icirc;' => "\xC3\x8E",
                    'icirc;' => "\xC3\xAE",
                    'iexcl' => "\xC2\xA1",
                    'iexcl;' => "\xC2\xA1",
                    'Igrave' => "\xC3\x8C",
                    'igrave' => "\xC3\xAC",
                    'Igrave;' => "\xC3\x8C",
                    'igrave;' => "\xC3\xAC",
                    'image;' => "\xE2\x84\x91",
                    'infin;' => "\xE2\x88\x9E",
                    'int;' => "\xE2\x88\xAB",
                    'Iota;' => "\xCE\x99",
                    'iota;' => "\xCE\xB9",
                    'iquest' => "\xC2\xBF",
                    'iquest;' => "\xC2\xBF",
                    'isin;' => "\xE2\x88\x88",
                    'Iuml' => "\xC3\x8F",
                    'iuml' => "\xC3\xAF",
                    'Iuml;' => "\xC3\x8F",
                    'iuml;' => "\xC3\xAF",
                    'Kappa;' => "\xCE\x9A",
                    'kappa;' => "\xCE\xBA",
                    'Lambda;' => "\xCE\x9B",
                    'lambda;' => "\xCE\xBB",
                    'lang;' => "\xE3\x80\x88",
                    'laquo' => "\xC2\xAB",
                    'laquo;' => "\xC2\xAB",
                    'lArr;' => "\xE2\x87\x90",
                    'larr;' => "\xE2\x86\x90",
                    'lceil;' => "\xE2\x8C\x88",
                    'ldquo;' => "\xE2\x80\x9C",
                    'le;' => "\xE2\x89\xA4",
                    'lfloor;' => "\xE2\x8C\x8A",
                    'lowast;' => "\xE2\x88\x97",
                    'loz;' => "\xE2\x97\x8A",
                    'lrm;' => "\xE2\x80\x8E",
                    'lsaquo;' => "\xE2\x80\xB9",
                    'lsquo;' => "\xE2\x80\x98",
                    'LT' => "\x3C",
                    'lt' => "\x3C",
                    'LT;' => "\x3C",
                    'lt;' => "\x3C",
                    'macr' => "\xC2\xAF",
                    'macr;' => "\xC2\xAF",
                    'mdash;' => "\xE2\x80\x94",
                    'micro' => "\xC2\xB5",
                    'micro;' => "\xC2\xB5",
                    'middot' => "\xC2\xB7",
                    'middot;' => "\xC2\xB7",
                    'minus;' => "\xE2\x88\x92",
                    'Mu;' => "\xCE\x9C",
                    'mu;' => "\xCE\xBC",
                    'nabla;' => "\xE2\x88\x87",
                    'nbsp' => "\xC2\xA0",
                    'nbsp;' => "\xC2\xA0",
                    'ndash;' => "\xE2\x80\x93",
                    'ne;' => "\xE2\x89\xA0",
                    'ni;' => "\xE2\x88\x8B",
                    'not' => "\xC2\xAC",
                    'not;' => "\xC2\xAC",
                    'notin;' => "\xE2\x88\x89",
                    'nsub;' => "\xE2\x8A\x84",
                    'Ntilde' => "\xC3\x91",
                    'ntilde' => "\xC3\xB1",
                    'Ntilde;' => "\xC3\x91",
                    'ntilde;' => "\xC3\xB1",
                    'Nu;' => "\xCE\x9D",
                    'nu;' => "\xCE\xBD",
                    'Oacute' => "\xC3\x93",
                    'oacute' => "\xC3\xB3",
                    'Oacute;' => "\xC3\x93",
                    'oacute;' => "\xC3\xB3",
                    'Ocirc' => "\xC3\x94",
                    'ocirc' => "\xC3\xB4",
                    'Ocirc;' => "\xC3\x94",
                    'ocirc;' => "\xC3\xB4",
                    'OElig;' => "\xC5\x92",
                    'oelig;' => "\xC5\x93",
                    'Ograve' => "\xC3\x92",
                    'ograve' => "\xC3\xB2",
                    'Ograve;' => "\xC3\x92",
                    'ograve;' => "\xC3\xB2",
                    'oline;' => "\xE2\x80\xBE",
                    'Omega;' => "\xCE\xA9",
                    'omega;' => "\xCF\x89",
                    'Omicron;' => "\xCE\x9F",
                    'omicron;' => "\xCE\xBF",
                    'oplus;' => "\xE2\x8A\x95",
                    'or;' => "\xE2\x88\xA8",
                    'ordf' => "\xC2\xAA",
                    'ordf;' => "\xC2\xAA",
                    'ordm' => "\xC2\xBA",
                    'ordm;' => "\xC2\xBA",
                    'Oslash' => "\xC3\x98",
                    'oslash' => "\xC3\xB8",
                    'Oslash;' => "\xC3\x98",
                    'oslash;' => "\xC3\xB8",
                    'Otilde' => "\xC3\x95",
                    'otilde' => "\xC3\xB5",
                    'Otilde;' => "\xC3\x95",
                    'otilde;' => "\xC3\xB5",
                    'otimes;' => "\xE2\x8A\x97",
                    'Ouml' => "\xC3\x96",
                    'ouml' => "\xC3\xB6",
                    'Ouml;' => "\xC3\x96",
                    'ouml;' => "\xC3\xB6",
                    'para' => "\xC2\xB6",
                    'para;' => "\xC2\xB6",
                    'part;' => "\xE2\x88\x82",
                    'permil;' => "\xE2\x80\xB0",
                    'perp;' => "\xE2\x8A\xA5",
                    'Phi;' => "\xCE\xA6",
                    'phi;' => "\xCF\x86",
                    'Pi;' => "\xCE\xA0",
                    'pi;' => "\xCF\x80",
                    'piv;' => "\xCF\x96",
                    'plusmn' => "\xC2\xB1",
                    'plusmn;' => "\xC2\xB1",
                    'pound' => "\xC2\xA3",
                    'pound;' => "\xC2\xA3",
                    'Prime;' => "\xE2\x80\xB3",
                    'prime;' => "\xE2\x80\xB2",
                    'prod;' => "\xE2\x88\x8F",
                    'prop;' => "\xE2\x88\x9D",
                    'Psi;' => "\xCE\xA8",
                    'psi;' => "\xCF\x88",
                    'QUOT' => "\x22",
                    'quot' => "\x22",
                    'QUOT;' => "\x22",
                    'quot;' => "\x22",
                    'radic;' => "\xE2\x88\x9A",
                    'rang;' => "\xE3\x80\x89",
                    'raquo' => "\xC2\xBB",
                    'raquo;' => "\xC2\xBB",
                    'rArr;' => "\xE2\x87\x92",
                    'rarr;' => "\xE2\x86\x92",
                    'rceil;' => "\xE2\x8C\x89",
                    'rdquo;' => "\xE2\x80\x9D",
                    'real;' => "\xE2\x84\x9C",
                    'REG' => "\xC2\xAE",
                    'reg' => "\xC2\xAE",
                    'REG;' => "\xC2\xAE",
                    'reg;' => "\xC2\xAE",
                    'rfloor;' => "\xE2\x8C\x8B",
                    'Rho;' => "\xCE\xA1",
                    'rho;' => "\xCF\x81",
                    'rlm;' => "\xE2\x80\x8F",
                    'rsaquo;' => "\xE2\x80\xBA",
                    'rsquo;' => "\xE2\x80\x99",
                    'sbquo;' => "\xE2\x80\x9A",
                    'Scaron;' => "\xC5\xA0",
                    'scaron;' => "\xC5\xA1",
                    'sdot;' => "\xE2\x8B\x85",
                    'sect' => "\xC2\xA7",
                    'sect;' => "\xC2\xA7",
                    'shy' => "\xC2\xAD",
                    'shy;' => "\xC2\xAD",
                    'Sigma;' => "\xCE\xA3",
                    'sigma;' => "\xCF\x83",
                    'sigmaf;' => "\xCF\x82",
                    'sim;' => "\xE2\x88\xBC",
                    'spades;' => "\xE2\x99\xA0",
                    'sub;' => "\xE2\x8A\x82",
                    'sube;' => "\xE2\x8A\x86",
                    'sum;' => "\xE2\x88\x91",
                    'sup;' => "\xE2\x8A\x83",
                    'sup1' => "\xC2\xB9",
                    'sup1;' => "\xC2\xB9",
                    'sup2' => "\xC2\xB2",
                    'sup2;' => "\xC2\xB2",
                    'sup3' => "\xC2\xB3",
                    'sup3;' => "\xC2\xB3",
                    'supe;' => "\xE2\x8A\x87",
                    'szlig' => "\xC3\x9F",
                    'szlig;' => "\xC3\x9F",
                    'Tau;' => "\xCE\xA4",
                    'tau;' => "\xCF\x84",
                    'there4;' => "\xE2\x88\xB4",
                    'Theta;' => "\xCE\x98",
                    'theta;' => "\xCE\xB8",
                    'thetasym;' => "\xCF\x91",
                    'thinsp;' => "\xE2\x80\x89",
                    'THORN' => "\xC3\x9E",
                    'thorn' => "\xC3\xBE",
                    'THORN;' => "\xC3\x9E",
                    'thorn;' => "\xC3\xBE",
                    'tilde;' => "\xCB\x9C",
                    'times' => "\xC3\x97",
                    'times;' => "\xC3\x97",
                    'TRADE;' => "\xE2\x84\xA2",
                    'trade;' => "\xE2\x84\xA2",
                    'Uacute' => "\xC3\x9A",
                    'uacute' => "\xC3\xBA",
                    'Uacute;' => "\xC3\x9A",
                    'uacute;' => "\xC3\xBA",
                    'uArr;' => "\xE2\x87\x91",
                    'uarr;' => "\xE2\x86\x91",
                    'Ucirc' => "\xC3\x9B",
                    'ucirc' => "\xC3\xBB",
                    'Ucirc;' => "\xC3\x9B",
                    'ucirc;' => "\xC3\xBB",
                    'Ugrave' => "\xC3\x99",
                    'ugrave' => "\xC3\xB9",
                    'Ugrave;' => "\xC3\x99",
                    'ugrave;' => "\xC3\xB9",
                    'uml' => "\xC2\xA8",
                    'uml;' => "\xC2\xA8",
                    'upsih;' => "\xCF\x92",
                    'Upsilon;' => "\xCE\xA5",
                    'upsilon;' => "\xCF\x85",
                    'Uuml' => "\xC3\x9C",
                    'uuml' => "\xC3\xBC",
                    'Uuml;' => "\xC3\x9C",
                    'uuml;' => "\xC3\xBC",
                    'weierp;' => "\xE2\x84\x98",
                    'Xi;' => "\xCE\x9E",
                    'xi;' => "\xCE\xBE",
                    'Yacute' => "\xC3\x9D",
                    'yacute' => "\xC3\xBD",
                    'Yacute;' => "\xC3\x9D",
                    'yacute;' => "\xC3\xBD",
                    'yen' => "\xC2\xA5",
                    'yen;' => "\xC2\xA5",
                    'yuml' => "\xC3\xBF",
                    'Yuml;' => "\xC5\xB8",
                    'yuml;' => "\xC3\xBF",
                    'Zeta;' => "\xCE\x96",
                    'zeta;' => "\xCE\xB6",
                    'zwj;' => "\xE2\x80\x8D",
                    'zwnj;' => "\xE2\x80\x8C"
                ];

                for ($i = 0, $match = null; $i < 9 && $this->consume() !== false; $i++) {
                    // Cast for PHPStan on PHP < 8.0: We consumed as per the loop condition,
                    // so `$this->consumed` is non-empty and the substr offset is valid.
                    $consumed = (string) substr($this->consumed, 1);
                    if (isset($entities[$consumed])) {
                        $match = $consumed;
                    }
                }

                if ($match !== null) {
                    $this->data = substr_replace($this->data, $entities[$match], $this->position - strlen($consumed) - 1, strlen($match) + 1);
                    $this->position += strlen($entities[$match]) - strlen($consumed) - 1;
                }
                break;
        }
    }
}

}

namespace {

use SimplePie\SimplePie as NamespacedSimplePie;

class_exists('SimplePie\SimplePie');

// @trigger_error(sprintf('Using the "SimplePie" class is deprecated since SimplePie 1.7.0, use "SimplePie\SimplePie" instead.'), \E_USER_DEPRECATED);

/** @phpstan-ignore-next-line */
if (\false) {
    /** @deprecated since SimplePie 1.7.0, use "SimplePie\SimplePie" instead */
    class SimplePie extends NamespacedSimplePie
    {
    }
}

/**
 * SimplePie Name
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAME instead.
 */
define('SIMPLEPIE_NAME', NamespacedSimplePie::NAME);

/**
 * SimplePie Version
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::VERSION instead.
 */
define('SIMPLEPIE_VERSION', NamespacedSimplePie::VERSION);

/**
 * SimplePie Build
 * @todo Hardcode for release (there's no need to have to call SimplePie_Misc::get_build() only every load of simplepie.inc)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\Misc::get_build() instead.
 */
define('SIMPLEPIE_BUILD', '20250912063641');

/**
 * SimplePie Website URL
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::URL instead.
 */
define('SIMPLEPIE_URL', NamespacedSimplePie::URL);

/**
 * SimplePie Useragent
 * @see SimplePie::set_useragent()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\Misc::get_default_useragent() instead.
 */
define('SIMPLEPIE_USERAGENT', \SimplePie\Misc::get_default_useragent());

/**
 * SimplePie Linkback
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LINKBACK instead.
 */
define('SIMPLEPIE_LINKBACK', NamespacedSimplePie::LINKBACK);

/**
 * No Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_NONE instead.
 */
define('SIMPLEPIE_LOCATOR_NONE', NamespacedSimplePie::LOCATOR_NONE);

/**
 * Feed Link Element Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_AUTODISCOVERY instead.
 */
define('SIMPLEPIE_LOCATOR_AUTODISCOVERY', NamespacedSimplePie::LOCATOR_AUTODISCOVERY);

/**
 * Local Feed Extension Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_LOCAL_EXTENSION instead.
 */
define('SIMPLEPIE_LOCATOR_LOCAL_EXTENSION', NamespacedSimplePie::LOCATOR_LOCAL_EXTENSION);

/**
 * Local Feed Body Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_LOCAL_BODY instead.
 */
define('SIMPLEPIE_LOCATOR_LOCAL_BODY', NamespacedSimplePie::LOCATOR_LOCAL_BODY);

/**
 * Remote Feed Extension Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_REMOTE_EXTENSION instead.
 */
define('SIMPLEPIE_LOCATOR_REMOTE_EXTENSION', NamespacedSimplePie::LOCATOR_REMOTE_EXTENSION);

/**
 * Remote Feed Body Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_REMOTE_BODY instead.
 */
define('SIMPLEPIE_LOCATOR_REMOTE_BODY', NamespacedSimplePie::LOCATOR_REMOTE_BODY);

/**
 * All Feed Autodiscovery
 * @see SimplePie::set_autodiscovery_level()
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOCATOR_ALL instead.
 */
define('SIMPLEPIE_LOCATOR_ALL', NamespacedSimplePie::LOCATOR_ALL);

/**
 * No known feed type
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_NONE instead.
 */
define('SIMPLEPIE_TYPE_NONE', NamespacedSimplePie::TYPE_NONE);

/**
 * RSS 0.90
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_090 instead.
 */
define('SIMPLEPIE_TYPE_RSS_090', NamespacedSimplePie::TYPE_RSS_090);

/**
 * RSS 0.91 (Netscape)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_091_NETSCAPE instead.
 */
define('SIMPLEPIE_TYPE_RSS_091_NETSCAPE', NamespacedSimplePie::TYPE_RSS_091_NETSCAPE);

/**
 * RSS 0.91 (Userland)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_091_USERLAND instead.
 */
define('SIMPLEPIE_TYPE_RSS_091_USERLAND', NamespacedSimplePie::TYPE_RSS_091_USERLAND);

/**
 * RSS 0.91 (both Netscape and Userland)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_091 instead.
 */
define('SIMPLEPIE_TYPE_RSS_091', NamespacedSimplePie::TYPE_RSS_091);

/**
 * RSS 0.92
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_092 instead.
 */
define('SIMPLEPIE_TYPE_RSS_092', NamespacedSimplePie::TYPE_RSS_092);

/**
 * RSS 0.93
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_093 instead.
 */
define('SIMPLEPIE_TYPE_RSS_093', NamespacedSimplePie::TYPE_RSS_093);

/**
 * RSS 0.94
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_094 instead.
 */
define('SIMPLEPIE_TYPE_RSS_094', NamespacedSimplePie::TYPE_RSS_094);

/**
 * RSS 1.0
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_10 instead.
 */
define('SIMPLEPIE_TYPE_RSS_10', NamespacedSimplePie::TYPE_RSS_10);

/**
 * RSS 2.0
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_20 instead.
 */
define('SIMPLEPIE_TYPE_RSS_20', NamespacedSimplePie::TYPE_RSS_20);

/**
 * RDF-based RSS
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_RDF instead.
 */
define('SIMPLEPIE_TYPE_RSS_RDF', NamespacedSimplePie::TYPE_RSS_RDF);

/**
 * Non-RDF-based RSS (truly intended as syndication format)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_SYNDICATION instead.
 */
define('SIMPLEPIE_TYPE_RSS_SYNDICATION', NamespacedSimplePie::TYPE_RSS_SYNDICATION);

/**
 * All RSS
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_RSS_ALL instead.
 */
define('SIMPLEPIE_TYPE_RSS_ALL', NamespacedSimplePie::TYPE_RSS_ALL);

/**
 * Atom 0.3
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_ATOM_03 instead.
 */
define('SIMPLEPIE_TYPE_ATOM_03', NamespacedSimplePie::TYPE_ATOM_03);

/**
 * Atom 1.0
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_ATOM_10 instead.
 */
define('SIMPLEPIE_TYPE_ATOM_10', NamespacedSimplePie::TYPE_ATOM_10);

/**
 * All Atom
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_ATOM_ALL instead.
 */
define('SIMPLEPIE_TYPE_ATOM_ALL', NamespacedSimplePie::TYPE_ATOM_ALL);

/**
 * All feed types
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::TYPE_ALL instead.
 */
define('SIMPLEPIE_TYPE_ALL', NamespacedSimplePie::TYPE_ALL);

/**
 * No construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_NONE instead.
 */
define('SIMPLEPIE_CONSTRUCT_NONE', NamespacedSimplePie::CONSTRUCT_NONE);

/**
 * Text construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_TEXT instead.
 */
define('SIMPLEPIE_CONSTRUCT_TEXT', NamespacedSimplePie::CONSTRUCT_TEXT);

/**
 * HTML construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_HTML instead.
 */
define('SIMPLEPIE_CONSTRUCT_HTML', NamespacedSimplePie::CONSTRUCT_HTML);

/**
 * XHTML construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_XHTML instead.
 */
define('SIMPLEPIE_CONSTRUCT_XHTML', NamespacedSimplePie::CONSTRUCT_XHTML);

/**
 * base64-encoded construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_BASE64 instead.
 */
define('SIMPLEPIE_CONSTRUCT_BASE64', NamespacedSimplePie::CONSTRUCT_BASE64);

/**
 * IRI construct
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_IRI instead.
 */
define('SIMPLEPIE_CONSTRUCT_IRI', NamespacedSimplePie::CONSTRUCT_IRI);

/**
 * A construct that might be HTML
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_MAYBE_HTML instead.
 */
define('SIMPLEPIE_CONSTRUCT_MAYBE_HTML', NamespacedSimplePie::CONSTRUCT_MAYBE_HTML);

/**
 * All constructs
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::CONSTRUCT_ALL instead.
 */
define('SIMPLEPIE_CONSTRUCT_ALL', NamespacedSimplePie::CONSTRUCT_ALL);

/**
 * Don't change case
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::SAME_CASE instead.
 */
define('SIMPLEPIE_SAME_CASE', NamespacedSimplePie::SAME_CASE);

/**
 * Change to lowercase
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::LOWERCASE instead.
 */
define('SIMPLEPIE_LOWERCASE', NamespacedSimplePie::LOWERCASE);

/**
 * Change to uppercase
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::UPPERCASE instead.
 */
define('SIMPLEPIE_UPPERCASE', NamespacedSimplePie::UPPERCASE);

/**
 * PCRE for HTML attributes
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::PCRE_HTML_ATTRIBUTE instead.
 */
define('SIMPLEPIE_PCRE_HTML_ATTRIBUTE', NamespacedSimplePie::PCRE_HTML_ATTRIBUTE);

/**
 * PCRE for XML attributes
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::PCRE_XML_ATTRIBUTE instead.
 */
define('SIMPLEPIE_PCRE_XML_ATTRIBUTE', NamespacedSimplePie::PCRE_XML_ATTRIBUTE);

/**
 * XML Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_XML instead.
 */
define('SIMPLEPIE_NAMESPACE_XML', NamespacedSimplePie::NAMESPACE_XML);

/**
 * Atom 1.0 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_ATOM_10 instead.
 */
define('SIMPLEPIE_NAMESPACE_ATOM_10', NamespacedSimplePie::NAMESPACE_ATOM_10);

/**
 * Atom 0.3 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_ATOM_03 instead.
 */
define('SIMPLEPIE_NAMESPACE_ATOM_03', NamespacedSimplePie::NAMESPACE_ATOM_03);

/**
 * RDF Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_RDF instead.
 */
define('SIMPLEPIE_NAMESPACE_RDF', NamespacedSimplePie::NAMESPACE_RDF);

/**
 * RSS 0.90 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_RSS_090 instead.
 */
define('SIMPLEPIE_NAMESPACE_RSS_090', NamespacedSimplePie::NAMESPACE_RSS_090);

/**
 * RSS 1.0 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_RSS_10 instead.
 */
define('SIMPLEPIE_NAMESPACE_RSS_10', NamespacedSimplePie::NAMESPACE_RSS_10);

/**
 * RSS 1.0 Content Module Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_RSS_10_MODULES_CONTENT instead.
 */
define('SIMPLEPIE_NAMESPACE_RSS_10_MODULES_CONTENT', NamespacedSimplePie::NAMESPACE_RSS_10_MODULES_CONTENT);

/**
 * RSS 2.0 Namespace
 * (Stupid, I know, but I'm certain it will confuse people less with support.)
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_RSS_20 instead.
 */
define('SIMPLEPIE_NAMESPACE_RSS_20', NamespacedSimplePie::NAMESPACE_RSS_20);

/**
 * DC 1.0 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_DC_10 instead.
 */
define('SIMPLEPIE_NAMESPACE_DC_10', NamespacedSimplePie::NAMESPACE_DC_10);

/**
 * DC 1.1 Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_DC_11 instead.
 */
define('SIMPLEPIE_NAMESPACE_DC_11', NamespacedSimplePie::NAMESPACE_DC_11);

/**
 * W3C Basic Geo (WGS84 lat/long) Vocabulary Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_W3C_BASIC_GEO instead.
 */
define('SIMPLEPIE_NAMESPACE_W3C_BASIC_GEO', NamespacedSimplePie::NAMESPACE_W3C_BASIC_GEO);

/**
 * GeoRSS Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_GEORSS instead.
 */
define('SIMPLEPIE_NAMESPACE_GEORSS', NamespacedSimplePie::NAMESPACE_GEORSS);

/**
 * Media RSS Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS', NamespacedSimplePie::NAMESPACE_MEDIARSS);

/**
 * Wrong Media RSS Namespace. Caused by a long-standing typo in the spec.
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS_WRONG', NamespacedSimplePie::NAMESPACE_MEDIARSS_WRONG);

/**
 * Wrong Media RSS Namespace #2. New namespace introduced in Media RSS 1.5.
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG2 instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS_WRONG2', NamespacedSimplePie::NAMESPACE_MEDIARSS_WRONG2);

/**
 * Wrong Media RSS Namespace #3. A possible typo of the Media RSS 1.5 namespace.
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG3 instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS_WRONG3', NamespacedSimplePie::NAMESPACE_MEDIARSS_WRONG3);

/**
 * Wrong Media RSS Namespace #4. New spec location after the RSS Advisory Board takes it over, but not a valid namespace.
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG4 instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS_WRONG4', NamespacedSimplePie::NAMESPACE_MEDIARSS_WRONG4);

/**
 * Wrong Media RSS Namespace #5. A possible typo of the RSS Advisory Board URL.
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_MEDIARSS_WRONG5 instead.
 */
define('SIMPLEPIE_NAMESPACE_MEDIARSS_WRONG5', NamespacedSimplePie::NAMESPACE_MEDIARSS_WRONG5);

/**
 * iTunes RSS Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_ITUNES instead.
 */
define('SIMPLEPIE_NAMESPACE_ITUNES', NamespacedSimplePie::NAMESPACE_ITUNES);

/**
 * XHTML Namespace
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::NAMESPACE_XHTML instead.
 */
define('SIMPLEPIE_NAMESPACE_XHTML', NamespacedSimplePie::NAMESPACE_XHTML);

/**
 * IANA Link Relations Registry
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::IANA_LINK_RELATIONS_REGISTRY instead.
 */
define('SIMPLEPIE_IANA_LINK_RELATIONS_REGISTRY', NamespacedSimplePie::IANA_LINK_RELATIONS_REGISTRY);

/**
 * No file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_NONE instead.
 */
define('SIMPLEPIE_FILE_SOURCE_NONE', NamespacedSimplePie::FILE_SOURCE_NONE);

/**
 * Remote file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_REMOTE instead.
 */
define('SIMPLEPIE_FILE_SOURCE_REMOTE', NamespacedSimplePie::FILE_SOURCE_REMOTE);

/**
 * Local file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_LOCAL instead.
 */
define('SIMPLEPIE_FILE_SOURCE_LOCAL', NamespacedSimplePie::FILE_SOURCE_LOCAL);

/**
 * fsockopen() file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_FSOCKOPEN instead.
 */
define('SIMPLEPIE_FILE_SOURCE_FSOCKOPEN', NamespacedSimplePie::FILE_SOURCE_FSOCKOPEN);

/**
 * cURL file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_CURL instead.
 */
define('SIMPLEPIE_FILE_SOURCE_CURL', NamespacedSimplePie::FILE_SOURCE_CURL);

/**
 * file_get_contents() file source
 * @deprecated since SimplePie 1.7.0, use \SimplePie\SimplePie::FILE_SOURCE_FILE_GET_CONTENTS instead.
 */
define('SIMPLEPIE_FILE_SOURCE_FILE_GET_CONTENTS', NamespacedSimplePie::FILE_SOURCE_FILE_GET_CONTENTS);

}
