<?php
/*		Freewheeling Easy Mapping Application
 *		A collection of routines for display of trail maps and amenities
 *		copyright Roy R Weil 2019 - https://royweil.com
 */
require_once "rrwParam.php";
require_once "rrwFormat.php";
require_once "rrw_util_inc.php";
class freewheeling_build_website
{
    private static $finalFileExtension = "html";
    private static $archiveCSVfilePointer = null;
    private static $archiveCSVfilename = "archive.csv";
    private static $locMap = "";
    static public function buildWebsiteFromUpload($attr)
    {
        // call freewheeling_Extract_Data_From_HtmlFile::Extract($attr) to move html file to database.
        // then build the website form the database in the directory $freewheeling_www_book_dire
        global $eol, $errorBeg, $errorEnd;
        global $freewheeling_www_book_dire;
        $msg = "";
        ini_set("display_errors", true);
        try {
            $msg .= freeWheeling_edit_setGlobals::setGlobals("buildWebsiteFromUpload");
            $trailId = freeWheelParam::trail($attr);
            $msg .= "I#2820 Building the Website <strong>$trailId</strong> $eol";
            if (empty($trailId))
                throw new Exception("$errorBeg E#2821 no trailId was entered $errorEnd");
            $web_dire = "$freewheeling_www_book_dire/$trailId";
            if (!is_dir($web_dire))
                mkdir($web_dire);
            $msg .= self::uploadWebsite($attr);
            $msg .= self::get_locMap_name($trailId);       // Set self::$locMap, if loc map is found.
            $msg .= "the loc map is at " . self::$locMap . $eol;
            $msg .= freewheeling_Extract_Data_From_HtmlFile::Extract($trailId);
            $msg .= "-------------------------------------------------  I#2822 extracted the web site $trailId, now build the new site $eol";
            $msg .= self::buildWebsiteFromDatabase($trailId);
        } catch (Exception $e) {
            $msg .= $errorBeg . $e->getMessage() . " E#2823 Exception at bottom of buildWebsiteFromUpload: look elsewhere for error $errorEnd";
        }
        return $msg;
    } // end function buildWebsiteFromUpload
    /**
     * Build a static website for a given trail by reading content from the database
     * and writing a set of HTML files into a target directory.
     *
     * This method orchestrates retrieval of trail-specific HTML content from the
     * database, composes multiple pages (index, local history, development,
     * access points, extensions, amenities, resources, store, about), writes them
     * to disk, generates an XML manifest, and saves a build report. It aggregates
     * diagnostic and error messages into a single return string.
     *
     * Behavior summary:
     *  - Opens an archive CSV pointer for the trail (self::$archiveCSVfilePointer).
     *  - Validates that exactly one HTML/trail record exists for $trailId.
     *  - Composes page content using helper methods:
     *      - writeTheHtmlFile()         : preliminary check / template write
     *      - outputParagraph()         : render paragraph fields
     *      - DataBlock()               : render data blocks
     *      - outputFile()              : write individual HTML files and log result
     *      - AmenityDetailPage()       : render amenities detail page
     *      - sellBooks()               : produce store content
     *      - createAboutPage()         : produce about page content
     *      - makeXML()                 : generate XML for the trail
     *  - Saves a build report via freewheeling_writeup::saveReport().
     *  - Returns an aggregated message log including success, informational and
     *    error messages. A display link to the local website is appended to the log.
     *
     * Global dependencies:
     *  - $eol               : end-of-line string used in log messages
     *  - $errorBeg, $errorEnd : wrappers for formatting error messages
     *  - $wpdbExtra         : database abstraction used to run SQL and fetch rows
     *  - $freewheeling_www_book_dire : base directory path where trail site is written
     *
     * Files and outputs:
     *  - Writes site files under "$freewheeling_www_book_dire/$trailId", including:
     *      index, localhistory, development, accesspoints, extension, amenities,
     *      resources, store, about (file names derived from the page key).
     *  - Opens/writes self::$archiveCSVfilename in the trail directory.
     *  - Calls makeXML($trailId) which generates additional XML output.
     *  - Persists a build report (HTML) via freewheeling_writeup::saveReport().
     *
     * Database usage:
     *  - Executes a SQL select that joins $wpdbExtra->html and $wpdbExtra->trails:
     *      select * from {html},{trails} where trId = '$trailId' and htmTrailId = '$trailId'
     *  - Expects exactly one matching record. If the number of rows != 1, the method
     *    appends an error message to the log and returns early.
     *
     * Error handling:
     *  - SQL mismatch and other known errors are recorded in the returned $msg.
     *  - The method contains a try/catch that captures Exceptions and appends
     *    their messages to the returned log; exceptions are not re-thrown.
     *  - Helper methods (outputFile, writeTheHtmlFile, etc.) may also append
     *    diagnostic or error text to the returned message.
     *
     * Parameters:
     *  @param string|int $trailId Identifier for the trail/website to build.
     *
     * Return:
     *  @return string Aggregated log and diagnostic message describing actions,
     *                 successes and errors encountered during the build.
     *
     * Notes & Caveats:
     *  - This method relies on a number of instance/static helpers and globals;
     *    unit testing or reuse should provide stubs/mocks for those dependencies.
     *  - The method writes directly to the filesystem and therefore requires the
     *    target directory to be writable by the PHP process.
     *  - The SQL uses interpolated variables; ensure $trailId is validated/sanitized
     *    at call site to avoid injection risks if $wpdbExtra does not handle it.
     *  - The returned message may contain HTML fragments and should be presented
     *    in a safe context (escaped where appropriate) if rendered in a browser.
     *
     * @see writeTheHtmlFile()
     * @see outputParagraph()
     * @see DataBlock()
     * @see outputFile()
     * @see AmenityDetailPage()
     * @see sellBooks()
     * @see createAboutPage()
     * @see makeXML()
     */
    static public function buildWebsiteFromDatabase($trailId)
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        global $freewheeling_www_book_dire;
        $msg = "";
        try {
            $direName = "$freewheeling_www_book_dire/$trailId";
            self::$archiveCSVfilePointer = fopen($direName . self::$archiveCSVfilename, "w");
            $msg .= str_repeat("=", 40) . " I#2824 build The Website($trailId) start " . str_repeat("=", 40) . $eol;
            $msg .= self::writeTheHtmlFile($trailId, "I%1238 html file not yet built", "index"); // check for writable file
            $sql = "select * from $wpdbExtra->html,$wpdbExtra->trails where trId = '$trailId' and htmTrailId = '$trailId' ";
            $htmlDataResult = $wpdbExtra->get_resultsA($sql);
            if (1 != $wpdbExtra->num_rows) {
                $msg .= "$errorBeg E#2825 found " . $wpdbExtra->num_rows .  " records for $trailId $eol $sql $errorEnd";
                return $msg;
            }
            $html = $htmlDataResult[0];
            $trailName = $html["trname"];
            $indexContent = "";  // the html to be written to the file
            $indexContent .= "<h1>$trailName</h1>";
            $indexContent .= self::outputParagraph($html, "htmFirstSection");
            $indexContent .= self::DataBlock($html);
            $indexContent .= self::outputParagraph($html, "htmDescriptionAfterDataBlock");
            $indexContent .= "</body></html>";
            $msg .= self::outputFile($html, $indexContent, "index"); // write the file
            $content = "<h1>$trailName - Local History</h1>";
            $content .= self::outputParagraph($html, "htmLocalHistory");
            $msg .= self::outputFile($html, $content, "localhistory"); // write the file
            $content = "<h1>$trailName - Development Plans</h1>";
            $content .= self::outputParagraph($html, "htmDevelopment");
            $msg .= self::outputFile($html, $content, "development"); // write the file
            $content = "<h1>$trailName - Access Points</h1>";
            $content .= self::outputParagraph($html, "htmAccessPoints");
            $content = str_replace(
                "plus code",
                "<a href='https:://freewheelingeasy.com/prebuilt/icon_google/pluscode.png'
                        alt=' plus code ' target='pluscode' </a>",
                $content
            );
            $msg .= self::outputFile($html, $content, "accesspoints"); // write the file
            $content = "<h1>$trailName - Extensions of the Ride</h1>";
            $content .= self::outputParagraph($html, "htmExtension");
            $msg .= self::outputFile($html, $content, "extension"); // write the file
            $content = "<h1>$trailName - Amenities Overview</h1>";
            $htmlSummary = $html["AmenitiesCollection"];
            $content .= self::AmenityDetailPage($trailId, $htmlSummary);
            $msg .= self::outputFile($html, $content, "amenities"); // write the file
            $content = "<h1>$trailName - Resources</h1>";
            $content .= self::outputParagraph($html, "htmResources");
            $content .= self::outputParagraph($html, "htmOrganizationAddress");
            $msg .= self::outputFile($html, $content, "resources"); // write the file
            $content = self::sellBooks();
            $msg .= self::outputFile($html, $content, "store"); // write the file
            $content = self::createAboutPage($html);
            $msg .= self::outputFile($html, $content, "about"); // write the file
            $msg .= self::makeXML($trailId);
        } catch (Exception $e) {
            $msg .= "$errorBeg E#2826 Exception at bottom of buildWebsiteFromDatabase: " . $e->getMessage() . $errorEnd;
        }
        $msg .= str_repeat("=", 40) . "I#2827 buildWebsiteFromDatabase($trailId) end " . str_repeat("=", 40) . $eol;
        // now display the home page of the website
        $websiteContentsDisplay = "$eol" . str_repeat("-", 100) . " display Website
                        <a href='http://127.0.0.1/w/$trailId' target='local-website'> http://127.0.0.1/w/$trailId </a> ";
        //$websiteContentsDisplay .= file_get_contents("https://$trailId.shaw-weil.com/");
        //$websiteContentsDisplay  .= $eol . str_repeat("&gt;", 100) . " building Website end $eol";
        $workFile = "build_web_" . date("Y-m-d") . ".html";
        $msg .= freewheeling_writeup::saveReport("buildWeb-$trailId-", $msg, "html");
        return $msg;
    } // end function buildWebsiteFromDatabase
    static private function AmenityDetailPage($trailId, $htmlSummary)
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        $debugAmenityDetailPage = rrwParam::Boolean("debugAmenityDetailPage");
        $content = "";  // the html to be written to the file
        $sqlAmenityList = "select htmAmenitiesList from $wpdbExtra->html where htmTrailId = '$trailId' ";
        $amenityList = $wpdbExtra->get_var("$sqlAmenityList");
        $content .= "$eol $eol $amenityList";
        if (strcmp($content, "<p") == 0) {
            $content .= substr($content, 0, -2); // remove <p from the end of the paragraph
        }
        if ($debugAmenityDetailPage) print self::displayContent($content, "Amenity Detail Page after the load of htmAmenitiesList ");
        $sqlIconAmenity = "select * from $wpdbExtra->icons where trailId = '$trailId' and onMap like '%amen%'
                            order by milepost, iconName";
        if ($debugAmenityDetailPage) print "sqlIconAmenity $sqlIconAmenity $eol";
        $icons = $wpdbExtra->get_resultsA($sqlIconAmenity);
        if (empty($htmlSummary) || "None" == $htmlSummary)
            $htmlSummary = "None on Trail";
        $content .= "<p style='text-size:larger; text-weight:bold;' >Summary</span> $htmlSummary</p> $eol";
        $content .= "<p > Amenities are organized by major trailheads.
        These trailheads are listed in order from the start of the trail to the end of the trail.
        In the trailhead listing the amenities are grouped around the nearest access place, and listed in the order of
        those closest to the trailhead to those farthest from the trailhead.
        </p> $eol";
        if ($debugAmenityDetailPage) print self::displayContent($content, "Amenity Detail Page after adding comment ");
        $content .= "<table maxwidth='400px'>" .
            rrwFormat::HeaderRow("fROM Mile", "AROUND Amenity", "TO MILE", " &nbsp; ");
        //   $content .= "<table> <tr><th> &nbsp;</th> <th width='50px' >from mile</th><th> &nbsp;</th><th width='200px'> around amenity </th><th> &nbsp;</th><th width='50px >to mile</th><th>&nbsp; </th> </tr>' \n";
        foreach ($icons as $icon) {
            $iconName = $icon["iconName"];
            $iconId = $icon["iconId"];
            $milepost = $icon["milepost"];
            $milePostPrefix = $icon["milePostPrefix"];
            $mileDisplayStart = $icon["mileDisplayStart"];
            $mileDisplayEnd = $icon["mileDisplayEnd"];
            $milepostDisplay = rrwFormat::milePostText($milepost, $milePostPrefix);
            $amenityLink = "<a class='external' href='https://freewheelingeasy.com/freewheelingeasy-amenity/?amenity=$iconName-$iconId' >$iconName - $milepostDisplay </a>";
            $content .= rrwFormat::CellRowColor($mileDisplayStart, $amenityLink, $mileDisplayEnd, " &nbsp; ");
        } // end foreach
        $content .= "</table>" . $eol;
        $content .= "<p > The following table amenities in oder of their access point along the trail.
        if there are multiple amenities of the same type at a given access point there are listed in closet to farthest from the access point.";
        $fullNameFull = freewheeling_WriteUp::fullNameFromShortName("amenities", "milepost-$trailId.html");
        print "amenity detail page $fullNameFull $eol";
        if (file_exists($fullNameFull)) {
            $content .= freewheeling_WriteUp::getFileContents("amenities", "milepost-$trailId.html") . $eol;
            print "amenity detail page $fullNameFull exists $eol";
        } else {
            print "amenity detail page $fullNameFull does not exist $eol";
        }
        if ($debugAmenityDetailPage) print "$eol $eol";
        return $content;
    } // end function AmenityDetailPage
    private static  function displayContent($content, $message)
    {
        global $eol;
        $msg = "";
        $msg .= "$eol start " . str_repeat("V", 120) . " start - $message $eol ";
        $msg .= htmlspecialchars($content);
        $msg .= "$eol end " . str_repeat("^", 120) . " end - $message $eol";
        return $msg;
    }
    /**
     * Generates the HTML header section for the website, including meta tags, stylesheets,
     * Google Tag Manager script, and the site masthead with logo, title, and a random trail image.
     *
     * @param array $html An associative array containing:
     *   - 'trname' (string): The name of the trail.
     *   - 'trid' (string): The trail ID, used for URLs and image lookups.
     *   - 'googleTagManager' (string): The Google Tag Manager script or code to be included in the header.
     * @param bool $homePage Indicates if the current page is the home page (adds a canonical link if true).
     * @return string The complete HTML header section as a string.
     *
     */
    static private function header($html, bool $homePage)
    {
        global $eol, $errorBeg, $errorEnd;
        static $firstTime = true;
        $trailName = $html["trname"];
        $trailId = $html["trid"];
        $googleTagManager = $html["googleTagManager"];
        if (empty($googleTagManager)) {
            if ($firstTime) {
                print "$errorBeg E#2828 no google Tag Manager content for $trailId see
                <a href='https://marketingplatform.google.com/about/analytics/' target='analytics'>
                add website to search, get google script tag </a> $eol
                &nbsp; then edit-html entry <a href='https://edit.shaw-weil.com/edit-html/?trailid=$trailId' target='edit-html' >
                      to update the google tag </a>$eol
                &nbsp; then reload the 127.0.0.1/w/website$errorEnd";
            }
        }
        if ($firstTime) {
            $hostname = "$trailId.freewheelingeasy.com";
            $DNSRecords = dns_get_record($hostname, DNS_CNAME);
            $buffer =  rrwUtil::fetchURLcontents($hostname, "", 120);
            if (strpos($buffer, "This site can’t be reached")) {
                print "$errorBeg E#2832 DNS CNAME record not found for $hostname
                    &nbsp; please add a CNAME record in <a href='https://www.pillowandpedal.org:2083/cpsess4601280003/frontend/jupiter/zone_editor/index.html#/list' target='edit'> the 'zone editor'</a> that points $hostname, 'rweilwww.github.io' to freewheelingeasy.com
                    &nbsp; then allow up to 48 hours for DNS propagation $errorEnd " . rrwUtil::print_r($DNSRecords, true, "DNS CNAME Records");
                print "$eol   $eol";
            } elseif (strpos($buffer, "Your connection is not private")) {
                print "$errorBeg E#2830 SSL Certificate not found for $hostname
                    &nbsp; then allow up to 48 hours for DNS propagation $errorEnd " . rrwUtil::print_r($DNSRecords, true, "DNS CNAME Records");
                print "$eol   $eol";
            } else {
                $firstTime = false;
            }
        }
        if ($homePage) {
            $canonicalLink = "<link rel='canonical' href='https://$trailId.freewheelingeasy.com/' />";
        } else {
            $canonicalLink = "";
        }
        $header = "<!DOCTYPE HTML>
<html>
    <head>
        <meta http-equiv=\"Cache-Control\" content=\"no-cache, no-store, must-revalidate\" />
        <meta http-equiv=\"Pragma\" content=\"no-cache\" />
        <meta http-equiv=\"Expires\" content=\"0\" />
        $googleTagManager
        <link rel='stylesheet' id='twentythirteen-css' href='https://freewheelingeasy.com/wp-content/themes/twentythirteen/style.css' media='all' />
        <link rel='stylesheet' id='roysheader-css' href='https://freewheelingeasy.com/wp-content/themes/roys-header/style.css' media='all' />
        <link rel='stylesheet' id='words-css' href='https://freewheelingeasy.com/prebuilt/trails-html/_css/trails_html.css' media='all' />
        $canonicalLink
        <script src='https://pictures.shaw-weil.com/randomTrailPicture.js'></script>
        <!-- themes style section based on url and customizations  - normal -->
        <style>
            .menucolor {
                color: #ffffff;
                background-color: #424ed6;
                background: #424ed6;
                min-height: 26px;
            }
            .menuitem {
                color: #ffffff;
                background-color: #424ed6;
                min-height: 26px;
            }
            .nav-menu a {
                color: #ffffff!important;
                background-color: #424ed6;
                min-height: 26px;
            }
            .nav-menu .current_page_item > a, .nav-menu .current_page_ancestor > a, .nav-menu .current-menu-item > a, .nav-menu .current-menu-ancestor > a {
                color: #424ed6!important;
                background-color: #ffffff;
            }
            .site-footer a {
                color: #ffffff!important;
            }
            .site-title {
                font-size: 36px;
            }
            .locMap {
                width:auto ;
                height:155px;
            }
            #trailer a:visited {
                color: #ffffff!important;
            }
        </style>
        <title>$trailName Trail</title>
    </head>
    <body>
        <a class=\"screen-reader-text skip-link\" href=\"#content\" title=\"Skip to content\"> Skip to content </a>
        <!--  ==================================================================================================== header -->
        <header id='masthead' style='text-align:left;'>
            <a class='screen-reader-text skip-link' href='#content' aria-label=' Skip to content' title='Skip to content'> </a>
            <div id='rrw_header_menu_block'>
                <table id='rrw_header_mastheadPhotos' style='min-height: 30px; border: 2px; ' role='presentation'>
                    <tr>
                        <td><a class='home-link site-description' href='https://freewheelingeasy.com'
                                title='FreeWheeling Easy in Western Pennsylvania' rel='home'>
                ";
        if (empty(self::$locMap))
            $header .= "
                        <img src='https://freewheelingeasy.com/wp-content/uploads/2018/10/ed4ne-cover25.jpg' alt='FreeWheeling Easy in Western Pennsylvania logo '
                        class='locMap' > </a>";
        else
            $header .= "<img src='" . self::$locMap . "' alt='Map showing the trail location within the surrounding region providing context for the trail environment' class='locMap' > </a>";
        $header .= "
                </td>
                <td style='text-align:center; border:thin;'>
                   <a href='https://$trailId.freewheelingeasy.com' title='$trailName' rel='home' target='remote-website'>
                    <h1 class='site-title'>$trailName</h1>
                    <h2 class='site-description'> &nbsp; </h2>
                    </a>
                </td>
                <td class='site-description'  >
                    <div id='randomTrailImageGoesHereDiv' style='margin-top:5px' >one moment while we fetch a trail picture
                        <script>
                            randomPicFunction('randomTrailImageGoesHereDiv');
                        </script>
                    </div>
                </td>
            </tr>
        </table>
        </div> <!-- end /div id= rrw_header_menu_block -->
        </header> <!-- end /header id=masthead -->
        <a id=\"content\" name=\"content\" > </a>
            ";
        return $header;
    } // end function header
    static private function outputFile($html, $content, $filename)
    {
        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        $trailId = $html["trid"];
        if ($filename == "index")
            $homepage = true;
        else
            $homepage = false;
        $outHtml = self::header($html, $homepage);
        $outHtml .= self::navMenu($html);
        $outHtml .= "$content $eol ";
        $outHtml .= self::Trailer($html);
        $outHtml .= "</body></html>";
        $msg .= self::writeTheHtmlFile($trailId, $outHtml, $filename); // write the file
        return $msg;
    }
    static private function navMenu($html)
    {
        $trailId = $html["trid"];
        $localhistory = $html["htmLocalHistory"];
        $navMenu = '
        <!--  =============================================================================== nav bar -->
        <div id="navbar" class="menucolor" style="z-level:1;">
        <nav id="site-navigation" class="navigation main-navigation menucolor">
            <table role="presentation">
                <tr class="nav-menu menucolor">
                    <td>
                  <div class="menu-menu-1-container">
                <ul id="menu-menu-1" class="nav-menu menucolor">
                    <li id="menu-item-home" class="menu-item">
                        <a href="index.html">Home</a></li>';
        $navMenu .= '
                    <li id="menu-item-development" class="menu-item">
                        <a href="https://freewheelingeasy.com/google-map/?trailId='
            . $trailId . '&nohead=please" target="map" >map</a></li>
                    ';
        $navMenu .= self::menuItem($html, "htmLocalHistory", "Local History");
        $navMenu .= self::menuItem($html, "htmDevelopment", "Development Plans");
        $navMenu .= self::menuItem($html, "htmAccessPoints", "Access Points");
        $navMenu .= self::menuItem($html, "htmExtension", "Extensions of the ride");
        $navMenu .= self::menuItem($html, "", "Amenities");
        /* $navMenu .= '
                        <ul class="sub-menu">
                            <li id="menu-lodging" class="menu-item"><a href="https://freewheelingeasy.com/notyet.html/">Lodging (0)</a></li>
                            <li id="menu-food" class="menu-item"><a href="https://freewheelingeasy.com/notyet.html/">Food (0)</a></li>
                            <li id="menu-bike" class="menu-item"><a href="https://freewheelingeasy.com/notyet.html/">Bike Shop (0)</a></li>
                        </ul>
                        ';
                        */
        $navMenu .= self::menuItem($html, "htmResources", "Resources");
        $navMenu .= self::menuItem($html, "", "Store");
        $navMenu .= self::menuItem($html, "", "About");
        $navMenu .= '
                     </ul>
</div>  <!-- menu-menu-1-container -->
                </td>
        </tr>
    </table>
</nav>
</div>      <!-- id=navbar  -->
        ';
        return $navMenu;
    } // end function navMenu
    private static function menuItem($html, $databaseItem, $menuDisplay)
    {
        // checks the database to if we have content for this page
        $menuItem = "";
        if (!empty($databaseItem)) {         // $content is not in the database
            $contents = $html[$databaseItem];
            if (empty($contents))       // nothing there - no menu link
                return "";
        } else
            $databaseItem = $menuDisplay;
        $menuLink = str_replace("htm", "", $databaseItem);
        $menuLink = strtolower($menuLink) . "." . self::$finalFileExtension;
        $menuItem = "
        <li id=\"menu-item-$menuLink\" class=\"menu-item\">
            <a href='$menuLink' >$menuDisplay</a></li>
        ";
        return $menuItem;
    }
    private static function createAboutPage($html)
    {
        $versionDate = $html['VersionText'];
        $oldest = $html["oldestSegment"];
        if (empty($oldest))
            $oldest = "unknown";
        if (empty($versionDate))
            $versionDate = "unknown";
        $sellBook = rrwFormat::sellTheBook();   // one Line to sell the book
        $content = "
        <h1>About this website</h1>
        <h2>Recency of the Information on This Website</h2>
        <p>Write-up last update on  $versionDate. Oldest segment check $oldest .</p>
        <p> The <strong>last update on date</strong> is the last time that we made a change to the writeup.</p>
<p>The <strong>Oldest Segment check date</strong>&quot;  is the date for which we have reliable information on that segment.
For shorter trails this probably means the entire trail.
For longer trails, we break the trail into segments and ride the trail over several days, which may be some time apart.
We also talk to the trail managers, when we know there is development activity in progress.</p>
<h2>How this Website is Created.</h2>
<p>This website was created via an automated process from the current daft of the book
<i>Freewheelingeasy in Western Pennsylvania</i>.
As we ride the trails, we are continuously updating the guide book.
The amenity information is updated based on data from the Google Map webpages.
$sellBook </p>
<p> &nbsp; </p>
<p>If you see any inaccuracies, problems,or have suggestions for additional content
please use the comment form on the Freewheeling website
<a href='https://freewheelingeasy.com/webmaster-feedback/' target='feedback'>https://freewheelingeasy.com/webmaster-feedback/.</a></p>
<p>Roy and Mary</p>
";
        return $content;
    }
    private static function get_locMap_name($trailId)
    {
        global $eol, $errorBeg, $errorEnd;
        global $freewheeling_www_book_dire;
        $debugVersion = true;
        $msg = "";
        try {
            if ($debugVersion) print "look for locmap in $freewheeling_www_book_dire/$trailId $eol";
            $direname = "$freewheeling_www_book_dire/$trailId";
            $locMapNotFound = true;
            foreach (new DirectoryIterator($direname) as $fileInfo) {
                if ($fileInfo->isDot()) continue;
                $filename = $fileInfo->getFilename();
                if ($debugVersion) print "try file $filename $eol";
                if (strpos($filename, "locmap") !== false) {
                    self::$locMap = $filename;
                    $locMapNotFound = false;
                    break;
                }
            } // end foreach
            if ($locMapNotFound) {
                $msg .=  "$errorBeg E#2836 did not found a locMap in $direname $errorEnd";
            }
        } catch (Exception $e) {
            $msg .= "$errorBeg E#2833 bottom of get_locMap_name: looking for locMap file in $direname $errorEnd";
        }
        return $msg;
    }
    static private function outputParagraph($html, $item)
    {
        $content = "
        $html[$item]
        ";
        return $content;
    }
    static private function     DataBlock($html)
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        $sql = "select AVG(minutesFromPgh) from $wpdbExtra->icons where trailId =  '$html[trid]' ";
        $minutes = $wpdbExtra->get_var($sql);
        $driveTime = self::convertMinutesToHoursMinutes($minutes);
        $DataBlock = "
        <div class=\"datablock\">
        <table style=\"border:2px solid black;\">";
        $DataBlock .= self::dataBlockRow("Location", $html["Location"]);
        $DataBlock .= self::dataBlockRow("Trailheads", $html["TrailheadCollection"]);
        $DataBlock .= self::dataBlockRow("Length, Surface", $html["surfacedescription"]);
        $DataBlock .= self::dataBlockRow("Character", $html["characterdescription"]);
        $DataBlock .= self::dataBlockRow("Usage restrictions", $html["Usage"]);
        $DataBlock .= self::dataBlockRow("Amenities", $html["AmenitiesCollection"]);
        $DataBlock .= self::dataBlockRow("Driving time from Pittsburgh", $driveTime);
        //  $DataBlock .= "<tr><td colspan=2 style=\"text-size:5px\"> &nbsp; </td></tr>\n";
        $DataBlock .= "</table>
        </div> <!-- class=datablock -->";
        return $DataBlock;
    }
    static private function convertMinutesToHoursMinutes($minutes)
    {
        if (empty($minutes))
            return ("");
        $hours = floor($minutes / 60);
        $minutes = $minutes - ($hours * 60);
        $minutes = round($minutes, 0);
        $driveTime = "$hours hours $minutes minutes";
        return $driveTime;
    } // end function convertMinutesToHoursMinutes
    static private function dataBlockRow($heading, $value)
    {
        if (empty($value))
            return ("");
        $tablerow = "
        <tr>
        <td style=\"width:5px\" > &nbsp;</td>
        <td ><span class=\"DB---Bold\" >$heading</span></td>
            <td >$value</td>
            <td > &nbsp;</td>
            </tr>
        ";
        return $tablerow;
    }
    static private function Trailer($html)
    {
        global $eol, $errorBeg, $errorEnd;
        $trailName = $html["trname"];
        $oldest = $html["oldestSegment"];
        $oldest = strtotime($oldest);
        $oldest = date("Y-M", $oldest);
        $trailer = $eol . ' <div id="trailer" class="menucolor" style="z-level:1;">';
        $trailer .= "$trailName - Oldest segment segment check $oldest  &nbsp; &nbsp; &nbsp; &nbsp; This page brought to you by the book
        <button onclick=\"window.location.href='https://freewheelingeasy.com/';\" class=\"menucolor\"  >FreeWheeling Easy in Western Pennsylvania</button>
                     ";
        $trailer .= '  </div> <!-- id=trailer  --> ';
        return $trailer;
    }
    static private function writeTheHtmlFile($trailId, $htmlData, $filenameShort)
    {
        global $eol, $errorBeg, $errorEnd;
        global $freewheeling_www_book_dire;
        $msg = "";
        $debugWriteTheHtmlFile = false;
        $dirname = "$freewheeling_www_book_dire/$trailId";      // also used in uploadWebsite
        if ($debugWriteTheHtmlFile) $msg .= "check for the website directory $dirname" . $eol;
        if (!is_dir($dirname)) {
            $msg .= "$errorBeg E#2838 directory does not exist $dirname $errorEnd";
            $msg .= self::mkDireList();
            return $msg;
        }   // dire exists
        if ("sitemap" == $filenameShort)
            $ext = "xml";
        else
            $ext = self::$finalFileExtension;
        $filenameFull = $dirname . "/$filenameShort.$ext";
        if ($debugWriteTheHtmlFile) $msg .= "open for writing the file $filenameFull " . $eol;
        $fpFinal = fopen($filenameFull, "w");
        if (!is_resource($fpFinal)) {
            $msg .= "$errorBeg E#2839 Cannot open $filenameFull for writing $errorEnd";
            return $msg;
        } // file exists
        if (false === fwrite($fpFinal, $htmlData)) {
            $msg .= "$errorBeg E#2840 Cannot write the accumulated html to $filenameFull $errorEnd";
            return $msg;
        }
        fclose($fpFinal);
        $htmlRef = "https://$trailId.freewheelingeasy.com/$filenameShort.$ext";
        if (false === fwrite(self::$archiveCSVfilePointer, "$htmlRef \n"))
            $msg .= "$errorBeg E#2841 Cannot write the $htmlRef  $errorEnd";
        $size = strlen($htmlData);
        $msg .= "wrote $filenameFull with $size characters $eol";
        return $msg;
    } //
    static private function mkDireList()
    {
        // create the directory structure for the trail websites
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra, $rrw_trails;
        $msg = "";
        $msg .= "<ol>
        <li>in windows</li>
        <li>copy the cd and mkdir commands to the clipboard</li>
        <li>open a command window</li>
        <li>paste the commands into the command window</li>
        <li>execute the commands</li>
        <li>kill the window</li>
        <li>open filezilla </li>
        <li>navicate to the /www-book directory </li>
        <li>select all the window dires, or only one if that one is missing </li>
        <li>copy to a2hosting machine.</li>
        </ol>$eol $eol $eol";
        $msg .= "cd /d \"D:\a2-pillowan\www-book\"" . $eol;
        $sql = "select trid from $rrw_trails";
        $trails = $wpdbExtra->get_resultsA($sql);
        foreach ($trails as $trail) {
            $trailId = $trail["trid"];
            $msg .= "mkdir $trailId" . $eol;
        } // end foreach
        $msg .= "$eol $eol";
        return $msg;
    } // end function mkDireList
    private static function sellBooks()
    {
        $msg = "";
        $site = site_url("");
        $site = str_replace(".com", "", $site);
        $site = str_replace(".org", "", $site);
        foreach (array(".", "/", ":") as $iiLookFor) {
            $ii = strrpos($site, $iiLookFor);
            if (false !== $ii)
                $site = substr($site, $ii + 1);
        }
        $msg .=
            "<!--===================================================================== " .
            "start code for FeewheelingeasySell -->";
        $msg .= self::DisplayStyle();
        $msg .= "Selected price includes sales tax, shipping, and handling.";
        //		$msg .= '<figure class="wp-block-table"><table><tbody><tr>';
        $msg .= '<ul class="FeewheelingeasySell" >';
        //  ----------------------------------- first book
        $options = array(
            "1 book",
            24.00,
            "2 books",
            45.00,
            "3 books why not by five",
            66.00,
            "5 books",
            74.00,
        );
        $msg .= self::oneItem_td(
            "https://freewheelingeasy.com/_images/cover-fwe 126x141.gif",
            //            "https://freewheelingeasy.com/_images/cover ed4ne 225x141.jpg",
            "Freewheelingeasy book cover",
            126,
            141,
            "FWE NE - $site",
            "FreeWheeling Easy<br />North East Edition ",
            $options
        );
        //  ----------------------------------- next book link up
        $options = array(
            "1 book",
            6.00,
            "2 books",
            10.00,
            "3 books",
            13.00,
            "5 books",
            18.00,
        );
        $msg .= self::oneItem_td(
            "https://freewheelingeasy.com/_images/cover-linkup 99x141.jpg",
            "Linking Up book cover",
            99,
            141,
            "link up - $site",
            "Linking Up<br />Planning DC to Pittsburgh ",
            $options
        );
        //  ----------------------------------- next book Companion
        $options = array(
            "1 book",
            44.00,
            "2 books",
            80.00,
            "3 books  why not by five",
            110.00,
            "5 books",
            150.00,
        );
        $msg .= self::oneItem_td(
            "https://freewheelingeasy.com/_images/cover-companion 87x141.jpg",
            "The Great Allegheny Companion book cover",
            87,
            141,
            "conpanion - $site",
            "The Great<br />Allegheny Companion ",
            $options
        );
        //  ----------------------------------- next book all three
        $options = array(
            "3 books",
            54.00,
            "6 books",
            100.00,
        );
        $msg .= self::oneItem_td(
            "https://freewheelingeasy.com/_images/cover-threebook 180x141.jpg",
            "three  Book covers",
            180,
            141,
            "three books - $site",
            "Buy all three books<br /> ",
            $options
        );
        $msg .= "\n</ul>\n";
        $msg .=
            "<!--===================================================================== " .
            "end code for FeewheelingeasySell -->";
        return $msg;
    }
    //  ===============================================================
    //  ===============================================================
    //	<figure class=\"wp-block-image size-full is-resized\">
    //	class=\"wp-image-33\"
    private static function oneItem_td(
        $image,
        $imageAlt,
        $imageWidth,
        $imageHeight,
        $itemId,
        $description,
        $options
    ) {
        $msg = "";
        $itemKey = str_replace("<br/>", "-", $description);
        $msg .= "<li class=\"FeewheelingeasySell\" >
 <img loading=\"lazy\"
src=\"$image\"
alt=\"$imageAlt\"
 height=\"$imageHeight\"/>
 ";
        if (strpos($description, "Buy all") !== false)
            $msg .= "<img src='https://freewheelingeasy.com/_images/white 1x1.jpg'
	 						height='1px' />";
        $msg .= "
  <form action=\"https://www.paypal.com/cgi-bin/webscr\" method=\"post\">
  <!-- Identify your business so that you can collect the payments. -->
  <input type=\"hidden\" name=\"business\" value=\"paypal@freewheelingeasy.com\">
  <!-- Specify a Buy Now button. -->
  <input type=\"hidden\" name=\"cmd\" value=\"_xclick\">
  <!-- Specify details about the item that buyers will purchase. -->
  <input type=\"hidden\" name=\"item_name\" value=\"$itemId\">
  <input type=\"hidden\" name=\"currency_code\" value=\"USD\">
  <!-- Provide a drop-down menu option field with prices. -->
  <input type=\"hidden\" name=\"on1\" value=\"$itemKey\">$description <br />
  <select name=\"os1\">";
        for ($ii = 0; $ii < count($options); $ii = $ii + 2) {
            $item = $options[$ii];
            $cost = $options[$ii + 1];
            $msg .= "<option value=\"$item\">$item - $cost USD</option>\n";
        }
        $msg .= "
  </select> <br />
   <!-- Specify the price that PayPal uses for each option. -->
   <input type=\"hidden\" name=\"option_index\" value=\"1\">
   ";
        $limit = count($options) / 2;
        for ($ii = 0; $ii < $limit; $ii++) {
            $item = $options[$ii * 2];
            $cost = $options[($ii * 2) + 1];
            $msg .= "
  <input type=\"hidden\" name=\"option_select$ii\" value=\"$item\">
  <input type=\"hidden\" name=\"option_amount$ii\" value=\"$cost\"> \n ";
        }
        $msg .= "
  <!-- Display the payment button. -->
  <input type=\"image\" name=\"submit\" border=\"0\"
    src=\"https://www.paypalobjects.com/en_US/i/btn/btn_buynowCC_LG.gif\"
    alt=\"Buy Now\"alt=\"Buy now via PayPal - The safer, easier way to pay online!\" />
  </form>
</li >
";
        return $msg;
    } // end function oneItem_td
    private static function DisplayStyle()
    {
        $msg = "
	<style>
		.FeewheelingeasySell {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(120pt, 1fr));
    grid-template-rows: masonry;
    grid-column-gap: 2px;
    grid-row-gap: 1px;
    padding: 0;
}
.FeewheelingeasySell li {
    margin: 3px;
 //   white-space: nowrap;
    list-style-type: none;
    padding: 0;
    text-overflow: clip;
    background-color: white;
    grid-column-gap: 1px;
}
</style>
";
        return $msg;
    } //
    private static function uploadWebsite($attr)
    {
        global $eol, $errorBeg, $errorEnd;
        global $freewheeling_prebuilt_dire_trails_html;
        global $freewheeling_www_book_dire;
        global $freeWheelingEasy_ipHome;
        $msg = "";
        $debugUpload = false;
        $ip = $_SERVER['REMOTE_ADDR'];
        if ($freeWheelingEasy_ipHome != $ip) {
            if (freeWheeling_edit_setGlobals::notAllowedToEdit("you must be logged in to upload an html file"))
                return "$errorBeg E#2842 you must be logged in to upload an html file? $errorEnd";
        }
        $msg .= freeWheeling_edit_setGlobals::setGlobals("uploadWebsite");
        if ($debugUpload) $msg .= rrwUtil::print_r($_FILES, true, "FILES");
        if (!is_array($_FILES)) { // was file name entered.{
            $msg .= rrwUtil::print_r($_POST, true, "POST");
            $msg .= rrwUtil::print_r($_FILES, true, "FILES");
            throw new Exception("$msg $errorBeg E#2843 no files are available to upload $errorEnd");
        }
        if (!array_key_exists("htmlFileToUpload", $_FILES)) { // was file name entered.{
            $msg .= rrwUtil::print_r($_POST, true, "POST");
            $msg .= rrwUtil::print_r($_FILES, true, "FILES");
            throw new Exception("$msg $errorBeg E#2844 no html file is available to upload $errorEnd");
        }
        $trailId = $_FILES["htmlFileToUpload"]["name"];
        $trailId = str_replace(".html", "", $trailId);
        //==================================================form submitted  upload
        $dirname = "$freewheeling_prebuilt_dire_trails_html";
        $msg .= self::moveOneFile("htmlFileToUpload", $dirname);
        // create the www-book directory
        $dirname = "$freewheeling_www_book_dire/$trailId";
        if (!is_dir($dirname)) {
            $result = mkdir($dirname, 0750, false);
            if ($result === false) {
                $msg .= "$errorBeg E#2845 directory does not exist, and could not be made '$dirname' $errorEnd";
                throw new Exception($msg);
            }
        }   // dire exists
        $locMapNotFound = true;
        //================================================== now upload the loc map and any other image files
        for ($ii = 0; $ii < count($_FILES); $ii++) {
            $postname = "imageUpload$ii";
            if (!array_key_exists($postname, $_FILES))
                continue;
            $fileName = $_FILES[$postname]["name"];
            if (strpos($fileName, "locmap") !== false)
                $locMapNotFound = false;
            $msg .= self::moveOneFile($postname, $dirname);
        } // end for each file
        if ($locMapNotFound)
            $msg .= "$errorBeg I#2846 No location map (locMap)file was uploaded $errorEnd";
        foreach (new DirectoryIterator($dirname) as $fileInfo) {
            $fname = $fileInfo->getFilename();
            $msg .= " in dir $dirname found file $fname $eol";
        } // end foreach
        return $msg;
    } // end function uploadWebsite
    /**
     * Move a single uploaded file from $_FILES to a target directory.
     *
     * Performs the following steps:
     *  - Verifies that an entry for $fileToUpload exists in $_FILES.
     *  - Resolves the original filename via basename().
     *  - Declares allowed extensions (jpg, jpeg, png, gif, html) — note: the current implementation declares these but does not enforce extension filtering.
     *  - Validates $dirname (must be non-empty), confirms the directory exists and is writable.
     *  - Constructs the destination path and, if a file already exists there, attempts to delete it if writable.
     *  - Validates the uploaded file with is_uploaded_file().
     *  - Moves the uploaded temporary file to the destination using move_uploaded_file().
     *  - On success, returns a message including the destination path, file size and upload timestamp.
     *
     *
     * @param string $fileToUpload The key/name of the file field as presented in the $_FILES superglobal.
     * @param string $dirname      The destination directory path where the file should be moved.
     *
     * @return string A status/logging message describing the outcome. On success this contains an informational message (including file size and timestamp). On some error conditions the method returns a message string; other error conditions throw an Exception.
     *
     * @throws Exception If:
     *   - $dirname is empty,
     *   - the directory does not exist,
     *   - the directory is not writable,
     *   - an upload error is detected,
     *   - the uploaded file is not a valid uploaded file,
     *   - move_uploaded_file fails.
     *
     * Notes:
     *  - The method expects $_FILES[$fileToUpload] to be present and populated by a valid HTTP file upload.
     *  - Callers should ensure the destination directory is correct and writable by the PHP process.
     *  - The method uses basename() on client-supplied filenames; treat filenames as untrusted input and normalize/sanitize as needed before trusting or exposing them.
     */
    private static function moveOneFile($fileToUpload, $dirname)
    {
        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        $debugMove = false;
        if (!array_key_exists($fileToUpload, $_FILES)) { // was file name entered.{
            $msg .= "$errorBeg E#2847 file specified by $fileToUpload was not uploaded $errorEnd";
            return $msg;
        }
        $filename = basename($_FILES[$fileToUpload]['name']);
        $allowedExtensions = array('jpg', 'jpeg', 'png', 'gif', 'html');
        $fileExt = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        if (empty($dirname))
            throw new Exception("$msg $errorBeg E#2848 no directory name entered for $fileToUpload $errorEnd");
        // Check if directory exists and is writable
        if (!is_dir($dirname)) {
            throw new Exception("$msg $errorBeg E#2849 A directory '$dirname' does not exist for $fileToUpload $errorEnd");
        }
        if (!is_writable($dirname)) {
            throw new Exception("$msg $errorBeg E#2850 directory '$dirname' is not writable for $fileToUpload $errorEnd");
        }
        $fileNameFull = "$dirname/$filename";
        if ($debugMove) $msg .= "$fileNameFull" . $eol;
        if (file_exists($fileNameFull)) {
            if (is_writable($fileNameFull)) {
                unlink($fileNameFull);
                if ($debugMove) $msg .= "Deleted existing $fileNameFull $eol";
            } else {
                $msg .= "$errorBeg E#2851 Cannot delete $fileNameFull because it is not writable $errorEnd";
                return $msg;
            }
        } // end if file exists
        if ($debugMove) $msg .= "uploading a file to $fileNameFull $eol";
        if ($_FILES[$fileToUpload]['error']) { //if  errors...
            $uploadErrors = array(
                UPLOAD_ERR_OK         => 'There is no error, the file uploaded with success.',
                UPLOAD_ERR_INI_SIZE   => 'The uploaded file exceeds the upload_max_filesize directive in php.ini.',
                UPLOAD_ERR_FORM_SIZE  => 'The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.',
                UPLOAD_ERR_PARTIAL    => 'The uploaded file was only partially uploaded.',
                UPLOAD_ERR_NO_FILE    => 'No file was uploaded.',
                UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder.',
                UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
                UPLOAD_ERR_EXTENSION  => 'A PHP extension stopped the file upload.'
            );
            $errorCode = $_FILES[$fileToUpload]["error"];
            $errorMessage = isset($uploadErrors[$errorCode]) ? $uploadErrors[$errorCode] : "Unknown upload error code: $errorCode";
            $fileNameFull = $dirname . DIRECTORY_SEPARATOR . $filename;
            throw new Exception("$msg ");
        }
        $tmpName = $_FILES[$fileToUpload]['tmp_name'];
        if (!is_uploaded_file($tmpName)) {
            $msg .= "$errorBeg E#2852 The file $filename is not a valid uploaded file $errorEnd";
            throw new Exception("$msg ");
        }
        $result = move_uploaded_file($tmpName, $fileNameFull);
        if ($result === false) {
            $msg .= "$errorBeg E#2853 Oops!  Your upload triggered the following error on move_uploaded_file:  " .
                $_FILES[$fileToUpload]["error"] . $errorEnd;
            throw new Exception("$msg ");
        } else {
            $fileSize = filesize($fileNameFull);
            $uploadTime = date("Y-m-d H:i:s");
            $msg .= "I#2854 moveOneFile: $fileNameFull uploaded successfully ($fileSize bytes) at $uploadTime $eol";
        }
        if ($debugMove) {
            foreach (new DirectoryIterator($dirname) as $fileInfo) {
                $fname = $fileInfo->getFilename();;
                $msg .= " in dir $dirname found file $fname $eol";
            } // end foreach
        }
        return $msg;
    } // end function moveOneFile
    private static function makeXML($trailId)
    {
        $msg = "";
        $debugMakeXML = false;
        $outXML = '<?xml version="1.0" encoding="UTF-8"?>
        <urlset
              xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
              xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
              xsi:schemaLocation="http://www.sitemaps.org/schemas/sitemap/0.9
                    http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd">
        <!-- created in freewheeling-build-website.php -->
        ';
        $outXML .= self::OutputXMLloc($trailId, "index.html", "1.0");
        $outXML .= self::OutputXMLloc($trailId, "about.html", "0.4");
        $outXML .= self::OutputXMLloc($trailId, "localhistory.html", "0.8");
        $outXML .= self::OutputXMLloc($trailId, "development.html", "0.8");
        $outXML .= self::OutputXMLloc($trailId, "accesspoints.html", "0.8");
        $outXML .= self::OutputXMLloc($trailId, "extension.html", "0.8");
        $outXML .= self::OutputXMLloc($trailId, "resources.html", "0.7");
        $outXML .= self::OutputXMLloc($trailId, "store.html", "0.6");
        $outXML .= self::OutputXMLloc($trailId, "amenities.html", "0.8");
        $outXML .= "</urlset>";
        $msg .= self::writeTheHtmlFile($trailId, $outXML, "sitemap"); // write the file
        return $msg;
    }
    private static function OutputXMLloc($trailId, $page, $priority)
    {
        $modDate = date("Y-m-d");
        $outXML = "
    <url>
        <loc>https://$trailId.freewheelingeasy.com/$page</loc>
        <lastmod>$modDate</lastmod>
        <changefreq>monthly</changefreq>
        <priority>$priority</priority>
    </url>
    ";
        return $outXML;
    }
} // end class freewheeling_build_website