<?php
class freewheelingLineTable
{
    public $distAlongMeters = array();
    private $lat = array();
    private $lon = array();
    private $lineName = array();
    public $milePostsAlong = array();
    private $line_id = array();
    private $seq = array();
    private $lineTrailId = array();
    private $trailId = "";
    private $InvalidMileage = 99999.9;
    /*
    *  Constructs an internal line table based on the trail id or group id
    Table contents:
        // lineId[] the lineId of the line back from$this point
                note:  lineId[I] to linId[I+1] represent a line segment of the trail
        // list of points lat[], lng[] along th line
        // distAlongMeters[] distance from start of line to$this point
        // milePostsAlong[] of start$this of$this line
        //trailId[] the trail id for$this line segment
        //trailName[] the trail name for$this line segment
    *
        // currently (2/2021) used by:
        // move icons to the line
        // Distance verification routine.\$trailForLine\
        // Typically used with GivenLatLonFindNearestLatLonLinePoint
        // note: sequence, trailName, lineName array built, but not global
        // older version of$this routine used global variables
                global $lat, $this->lon, $milePostsAlong, $line_id, $seq, $distAlongMeters, $trailIdForLine;
        Return types:
            $pointInformation = array(
                "matchPoint" => -1,
                "msg" => $msg,
                "metersToPoint" => 0,
                "sequence" => 0,
                "lineId" => 0),
                "latLine" => 0, "lngLine" => 0, "milepost" => 0                                    // at the given "matchPont"
            );
    USage:
        $latTable = new freewheelingLineTable($trailOrGroupId, $description, $msg);             // builds the internal table
        :int numPoints = $latTable->get_latCount();                                                // number of points in the table
        :float milesAtPoint = $latTable->findMilePostsAlongAt($latitude, $longitude, $description);    // find miles along at a given lat, lon
        :pointInformation = $latTable->findNearestLatLonLinePoint($latSearch, $lonSearch, $description = "description not supplied"); // find nearest point on line
    Internal functions:
        :pointInformation = $this->FillResults($foundPoint);                                            // given point on the array fill common result fields
        $msg .= $latTable->dumpMilageTableByPoints($start = 0, $end = 32000, $title = "", $matchPoint = -1)
*
*/
    public function __construct($trailOrGroupId, $description, &$msg)
    {
        global $eol, $errorBeg, $errorEnd;;
        global $wpdbExtra;
        global $freeWheelingEasy_ipHome;
        try {              //collection of points
            $debugBuildDistance = false;
            $this->distAlongMeters = array();
            $this->lat = array();
            $this->lon = array();
            $this->lineName = array();
            $this->milePostsAlong = array();
            $this->line_id = array();
            $this->seq = array();
            $this->lineTrailId = array();
            // the trail Id of the data re are working on
            $this->trailId = $trailOrGroupId;
            if ($debugBuildDistance) $msg .= "I#1045 BuildTableOfPointsALongLines:( " . $this->trailId . ", $description $eol)";
            //  -----------------------------  build an array of begin end points, sequence
            if (freeWheeling_edit_setGlobals::notAllowedToEdit("distance calculator"))
                throw new Exception("$errorBeg E#1017 doKmlRead permission has not been granted. Are you logged in? $errorEnd");
            if (empty($trailOrGroupId)) {
                $msg .= rrwFormat::backtrace("E#1021 did not get a trailD or groupId to work with");
                throw new Exception("$msg $errorBeg lineTable($trailOrGroupId, $description)$errorEnd");
            }
            $sqlWhere = freewheeling_kml_common::sqlWhereForGroupOrTrailLines($trailOrGroupId); // returns a where clause
            // print "sqlWhere = $sqlWhere $eol";
            $sqlLines = "select distinct trailId, latStart, lngStart, latEnd, lngEnd,
				            lineName, lengthMeters, sequence, pointList, lineId
				    from $wpdbExtra->lines line
                             where $sqlWhere and lengthMeters > 0
				    order by sequence";
            $recLines = $wpdbExtra->get_resultsA($sqlLines);
            $numLats = 0;
            $distSum = 0; // distance along trail at end of line from start
            if (0 == $wpdbExtra->num_rows || is_null($recLines)) {
                $msg .= rrwFormat::backtrace("E#1038 did not find any lines to work with
                                perhaps \"$sqlWhere\" is in error $errorEnd $sqlLines $eol");
                throw new Exception($msg);
            }
            //
            if ($debugBuildDistance) $msg .= "I#1019 found " . count($recLines) . " lines to work with $eol";
            $distSum = 0.0;   // length of all the lines
            $lineIdPast = $recLines[0]["lineId"];
            $latLinePast = $recLines[0]["latStart"];
            $lonLinePast = $recLines[0]["lngStart"];
            $lineNamePast = $recLines[0]["lineName"];
            $sequencePast = $recLines[0]["sequence"];
            $firstTime = true;      // deals wit the first point has no previous point
            $latLinePast = 0;       // set so the first point is not = to non existent past point
            $lonLinePast = 0;       // therefore we will load the first point into the lat array
            foreach ($recLines as $recLine) {
                $trailId = $recLine["trailId"];
                $lineName = $recLine["lineName"];
                $sequence = $recLine["sequence"];
                $pointList = $recLine["pointList"];
                $lengthMeters = $recLine["lengthMeters"];
                $LatStart  = $recLine["latStart"];
                $lonStart = $recLine["lngStart"];
                $latEnd = $recLine["latEnd"];
                $lonEnd = $recLine["lngEnd"];
                $lineId = $recLine["lineId"];
                if (-999 == $lineId ||  -999 == $sequence)  // debug particular lines
                    $debugBuildDistance = true;
                else
                    $debugBuildDistance = false;
                if ($debugBuildDistance) {
                    $msg .= "I#1032 $trailId, seq # $sequence length = $lengthMeters, name $lineName ";
                    $msg .= "start distSum = $distSum $eol $pointList $eol ";
                }
                if ((!$firstTime && ($latLinePast != $LatStart || $lonLinePast != $lonStart))) { // by the time we are i$this routine, lines should connect
                    $distErrorMeters = freewheeling_calculations::distanceMeters($latLinePast, $lonLinePast, $LatStart, $lonStart);
                    $mapLink = freewheelFormat::lineIdMapLink($lineId);
                    $EditLineLink = freewheelFormat::editLineLink($lineId);
                    $msg .= "$errorBeg I#1026 lineTable trying to connect $lineIdPast to $lineId - $mapLink line $EditLineLink, $sequence, start point does not match end of last line
                                    there is  a gap of $distErrorMeters meters in the line work $errorEnd
                                        $latLinePast, $lonLinePast, end previous$eol
                                        $LatStart, $lonStart start current $eol$eol";
                    if ($debugBuildDistance) {
                        $msg .= rrwUtil::print_r($points, true, "there are " . count($points) . " in line $lineName  $lineId");
                        $msg .= "<table>\n" . rrwFormat::HeaderRow("newLat", "newLng", "latLinePast", "lngPast", "distSum", "thisLineCalcMeters", "lineName");
                    }
                    throw new Exception($msg);
                }
                //
                $points = explode(" ", $pointList);
                $thisLineCalcMeters = 0;
                $latLon = explode(",", $points[0]);
                if (count($latLon) != 3) {
                    throw new Exception("$errorBeg E#1042 found " . count($latLon) . " points expected 3
                                    in $point on split of commas, in pointList on the line
                                    #$lineId, '$lineName' $errorEnd ");
                }

                foreach ($points as $point) {   // loop over points in this line
                    if (empty($point))
                        continue;
                    $latLon = explode(",", $point);
                    if (count($latLon) != 3) {
                        throw new Exception("$errorBeg E#1043 found " . count($latLon) . " points expected 3
                                    in $point on split of commas, in pointList on the line
                                    #$lineId, '$lineName' $errorEnd ");
                    }
                    $newLat = $latLon[1];  //
                    $newLng = $latLon[0];  //
                    if ($newLat == $latLinePast && $newLng == $lonLinePast)
                        continue; // avoid duplicate points
                    $this->lat[$numLats] = $newLat;
                    $this->lon[$numLats] = $newLng;
                    $this->seq[$numLats] = $recLine["sequence"];
                    $this->line_id[$numLats] = $lineId;
                    $this->lineTrailId[$numLats] = $recLine["trailId"];
                    $this->milePostsAlong[$numLats] = null;            // filled in by DistributeMileposts
                    if ($debugBuildDistance)
                        $msg .= "I#1641 point $numLats: $newLat, $newLng $eol";
                    if ($firstTime) {
                        $distMeters = 0;  // skip first point since no distance to prior point
                        $firstTime = false;
                    } else {
                        $distMeters = distanceMeters($newLat, $newLng, $latLinePast, $lonLinePast);
                        if ($distMeters > 10000) {
                            $msg .= "E#1065 Distance between two points is $distMeters bigger then " . 10000 / freeWheel::metersPerMile . " Miles ------------------------------------ $eol";
                            $msg .= rrwFormat::backtrace("E#1066 found in line $lineId, sequence $sequence, $lineName,
                                        found in line $lineId, sequence $sequence, $lineName,
                        between points $latLinePast, $lonLinePast and $newLat, $newLng ");
                        }
                    }
                    $distSum += $distMeters;
                    $thisLineCalcMeters += $distMeters;
                    $this->distAlongMeters[$numLats] = $distSum;
                    //
                    if ($debugBuildDistance)
                        $msg .= rrwFormat::CellRow(
                            $newLat,
                            $newLng,
                            $latLinePast,
                            $lonLinePast,
                            $distMeters,
                            $distSum,
                            $thisLineCalcMeters,
                            $lineName
                        );


                    $latLinePast = $newLat;
                    $lonLinePast = $newLng;

                    $this->lineName[$numLats] = $recLine["lineName"];
                    //
                    $numLats++;
                } // end foreach ($points as $point)
                // check overall length of line
                $diffCheckMeters = abs($thisLineCalcMeters - $lengthMeters);
                if ($diffCheckMeters > 5) {
                    $msg .= "E#1651 line id #$lineId, sequence $sequence, $points[0].lat, $points[0].lng, database length $lengthMeters, $eol
                            E#1665 line id #$lineIdPast, sequence $sequencePast, -
                                    $lineNamePast, $eol ";
                }
                if ($debugBuildDistance) {
                    $msg .= "</table>\n";
                    $msg .= $this->dumpMilageTableByPoints(0, 100, "number of points now = $numLats -
                                distSum at end of line total $distSum meters, line database $thisLineCalcMeters meters
                                -- database $lengthMeters $eol");
                }
                if (abs($thisLineCalcMeters - $lengthMeters) > 1) {
                    $diff = $thisLineCalcMeters - $lengthMeters;
                    $msg .= "$errorBeg E#1088 line id #$lineId, sequence $sequence, -
						            $lineName, length summed = $thisLineCalcMeters, but database length $lengthMeters,
                                    difference is $diff,  maybe a gap between segments of the line work. $errorEnd ";
                } // end abs( $thisLineCalcMeters - $lengthMeters )
                if ($debugBuildDistance) {  // at the end of each line
                    $msg .= "distance along the $numLats lines
								summed to $distSum " . round($distSum / freeWheel::metersPerMile, 1) . $eol;
                    $msg .= "I#1048 BuildTableOfPointsALongLines: built a lat,lng, length table of
                            " . count($this->lat) . " points $eol";
                    //         $msg .= $this->dumpMilageTableByPoints(0, 9999, "BuildTableOfPointsALongLines");
                }
                $latLinePast = $latEnd;
                $lonLinePast = $lonEnd;
            } // end foreach ($recLines as $recLine)
            $lastPoint = $this->get_latCount();
            if (1 == 0) {   // debugging
                for ($ii = 0; $ii < $lastPoint; $ii++) {
                    $msg .= $this->lat[$ii] . ", " . $this->lon[$ii] . ", 0 ";
                }
            } // end
            $msg .= "$eol I#1677 Build line Table:: built a lat,lng, length table of " . $this->get_latCount() . " points $eol";
            $msg .= "I#1668 total trail length is " . $this->distAlongMeters[$lastPoint - 1] . "  meters, " .  ($this->distAlongMeters[$lastPoint - 1] / freeWheel::metersPerMile) . " miles $eol";
        } // end try
        catch (Exception $ex) {
            $msg .= "$errorBeg  I#1054 thrown out of Build line Table:: " . $ex->getMessage() .  $errorEnd;
            throw new Exception("$msg");
        }
    } // end function __construct
    public function __destruct()
    {
        $this->distAlongMeters = array();
        $this->milePostsAlong   = array();
        $this->lat              = array();
        $this->lon              = array();
        $this->lineName         = array();

        $this->seq              = array();
        // Cleanup code here
    }
    public function get_latCount()
    {
        if (!is_array($this->lat))
            throw new Exception("E#1662 get_latCount: lat is not an array");
        return count($this->lat);
    }
    public function get_milePostsAlong($ii): float
    {
        $temp = $this->milePostsAlong[$ii];
        if (is_null($temp))
            return $this->InvalidMileage;
        return $temp;
    }
    public function get_invalidMileage(): float
    {
        return $this->InvalidMileage;
    }

    //  fills the table $milePostsAlong
    //      which is keyed by match point with value Milepost
    //  Assume BuildTableOfPointsALongLines has been run
    //      ie.e $lat, $lng, $milePostsAlong exists
    //  1)  Enter milage for each icon into table
    //  2)  distribute milage between two known point in the table.
    //  3)  deal with the end points not having a milepost
    public function BuildTableOfMilePostsAlongLines()
    {

        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        ini_set("display_errors", 1);
        $debugSetMiles = rrwParam::isDebugMode("debugSetMiles");
        try {
            $msg .= $this->verifyOrFixMilePointsOnLine();   // move any milepost icons that are not on the line to the line
            $msg .= $this->checkMilePointsOnLine();        // verify that the pervious line worked
            $msg .= "should have all points on the line now $eol";
            $msg .= $this->DistributeMileposts();
            $msg .= $this->fillInMileposts();
            $msg .= $this->addEstimatedMilepost();
            $msg .= $this->assignMilageToNulls();
            return $msg;
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg E#1001 bottom of BuildTableOfMilePostsAlongLines $errorEnd";
            throw new Exception($msg);
        }
    } // end function BuildTableOfMilePostsAlongLines


    public function verifyOrFixMilePointsOnLine()
    {
        // ------------------------------------------verify mileage points are on line
        // if not on line, add a point at that place.
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra, $wpdb;
        $msg = "";
        try {
            $trailOrGroupId = $this->trailId;
            $sqlWhere = $wpdbExtra->sqlWhereIcons($trailOrGroupId);
            $sqlIcons = "select iconId, iconName, latitude, longitude, milepost, onMap, trailId
                        from $wpdbExtra->icons where $sqlWhere and (iconName like 'mile%' or
                        onMap like 'milepost%' or onMap= 'launch' or iconName like 'break Line%')
                        order by milepost ";
            $recIcons = $wpdbExtra->get_resultsA($sqlIcons);
            $msg .= "I#1632 There are " . $wpdbExtra->num_rows . " milepost/launch points located on the line $eol";
            //      first check that all mile and launch points are on the line
            foreach ($recIcons as $recIcon) {
                $msg .= $this->verifyOrFixRecPointOnLine($recIcon);
            } // end foreach - on to next icon
            return $msg;
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg verifyOrFixMilePointsOnLine() $errorEnd";
            throw new Exception($msg);
        }
    } // end verifyOrFixMilePointsOnLine
    private function verifyOrFixRecPointOnLine($recIcon)
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        $msg = "";
        $debugVerify = rrwParam::isDebugMode("debugVerify");
        try {
            $iconId = $recIcon["iconId"];
            $iconName = $recIcon["iconName"];
            $milepost = $recIcon["milepost"];
            $latitude = $recIcon["latitude"];
            $longitude = $recIcon["longitude"];
            $description = "$iconName-$iconId at milepost $milepost";
            if ($milepost == 72)
                $debugVerify = true;
            if ($debugVerify) {
                $msg .= "$eol I#1685 Check for <strong>$iconName-$iconId</strong> at $latitude, $longitude on line $eol";
                $cnt3 = $this->get_latCount();
                for ($ii = 0; $ii < $cnt3; $ii++) {
                    //           $msg .= $this->lat[$ii] . ", " . $this->lon[$ii] . ", 0 ";
                }
            }
            $checkResults = $this->GivenLatLonFindNearestLatLonLinePoint(latSearch: $latitude, lonSearch: $longitude, description: $description, msg: $msg);
            $newLat = $checkResults->lat;
            $newLon = $checkResults->lon;
            // icon maybe close to existing point, but not zero distance away.
            if ($newLat == 0 || $newLon == 0) {
                $msg .= "$eol I#1690 <strong>GivenLatLonFindNearestLatLonLinePoint</strong> did not find any point near by $description $eol";
                $msg .= $this->dumpMilageTableByLatitude($latitude, 0, "I#1692 dump of line points for $description ");
                throw new Exception($msg);
            }

            $msg .= $this->GivenLatLonDoInsertIfRequired($checkResults);
            // now there is a point on the line at newLat,newLon move there icon to that point
            $sqlMove = "update $wpdbExtra->icons set latitude = $newLat, longitude = $newLon where iconId = $iconId";
            if ($debugVerify) $msg .= "I#1675 move icon to  $sqlMove $eol";
            $wpdbExtra->query($sqlMove);
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg verifyOrFixMilePointsOnLine() $errorEnd";
            throw new Exception($msg);
        }
        return $msg;
    } // end verifyOrFixMilePointsOnLine

    public  function DistributeMileposts()
    {
        // puts existing milePoints into the milePostsAlong array
        global $eol, $errorBeg, $errorEnd;;
        global $wpdbExtra;
        global $milesDirectionAlongTheLine;       // set this to 1 or -1 depending on the direction of the mileposts along the line.
        //                                          // set based on difference in paired mileposts or
        //                                          // if only mile post then by onMap contents (neg or not)
        $msg = "";
        $debugMiles = rrwParam::Boolean("debugMiles");
        $debugMatch = rrwParam::Boolean("debugMatch");
        try {
            if ($debugMiles || $debugMatch) $msg .= "I#1667 DistributeMileposts for $this->trailId $eol";
            $numOfLats = $this->get_latCount();
            if ($debugMiles) {
                $msg .= "I#1701 total number of points is $numOfLats $eol";
                for ($ii = 0; $ii < $numOfLats - 1; $ii++) {
                    $msg .= $this->lat[$ii] . ", " . $this->lon[$ii] . ", 0 ";
                }
            }
            $this->milePostsAlong = array();
            for ($ii = 0; $ii < $numOfLats; $ii++) {
                $this->milePostsAlong[$ii] = null;
            } // end for
            $sqlIcons = "select iconName, latitude, longitude, milepost, onMap from $wpdbExtra->icons
                            where trailId = '$this->trailId' and onMap like 'milepost%' order by milepost";
            $recIcons = $wpdbExtra->get_resultsA($sqlIcons);
            if ($debugMiles) $msg .= "I#1669 found " . $wpdbExtra->num_rows . " points with a milepost value - $sqlIcons $eol";
            $badMilePoint = 0;
            $FoundOnMapNegative = false;
            $cntLimit = 0;  // prevent runaway looping
            foreach ($recIcons as $recIcon) {
                $cntLimit++;
                if ($cntLimit > 500) {
                    $msg .= rrwFormat::backtrace("DistributeMileposts");
                    throw new Exception("$errorBeg E#1689 too many milepost icons found $errorEnd");
                }
                $iconName = $recIcon["iconName"];
                $latitude = round($recIcon["latitude"], freeWheel::latRoundTo);
                $longitude = round($recIcon["longitude"], freeWheel::latRoundTo);
                $milepost = $recIcon["milepost"];
                $onMap = $recIcon["onMap"];
                if (strpos($onMap, "neg") !== false) {
                    $FoundOnMapNegative = true;
                }
                if (!is_numeric($milepost)) {
                    throw new Exception("$errorBeg E#1674 milepost value is not numeric for $iconName with milepost value '$milepost' $errorEnd");
                }
                $result =  $this->GivenLatLonFindExactMatch($latitude, $longitude);
                if ($result->matchPoint < 0) {
                    // should not happen since verifyOrFixMilePointsOnLine should have fixed it
                    $msg .= str_repeat("V", 80);
                    $msg .= " E#1031 $onMap - $iconName at milepost $milepost with $latitude, $longitude was not in lat list$eol";
                    $result =  $this->GivenLatLonFindNearestLatLonLinePoint($latitude, $longitude, $iconName, $msg);
                    $matchPoint = $result->matchPoint;
                    $msg .= $this->dumpMilageTableByPoints($matchPoint - 5, $matchPoint + 5, "I#1710 in DistributeMileposts icon latLon not in pointList", $matchPoint);
                    $msg .= rrwFormat::backtrace("I#1718 DistributeMileposts");
                    $msg .= str_repeat("^", 80);
                    throw new Exception($msg);
                }
                if ($debugMatch) $msg .= "$iconName --- $milepost";
                $matchPoint = $result->matchPoint;
                if ($debugMatch) $msg .= "setting $matchPoint to $milepost for $iconName  $eol";
                $this->milePostsAlong[$matchPoint] = $milepost;
                if ($debugMiles) {
                    $msg .= "I#1700 found milepost $milepost for $iconName at point $matchPoint with lat,lon = " . $this->lat[$matchPoint] . "," . $this->lon[$matchPoint] . " $eol";
                    $msg .= $this->dumpMilageTableByPoints($matchPoint - 5, $matchPoint + 5, "Distribute s found Mileposts", $matchPoint);
                } // end debug
            }
            if ($badMilePoint > 0)
                $msg .= "E#1696 Found $badMilePoint mileposts that were not in the lat,lng list";
            // build and check direction of the mileposts
            $preMilepost = null;
            $preII = null;
            $milesDirectionAlongTheLine = 0;    // not yet chosen
            for ($ii = 0; $ii < $numOfLats; $ii++) {
                if (is_null($this->milePostsAlong[$ii]))
                    continue;
                if (!is_null($preMilepost)) {
                    $direction = $this->milePostsAlong[$ii] <=> $preMilepost;
                    if ($debugMiles) $msg .= "I#1697 milepost $ii - " . $this->milePostsAlong[$ii] . " - $preMilepost = $direction, previous $milesDirectionAlongTheLine $eol";
                    if (0 == $milesDirectionAlongTheLine)
                        $milesDirectionAlongTheLine = $direction;
                    if ($milesDirectionAlongTheLine != $direction) {
                        $msg .= "$errorBeg E#1681 $this->trailId  milepost direction changed between " . $this->milePostsAlong[$ii] . " and $preMilepost $errorEnd ";
                        $msg .= "<table>";
                        foreach ($recIcons as $recBad) {
                            $msg .= rrwFormat::CellRowColor($recBad["iconName"], $recBad["milepost"], $recBad["latitude"], $recBad["longitude"], $recBad["onMap"]);
                        }
                        $msg .= self::dumpMilageTableByPoints($ii - 5, $ii + 5, "I#1716 DistributeMileposts", $ii);
                        $msg .= self::dumpMilageTableByPoints($preII - 5, $preII + 5, "I#1717 DistributeMileposts", $preII);
                        throw new Exception($msg);
                    }
                }
                $preMilepost = $this->milePostsAlong[$ii];
                $preII = $ii;
            } // end for ($ii = 0; $ii < $numOfLats; $ii++)
            if (0 == $milesDirectionAlongTheLine) {
                if ($FoundOnMapNegative) {
                    $msg .= "$errorBeg I#1015 only one milepost, direction set to opposite line direction $errorEnd";
                    $milesDirectionAlongTheLine = -1;
                } else {
                    $msg .= "$errorBeg  I#1014 only one milepost, direction set to direction of line $errorEnd";
                    $milesDirectionAlongTheLine = 1;
                }
            }
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg E#1002 bottom of DistributeMileposts $errorEnd";
            throw new Exception($msg);
        }
        if ($debugMiles || $debugMatch) $msg .= "I#1698 DistributeMileposts done for $this->trailId $eol";
        return $msg;
    } // end DistributeMileposts

    /**
     * Finds the distance in miles along a route at a given latitude and longitude point.
     *
     * This method searches for the nearest point on the route line to the given coordinates
     * and calculates the distance along the route to that point. If the coordinates are not
     * exactly on a route point, it interpolates between adjacent points to determine the
     * precise position along the route.
     *
     * @param float $latitude The latitude coordinate to search for
     * @param float $longitude The longitude coordinate to search for
     * @param string $description Optional description for error messages (default: "description not supplied")
     *
     * @return float The distance in miles along the route to the specified coordinates
     *
     * @throws Exception When the coordinates cannot be found within acceptable distance of the route
     *                   or when the point is beyond the available route points
     *
     */
    public function findMilePostsAlongAt($latitude, $longitude, $description = "description not supplied"): float
    {
        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        $debugMiles = rrwParam::isDebugMode("milePostsAlong");
        $results = $this->GivenLatLonFindNearestLatLonLinePoint($latitude, $longitude, $description, $msg);
        if (! $results->linePointExists) {
            $numPoints = count($this->lat);
            if ($debugMiles) $msg .= "I#1735 looking for $latitude, $longitude, $description  among $numPoints $eol";
            $msg .= rrwUtil::print_r($results, true, "the results of the search");
            $msg .= $this->dumpMilageTableByPoints(0, count($this->lat), "findMilePostsAlongAt");
            throw new Exception("$msg E#1648 findMilePostsAlongAt: did not find a point");
        }
        $metersToPoint = distanceMeters($latitude, $longitude, $results->lat, $results->lon);
        if ($metersToPoint < 5)
            return $metersToPoint / freeWheel::metersPerMile;        // close enough for a match
        $matchPoint = $results->matchPoint;
        if (
            $this->between($this->lat[$matchPoint], $latitude, $this->lat[$matchPoint - 1]) &&
            $this->between($this->lon[$matchPoint], $longitude, $this->lon[$matchPoint - 1])
        ) {
            $ratio = ($latitude - $this->lat[$matchPoint]) / ($this->lat[$matchPoint - 1] - $this->lat[$matchPoint]);
            $meters = $this->distAlongMeters[$matchPoint - 1] +
                $ratio * ($this->distAlongMeters[$matchPoint - 1] - $this->distAlongMeters[$matchPoint]);
            $msg .= "I#3334" . $this->distAlongMeters[$matchPoint - 1] . ", " . $this->distAlongMeters[$matchPoint] . ", $ratio, $meters $eol";
            return ($meters / freeWheel::metersPerMile);
        } elseif ($matchPoint == count($this->lat)) {
            $numPoints = count($this->lat);
            if ($debugMiles) $msg .= "I# 1734 looking for $latitude, $longitude, $description among $numPoints $eol";
            $msg .= "it is beyond the list of available points $eol";
            $msg .= $this->dumpMilageTableByPoints($matchPoint - 5, $matchPoint + 5, "findMilePostsAlongAt", $matchPoint);
            $msg .= rrwFormat::backtrace("E#1077 findMilePostsAlongAt:point '$description' was not found within 10 meters of some point on the line");
            throw new Exception("$msg");
        } elseif (
            $this->between($this->lat[$matchPoint], $latitude, $this->lat[$matchPoint + 1]) &&
            $this->between($this->lon[$matchPoint], $longitude, $this->lon[$matchPoint + 1])
        ) {
            $ratio = ($latitude - $this->lat[$matchPoint]) / ($this->lat[$matchPoint + 1] - $this->lat[$matchPoint]);
            $meters = $this->distAlongMeters[$matchPoint] +
                $ratio * ($this->distAlongMeters[$matchPoint + 1] - $this->distAlongMeters[$matchPoint]);
            if ($debugMiles) {
                $msg .= $this->distAlongMeters[$matchPoint - 1] . ", " . $this->distAlongMeters[$matchPoint] . ", $ratio, $meters $eol";
                $msg .= "I#3335 " . $this->distAlongMeters[$matchPoint - 1] . "," . $this->distAlongMeters[$matchPoint] . ", $ratio, $meters $eol";
            }
            return $meters / freeWheel::metersPerMile;
        } else {
            $msg .= rrwUtil::print_r($results, true, "the results of the search");
            $numPoints = count($this->lat);
            $msg .= "I# 1723 looking for $latitude, $longitude, $description among $numPoints $eol";
            $msg .= $this->dumpMilageTableByPoints($matchPoint - 5, $matchPoint + 5, "findMilePostsAlongAt", $matchPoint);
            $msg .= rrwFormat::backtrace("E#1078 findMilePostsAlongAt:point '$description' was did not found within 10 meters of some point on the line");
            throw new Exception("$msg");
        }
    }
    /**
     * Find the exact point on a line to a given latitude and longitude coordinate.
     *
     * This method searches through the line's coordinate points to find the exact match
     * to the provided search coordinates.
     *
     * @param float $latSearch The latitude coordinate to search for
     * @param float $lonSearch The longitude coordinate to search for
     * @param string $description Optional description for error/debug messages (default: "description not supplied")
     *
     * @return array Returns a LineTableInsert object  with the following keys:
     *              - matchPoint: Index of the nearest point (-1 if no exact point found )
     *              - latitude, longitude of the point on the line
     *              - pointIsOn: Boolean indicating if the point is exactly on the line.  i.e. no need to insert a new point
     */
    public  function GivenLatLonFindExactMatch($latSearch, $lonSearch): lineTableInsert
    {
        for ($ii = 0; $ii < count($this->lat); $ii++) {
            if ($latSearch == $this->lat[$ii] && $lonSearch == $this->lon[$ii]) {
                $result = $this->FillResults($ii);
                $result->linePointExists = true;        // the point is exactly on the line, so we can move the source point to this point
                return $result;
            }
        } // end for
        $result = $this->FillResults(-1); // not found, return an invalid index
        return $result;
    } // end function GivenLatLon_ret_mileAlong
    private function givenLatLonFindNearestMatchPoint($latSearch, $lonSearch, $description, &$msg): lineTableInsert
    {
        global $eol, $errorBeg, $errorEnd;
        $debugInsert = rrwParam::isDebugMode("debugInsert");
        if ($debugInsert) $msg .= "I#1673 givenLatLonFindNearestMatchPoint: looking for $description at $latSearch, $lonSearch $eol";
        // assume that $lat ... exists since they are set up in the constructor
        $dist = 1;
        for ($ii = 0; $ii < count($this->lat); $ii++) {
            $latPoint = $this->lat[$ii];
            $lonPoint = $this->lon[$ii];
            $DeltaLat = abs($latSearch - $latPoint);
            $DeltaLon = abs($lonSearch - $lonPoint);
            $sudoDistanceDegrees = $DeltaLat + $DeltaLon;
            if ($sudoDistanceDegrees > $dist)
                continue;
            $dist = $sudoDistanceDegrees;
            $iiSave = $ii;
        } // end for
        if ($dist == 1) {
            $msg .= "$eol I#1691 GivenLatLonFindNearestMatchPoint: did not find any point near by $description $eol";
            $msg .= $this->dumpMilageTableByLatitude($latSearch, 0, "I#1693 dump of line points for $description ");
            throw new Exception($msg);
        }
        // assert $dist < 1. i.e. something found
        $foundIt = $this->FillResults($iiSave);
        return $foundIt;
    } // end function givenLatLonFindNearestMatchPoint
    /**
     * Find the nearest point on this line (defined by $this->lat/$this->lon vertices) to a given latitude/longitude.
     *
     * Behavior:
     * - Checks both line endpoints first; accepts a match if within 3 meters.
     * - If the search point matches an existing vertex within ~1e-6 degrees, returns that vertex.
     * - Iterates over segments, projects the search latitude onto the segment to compute a candidate point,
     *   measures distance to the projected point, and selects the closest candidate within 3 meters.
     * - If a suitable point is found, returns an insertion descriptor with index, coordinates, and a flag
     *   indicating whether the point lies exactly on an existing vertex; otherwise returns an invalid index (-1).
     * - Appends diagnostic messages to $msg when debug flags are enabled via rrwParam::isDebugMode().
     *
     * Preconditions:
     * - $this->lat and $this->lon are aligned arrays of equal length (>= 2).
     * - Helper methods/functions exist: $this->between(), distanceMeters(), freewheeling_calculations::distanceMeters().
     *
     * Side effects:
     * - Modifies $msg by appending debug and error information.
     * - Uses global formatting strings $eol, $errorBeg, $errorEnd for messages.
     *
     * @param float  $latSearch    Latitude of the target point.
     * @param float  $lonSearch    Longitude of the target point.
     * @param string $description  Contextual description used in debug output.
     * @param string &$msg         Accumulator for diagnostic messages (appended by this method).
     * @return lineTableInsert     Descriptor of the insertion location and computed coordinates.
     *      public $matchPoint;   index of the nearest point on the line (-1 if none found)
     *      public $newLat;        projected latitude or existing close point on the line
     *      public $newLon;        projected longitude or existing close point on the line
     *      public $linePointExists;    true if the point is close to a point on the line (move the source point)
     *      public $description;    description of the point from input
     * @throws Exception           On unexpected errors; contextual information is appended to $msg.
     */
    public function GivenLatLonFindNearestLatLonLinePoint($latSearch, $lonSearch, $description, &$msg): lineTableInsert
    {
        global $eol, $errorBeg, $errorEnd;
        try {
            $debugInsert = rrwParam::isDebugMode("debugInsert");
            $debugEnds = rrwParam::isDebugMode("debugEnds");
            $debugOneLineName = rrwParam::isDebugMode(item: "debugOneLine");;
            if ($debugInsert) $msg .= "I#1672 GivenLatLonFindNearestLatLonLinePoint: looking for $description at $latSearch, $lonSearch $eol";
            if (strpos($description, "Rangos") === false)
                $debugOneLine = false;
            else
                $debugOneLine = true;
            // assume that $lat ... exists since they are set up in the constructor
            $acceptableDistanceMeters = 3;
            $lastPoint = count($this->lat) - 1;

            foreach (array(0, $lastPoint) as $ii) {  // check the ends first
                $dist = distanceMeters($this->lat[$ii], $this->lon[$ii], $latSearch, $lonSearch);
                if ($debugEnds)
                    $msg .= "I#1055 checking end point $ii: " . $this->lat[$ii] . ", " . $this->lon[$ii] . " <$latSearch>, $lonSearch, dist = $dist $eol";
                if ($dist > $acceptableDistanceMeters)
                    continue;
                $foundIt = $this->FillResults($ii);
                if ($dist == 0) {
                    $foundIt->linePointExists = true;        // the point is exactly on the line, so we can move the source point to this point
                    $foundIt->description = $description;
                } else {
                    $foundIt->linePointExists = false;       // the point is close to the line, but not exactly on it, so we need to insert a new point
                    $foundIt->description = "close to  point $description";
                }
                if ($debugOneLine || $debugInsert) $msg .= "I#1644 found close end point match at point $ii dist = $dist $eol";
                return $foundIt;
            } // end foreach checking the lne end points
            $bestDistMeter = 1000000; // start big
            $iiSave = -1;
            if ($debugInsert) $msg .= "I#2071 did not find a match at the ends, now looking for $latSearch, $lonSearch, $description on line $eol";
            foreach ($this->lat as $ii => $latPoint) {
                if ($ii >= $lastPoint)
                    break; // we are at the end, there is  no $ii + 1 point
                // if ($debugInsert) $msg .= "I#1642 checking point $ii: $latPoint, " . $this->lat[$ii] . " <$latSearch> " . $this->lat[$ii + 1] . " longitude " . $this->lon[$ii] . $eol;
                // is it exact or really close to an existing point?
                if (abs($this->lat[$ii] - $latSearch) < 0.000001 && abs($this->lon[$ii] - $lonSearch) < 0.000001) {
                    $foundIt = $this->FillResults($ii);
                    $foundIt->linePointExists = true;        // the point is exactly on the line, so we can move the source point to this point
                    if ($debugInsert) $msg .= "I#1645 found point match at point $ii dist = 0 $eol";
                    return $foundIt; // our work is done
                }
                // is the lat between the two points
                if (! $this->between($this->lat[$ii], $latSearch, $this->lat[$ii + 1])) {
                    //                   if ($debugInsert) $msg .= "I#1643 $description is NOT between two points, continue$eol";
                    continue;   // not between. keep looking
                }
                if (! $this->between($this->lon[$ii], $lonSearch, $this->lon[$ii + 1])) {
                    if ($debugInsert) $msg .= "I#1646 $ii lat was between two points, but lon was not , continue$eol" .
                        $this->lat[$ii] . ", " . $this->lon[$ii] . "$eol $latSearch, $lonSearch" . $eol . $this->lat[$ii + 1] . ", " . $this->lon[$ii + 1] . $eol;
                    continue;   // not between. keep looking
                }
                if ($debugInsert) $msg .= "I#1647 $ii lat, lon was between two points, $eol" .
                    $this->lat[$ii] . ", " . $this->lon[$ii] . "$eol $latSearch, $lonSearch" . $eol . $this->lat[$ii + 1] . ", " . $this->lon[$ii + 1] . $eol;

                // is between two points on the line.
                if ($debugInsert) $msg .= "yes $description is between two points, // calculate the point on the line based on lat ratios$eol";
                $ratioLat = ($latSearch - $this->lat[$ii]) / ($this->lat[$ii + 1] - $this->lat[$ii]);
                // determine the corresponding longitude,latitude for a point on the line on the line
                $calcLatitude = round(($ratioLat * ($this->lat[$ii + 1] - $this->lat[$ii])) + $this->lat[$ii], freeWheel::latRoundTo);
                $calcLongitude = round($ratioLat * ($this->lon[$ii + 1] - $this->lon[$ii]) + $this->lon[$ii], freeWheel::latRoundTo);
                // calcLat, calLng is on the line maybe near the input point
                if ($debugInsert) $msg .= "I#1652 $description ratioLat = $ratioLat, calcLat = $calcLatitude, calcLng = $calcLongitude $eol";
                $distanceMeterToLine = freewheeling_calculations::distanceMeters($latSearch, $lonSearch, $calcLatitude, $calcLongitude);
                if ($distanceMeterToLine > 3) {
                    $bestDistMeter = min($bestDistMeter, $distanceMeterToLine);
                    if ($debugInsert) $msg .= "I#1653 <strong>distanceMeterToLine</strong> = $distanceMeterToLine which is too far > 3 meters $eol";
                    continue; //  not close enough to the line, continue looking
                }
                $iiSave = $ii;  // save this one for now
                $calcLatitudeSaved = $calcLatitude;
                $calcLongitudeSaved = $calcLongitude;
                if ($debugInsert) $msg .= "I#1655 $description found between point $iiSave " . ($iiSave + 1) . ",
                            ratioLat = $ratioLat, calcLat = $calcLatitude, calcLng = $calcLongitude,
                            <strong>distanceMeterToLine</strong> = $distanceMeterToLine $eol";
                continue; // found distance < 3, bt there may be one closer
            }  // end foreach over lat array
            // done looking
            if (-1 == $iiSave) {
                if ($debugInsert) $msg .= "I#1659 did not find a calculated point within 3 meters $eol";
                $foundIt = $this->FillResults(-1);
                return $foundIt;
            }
            if ($debugInsert) $msg .= "I#1656 find a calculated point ($calcLatitudeSaved, $calcLongitudeSaved) within 3 meters best distance was $bestDistMeter meters $eol";
            $foundIt = $this->FillResults($iiSave);
            $foundIt->lat = $calcLatitudeSaved;
            $foundIt->lon = $calcLongitudeSaved;
            $distanceMeterToLine = freewheeling_calculations::distanceMeters($latSearch, $lonSearch, $calcLatitudeSaved, $calcLongitudeSaved);
            if (0 == $bestDistMeter) {
                $foundIt->linePointExists = true;
                return $foundIt;
            } else {
                $foundIt->linePointExists = false;
            }
            return $foundIt;
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg GivenLatLonFindNearestLatLonLinePoint($latSearch, $lonSearch, $description) $errorEnd";
            throw new Exception($msg);
        }   // end catch
    }
    public function GivenLatLonDoInsertIfRequired(lineTableInsert $lineTableInsert): string
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        // given lat lon point find out if all ready on the line
        // if not on the line, insert it to the line by
        //      updating the point array in the lines table
        //      insert the point into the line table at the right place
        try {
            $msg = "";
            $debugInsert = rrwParam::isDebugMode("debugInsert");
            $debugPointArray = rrwParam::isDebugMode("debugPointArray");
            if ($lineTableInsert->milePostsAlong == 72) {
                $debugInsert = true;
                $debugPointArray = true;
            }
            if ($lineTableInsert->linePointExists) {
                if ($debugInsert) $msg .= "I#1635 GivenLatLonDoInsertIfRequired: point is already on the line at point $eol";
                return $msg; // our work is done
            }
            $iiSave = $lineTableInsert->matchPoint;
            $calcLatitude = $lineTableInsert->lat;
            $calcLongitude = $lineTableInsert->lon;
            $description = $lineTableInsert->description;
            if ($debugInsert) {
                $msg = rrwFormat::backtrace("I#1634 GivenLatLonDoInsertIfRequired for " . $lineTableInsert->description);
                $msg .= "$eol I#1639 <strong>GivenLatLonDoInsertIfRequired</strong>($iiSave,$calcLatitude,$calcLongitude,$description)";
            }
            if (-1 == $iiSave) {
                $msg .= "$eol I#1640 <strong>Expected point " . $lineTableInsert->description . "to be near line. not found</strong>" .
                    $lineTableInsert->display("$eol Trying to Inserting this point");
                $msg .= $this->dumpMilageTableByLatitude($calcLatitude, 0, "I#1657 dump of line points  $description ");
                $msg .= "$errorBeg E#1658 $description not found within 3 meters of line $errorEnd";
                $msg .= rrwFormat::backtrace("GivenLatLonDoInsertIfRequired");
                throw new Exception($msg);
            }
            // we have found the point is between ii and ii+1
            // format the coordinates to match contents of the point array
            $lat = $this->lat[$iiSave];
            $lon = $this->lon[$iiSave];
            $CoordinateInPointArray = $lon . "," . $lat . ",0 ";
            $coordinate2Insert = $calcLongitude . "," . $calcLatitude . ",0 ";
            if ($debugInsert) {
                $msg .= "I#1694 CoordinateInPointArray = $CoordinateInPointArray $eol";
                $msg .= "I#1695 coordinate2Insert = $coordinate2Insert $eol";
            }
            $sqlPoints = "select pointList from $wpdbExtra->lines where lineId = " . $this->line_id[$iiSave];
            $recPoints = $wpdbExtra->get_var($sqlPoints);
            $iiPlace = strpos($recPoints, $CoordinateInPointArray);
            if (false === $iiPlace) {
                $msg .= "$errorBeg E#1686 did not find $CoordinateInPointArray in point array $recPoints for lineId " . $this->line_id[$iiSave] . $errorEnd;
                throw new Exception($msg);
            }
            $debugPointArray = true;

            $lengthCoord = strlen($CoordinateInPointArray);
            $iiPlace = $iiPlace + $lengthCoord;
            if ($debugPointArray) $msg .= "I#1679 existing point array = $eol $recPoints $eol";
            $pointArrayNew = substr($recPoints, 0, $iiPlace) .
                $coordinate2Insert .
                substr($recPoints, $iiPlace);
            if ($debugPointArray) $msg .= "I#1676 new point array = $eol $pointArrayNew $eol";
            $sqlUpdate = "update $wpdbExtra->lines set pointList = '$pointArrayNew' where lineId = " . $this->line_id[$iiSave];
            if ($debugPointArray) $msg .= "I#1711 update line table with new point array = $sqlUpdate $eol";
            $updateCnt = $wpdbExtra->query($sqlUpdate);
            //
            $msg .= "updated $updateCnt database entry, now update table for  $CoordinateInPointArray , insert with $coordinate2Insert at $iiSave $eol";

            // database is updated, now rebuild the line table
            if ($debugInsert) $msg .= $this->dumpMilageTableByPoints($iiSave - 6, $iiSave + 6, "I#1630 just before inserting the new point", $iiSave);
            for ($ii = $this->get_latCount() - 1; $ii >= $iiSave; $ii--) {    // move all up by one to make space
                $this->lat[$ii + 1] = $this->lat[$ii];
                $this->lon[$ii + 1] = $this->lon[$ii];
                $this->seq[$ii + 1] = $this->seq[$ii];
                $this->line_id[$ii + 1] = $this->line_id[$ii];
                $this->lineTrailId[$ii + 1] = $this->lineTrailId[$ii];
                $this->milePostsAlong[$ii + 1] = $this->milePostsAlong[$ii];
                $this->distAlongMeters[$ii + 1] = $this->distAlongMeters[$ii];
            }
            if ($debugInsert) $msg .= "old latLon = " . $this->lat[$iiSave] . ", " . $this->lon[$iiSave] . " new latLon = $calcLatitude, $calcLongitude $eol";
            $this->lat[$iiSave] = $calcLatitude; // insert the new point
            $this->lon[$iiSave] = $calcLongitude;
            for ($ii = $iiSave - 2; $ii <= $iiSave + 2; $ii++) {    // update the mileposts and distance for the new point and the next two points
                $distanceMeters = freewheeling_calculations::distanceMeters($this->lat[$ii - 1], $this->lon[$ii - 1], $this->lat[$ii], $this->lon[$ii]);
                $this->distAlongMeters[$ii] = $this->distAlongMeters[$ii - 1] + $distanceMeters;
                $this->milePostsAlong[$ii] = $this->distAlongMeters[$ii] / freeWheel::metersPerMile;
                $this->seq[$ii] = $this->seq[$ii - 1] + 2;
            }
            // we assume that the caller will deal with the moving of the icon to the new location]
            if ($debugInsert) $msg .= $this->dumpMilageTableByPoints($ii - 6, $ii + 6, "I#1636 just after inserting the new point ", $iiSave);
            if ($debugInsert) $msg .= "I#1678 Inserted new point at $calcLatitude, $calcLongitude for icon $description $eol";
        } catch (Exception $e) {
            $msg .= $e->getMessage() . $lineTableInsert->display("$errorBeg E#1660 at bottom of GivenLatLonDoInsertIfRequired $errorEnd");
            throw new Exception($msg);
        }
        return $msg;
    } // end GivenLatLonDoInsertIfRequired

    public  function FillResults($matchPoint): lineTableInsert
    {
        // fill common stuff
        $result = new LineTableInsert();
        $result->matchPoint = $matchPoint;        // private
        if ($matchPoint < 0) {
            $result->lat = 0;
            $result->lon = 0;
            $result->distAlongMeters = 0;
            $result->milePostsAlong = 0;
            $result->linePointExists = false;
            $result->description = "point not found";
        } else {
            $result->lat = $this->lat[$matchPoint];
            $result->lon  = $this->lon[$matchPoint];
            $dist1 = $this->distAlongMeters[$matchPoint];
            if (is_null($dist1)) {
                $dist1 = 0;
            }
            $result->distAlongMeters = $dist1;
            $dist2 = $this->milePostsAlong[$matchPoint];
            if (is_null($dist2))
                $dist2 = 0;
            $result->milePostsAlong = $dist2;
            $result->linePointExists = true;
            $result->description =  $this->line_id[$matchPoint];
        }
        return $result;
    }

    /**
     *  fills in the milePostsAlong table for all points between known mileposts
     *  and for the end points if they are not filled in.
     */
    private  function fillInMileposts()
    {
        // assert that we have a milePostsAlong [entry] for each existing milepost
        // assert have at least one milepost entry in table
        // now distribute mileposts between two known points
        global $eol, $errorBeg, $errorEnd;;
        global $milesDirectionAlongTheLine;

        $msg = "";
        $debugFillMiles = rrwParam::isDebugMode("debugFillMiles");
        if ($debugFillMiles) $msg .= "I#1720 fillInMileposts for $this->trailId $eol";
        try {
            $lastPoint = count($this->milePostsAlong) - 1;
            $milePostStart = -1;                        // used to determine if there are any paired mileposts to distribute between
            for ($ii = 0; $ii <= $lastPoint; $ii++) {
                if (is_null($this->milePostsAlong[$ii])) {
                    continue; // find first milepost with a milage.
                }
                if ($milePostStart < 0) {
                    if ($debugFillMiles) $msg .= "first milepost found at $ii with " . $this->milePostsAlong[$ii] .  $eol;
                    $milePostStart = $ii;
                    continue; // first one found
                }
                if ($debugFillMiles) $msg .= "second milepost found at $ii with " . $this->milePostsAlong[$ii] .  $eol;
                $msg .= self::assignMilageBetween($milePostStart, $ii, "middle of line");
                $milePostStart = $ii;
            } // end for ($ii = 0; $ii <= $lastPoint; $ii++)
            // DEAL WITH NO MILEPOSTS AT ALL
            if (-1 == $milePostStart) {
                //   return "$errorBeg E#1028 no milepost found in fillInMileposts $errorEnd";
                //  deal with the start points not having a milepost
                $milePostStart = 0; // start at zero
                $this->milePostsAlong[0] = 0;
                $msg .= self::assignMilageBetween(0, $lastPoint, "No mileposts at all");
            }
            if (is_null($this->milePostsAlong[0])) {
                //  assign miles between first (zero) and known milepost
                if ($debugFillMiles) $msg .= "assign miles between first (zero) and known milepost $eol";
                for ($ii = 0; $ii <= $lastPoint; $ii++) {
                    if (is_null($this->milePostsAlong[$ii]))
                        continue;   // find first milepost with a milage.
                    $distMiles = $this->distAlongMeters[$ii] / FreeWheel::metersPerMile;  // between first and known milepost
                    $this->milePostsAlong[0] = $this->milePostsAlong[$ii] - ($distMiles * $milesDirectionAlongTheLine);
                    $msg .= self::assignMilageBetween(0, $ii, "start points  no milage");
                    break;
                } // end for ($ii = 0; $ii <= $lastPoint; $ii++)
            } // end if (is_null($this->milePostsAlong[0]))

            if (is_null($this->milePostsAlong[$lastPoint])) {
                //  assign miles between last and known milepost
                for ($ii = $lastPoint; $ii >= 0; $ii--) {
                    if (is_null($this->milePostsAlong[$ii]))
                        continue;   // look for last known milepost
                    $distMiles = ($this->distAlongMeters[$lastPoint] - $this->distAlongMeters[$ii]) / FreeWheel::metersPerMile;
                    $this->milePostsAlong[$lastPoint] =  $this->milePostsAlong[$ii] + ($distMiles * $milesDirectionAlongTheLine); // between last and known milepost
                    $msg .= self::assignMilageBetween($ii, $lastPoint, "end points no milage"); //remaining
                    if ($debugFillMiles)
                        $msg .= "assign to last point distance $distMiles, milepost " . $this->milePostsAlong[$lastPoint] . $eol;
                    break;
                } // end for ($ii = $lastPoint; $ii >= 0; $ii--)
            }  // end if (is_null($this->milePostsAlong[$lastPoint]))
            if ($debugFillMiles) $msg .= "last point $lastPoint has a milepost of " . $this->milePostsAlong[$lastPoint] . $eol;
            if ($debugFillMiles) {
                $msg .= $this->dumpMilageTableByPoints(0, 20, "I#1721 fillInMileposts after all done");
                $msg .= $this->dumpMilageTableByPoints($lastPoint - 5, 32000, "I#1723 fillInMileposts after all done");
                //              $msg .= $this->dumpMilageTableBetweenMiles(4.8, 5.2, "I#1724 fillInMileposts ");
            }
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg E#1722 fillInMileposts $errorEnd";
            throw new Exception($msg);
        }
        return $msg;
    } // end fillInMileposts

    private  function assignMilageBetween($startII, $endII, $title = "")
    {
        global $eol, $errorBeg, $errorEnd;;
        global $meters2Mile;
        $msg = "";
        $debugMiles = false;
        $debugParameters = false;
        if (count(value: $this->milePostsAlong) == 0) {
            $msg .= "$errorBeg E#1011 milePostsAlong is empty $errorEnd";
            throw new Exception($msg);
        }
        if ($debugParameters) $msg .= "I#1680 assignMilageBetween (point [$startII] - mile " . $this->milePostsAlong[$startII] . ") and
                        (point [$endII] with milepost " . $this->milePostsAlong[$endII] . " )type of  $title ) $eol ";
        if ($endII == $startII)
            return "$msg one point only - assignMilageBetween( $startII, $endII, $title) $eol ";
        if ($endII < $startII)
            return "$msg range incorrect - assignMilageBetween( $startII, $endII, $title) $eol ";
        // $ii and $iiNext have mileage - distribute to point between
        if ($endII >= count($this->milePostsAlong) || $startII > count($this->milePostsAlong)) {
            $msg .= rrwFormat::backtrace("assignMilageBetween - out of range");
            return "$msg range incorrect - assignMilageBetween( $startII, $endII, $title) $eol ";
        }
        if (is_null($this->milePostsAlong[$endII]) || is_null($this->milePostsAlong[$startII])) {
            $msg .= "$errorBeg E#1682 one of these is_null between( [$startII] " . $this->milePostsAlong[$startII] . "
                [$endII] " . $this->milePostsAlong[$endII]  . $errorEnd;
        }
        $milepostMiles = $this->milePostsAlong[$endII] - $this->milePostsAlong[$startII];
        $lineMeters = $this->distAlongMeters[$endII] - $this->distAlongMeters[$startII];
        if (0 == $milepostMiles) {
            $mile1 = $this->milePostsAlong[$endII];
            $mile2 = $this->milePostsAlong[$startII];
            $msg .= "E#1703 milepost has zero difference  [$startII] $mile2 , [$endII] $mile1 $eol";
            $msg .= self::dumpMilageTableByPoints(0, $this->get_latCount(), "E#1704 milepost has zero difference  [$startII] $mile2 , [$endII] $mile1 $eol");
            return $msg;
            // possible duplicate millage value
        }
        $ratio = $lineMeters / ($milepostMiles * $meters2Mile);
        if ($debugMiles) $msg .= " $ratio = $lineMeters / ($milepostMiles * $meters2Mile);";
        $calcMeter = $ratio * $meters2Mile;
        if ($debugMiles) $msg .= "I#1699 meters $lineMeters, miles $milepostMiles,
                is $ratio should be close to 1.0000 , calculated mile is $calcMeter meters long$eol";
        if (false) {    // ignore ratio test - cando is really bad
            if ((abs($ratio) > 1.2 || abs($ratio) < 0.9)) {
                $msg .= "$errorBeg I#1041 in
                assignMilageBetween( $startII, $endII ) calculated ratio $ratio should be close to 1.0000 $errorEnd
                    between mile " . $this->milePostsAlong[$startII] . " and " . $this->milePostsAlong[$endII] . "$eol
                        lineMeters = $lineMeters , milepostMeters = " . ($milepostMiles * $meters2Mile) . ",
                        line miles = " . ($lineMeters / $meters2Mile) . " vs milepostMiles = $milepostMiles $eol";
                //$msg .= self::dumpMilageTableByPoints( $startII - 5, $ii + 5, "assignMilageBetween" );
            }
        }
        for ($jj = $startII + 1; $jj < $endII; $jj++) {
            $diffMeters = $this->distAlongMeters[$jj] - $this->distAlongMeters[$startII];  // meters between two points
            $diffMiles = ($diffMeters / $ratio) / $meters2Mile;    // miles between two points
            $this->milePostsAlong[$jj] = $this->milePostsAlong[$startII] + $diffMiles;
        } // end for ($jj = $startII + 1; $jj < $endII; $jj++)
        return $msg;
    } // end assignMilageBetween
    private function addEstimatedMilepost()
    {
        global  $eol, $errorBeg, $errorEnd;
        try {
            ini_set("display_errors", "1");
            $msg = "";
            $debugAddEst = rrwParam::isDebugMode("debugAddEst");
            $debugMileIncrements = rrwParam::Boolean("debugMileIncrements");
            if ($debugAddEst) {
                $msg .= "I#1708 addEstimatedMilepost for $this->trailId $eol";
                $msg .= $this->dumpMilageTableByPoints(-1, 10, "I#1709 addEstimatedMilepost before adding estimated mileposts");
            }
            $mileIncrement = rrwParam::Number("increment", array(), 1); // maybe want half/quarter/tenth miles
            // determine min max of the existing line
            $lastPoint = $this->get_latCount() - 1;
            if (! is_array($this->milePostsAlong)) {
                $msg .= "$errorBeg E#1729 milePostsAlong is not $errorEnd";
                throw new Exception($msg);
            }
            $mileBeg = $this->milePostsAlong[0];            // on end
            $mileEnd = $this->milePostsAlong[$lastPoint];   // or the other end
            if ($mileEnd < $mileBeg) {   // check the order
                $temp = $mileBeg;
                $mileBeg = $mileEnd;
                $mileEnd = $temp;
            }
            $mileBeg = floor($mileBeg + 1);  // start at the next milepost
            if ($debugMileIncrements) {
                $msg .= $this->dumpMilageTableByPoints(-5, 5, "I#17210 before adding estimated mileposts");
                $msg .= $this->dumpMilageTableByPoints($lastPoint - 5, 32000, "I#17211 before adding estimated mileposts");
                $msg .= "I#1712 processing make  estimated miles from (mile = $mileBeg to  $mileEnd; by $mileIncrement) $eol";
            }

            for ($mile = $mileBeg; $mile <= $mileEnd; $mile += $mileIncrement) {
                if ($debugMileIncrements) $msg .= " processing mile $mile $eol";
                $msg .= $this->makeOneMile($mile);
            } // end for adding mile points
        } catch (Exception $e) {
            $msg .= $e->getMessage() . "$errorBeg E#1725 bottom of addEstimatedMilepost $errorEnd";
            throw new Exception($msg);
        }
        return $msg;
    } // end addEstimatedMilepost

    private function makeOneMile($begPost)
    {
        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        try {
            if ($begPost == 997)
                $debugMakeMile = true;
            else
                $debugMakeMile = false;  // for now
            if ($debugMakeMile)                 $msg .= "I#1713<strong>Working on milepost $begPost -----------</strong>$eol";
            if (rrwUtil::isTimeUp()) {
                throw new Exception("$msg $errorBeg E#1714 time limit exceeded, at makeOneMile($this->trailId, $begPost) $errorEnd");
            }
            $lastPoint = count($this->milePostsAlong) - 1;    // last valid entry
            for ($ii = 0; $ii < $lastPoint; $ii++) {
                $milepost1 = $this->milePostsAlong[$ii];
                $milepost2 = $this->milePostsAlong[$ii + 1];
                if ($milepost1 == $begPost || $milepost2 == $begPost) {
                    if ($debugMakeMile) $msg .= "I#1715 mile $begPost already exists between [$ii] $milepost1 and [$ii+1] $milepost2  $eol";
                    return $msg; // it exist so our job is done
                }
                //if ($debugMakeMile) $msg .= "testing $begPost between $milepost1 and $milepost2 $eol    ";
                if (! $this->between($milepost1, $begPost, $milepost2))
                    continue;;

                // found a place to insert the new milepost
                if ($debugMakeMile) {
                    $iiNewHere = $ii + 1;
                    $msg .= "found $begPost between $milepost1 and $milepost2,  at [$ii] got $milepost1, at [";
                    $msg .= $ii + 1 . "] got $milepost2, put $begPost at $iiNewHere  $eol ";
                    $msg .= $this->dumpMilageTableByPoints($ii - 2, $ii + 4, "1704 (before) found place in makeOneMile", $ii);
                    $msg .= $this->dumpMilageTableByPoints($lastPoint - 3, $lastPoint + 5, "I#1705 check end not messed up");
                }
                $deltaMilepost = $milepost2 - $milepost1;
                if (0 == $deltaMilepost)
                    continue;
                $deltaLat = $this->lat[$ii + 1] - $this->lat[$ii];
                $deltaLng = $this->lon[$ii + 1] - $this->lon[$ii];
                $deltaMeter = $this->distAlongMeters[$ii + 1] - $this->distAlongMeters[$ii];
                $begRatio = ($begPost - $milepost1) / $deltaMilepost;
                if ($debugMakeMile) {
                    $xx = $this->distAlongMeters[$ii + 1];
                    $msg .= "delta meter $deltaMeter = $xx - " . $this->distAlongMeters[$ii] . " ;$eol";
                    $msg .= "delta milepost$deltaMilepost = $milepost2 - $milepost1;$eol";
                    $msg .= "rato $begRatio = ($begPost - $milepost1) / $deltaMilepost; $eol ";
                    $msg .= "delta meter $deltaMeter $eol;";
                }
                $latitudeMid = $this->lat[$ii] + ($deltaLat * $begRatio);
                $longitudeMid = $this->lon[$ii] + ($deltaLng * $begRatio);
                $latitudeMid = round($latitudeMid, freeWheel::latRoundTo);
                $longitudeMid = round($longitudeMid, freeWheel::latRoundTo);
                $msgErr = "I#266 adjusting the lat array from <strong>$ii</strong> to $lastPoint $eol ";
                if ($debugMakeMile)
                    $msg .= " $msgErr ";
                $cnt = 0;
                $iiNewHere = $ii + 1;
                if (count($this->seq) != count($this->lat) || count($this->lat) != count($this->lon) || count($this->lat) != count($this->line_id) || count($this->lat) != count($this->milePostsAlong) || count($this->lat) != count($this->distAlongMeters)) {
                    $msg .= "$errorBeg count of arrays do not match latCount = " . count($this->lat) . ", sequence " .
                        count($this->seq) . ", longitude " . count($this->lon) . ", line id" . count($this->line_id) .
                        ", Miles post along " . count($this->milePostsAlong) . ",dist along meter" . count($this->distAlongMeters) . $errorEnd;
                }
                for ($jj = $lastPoint; $jj >= $ii; $jj--) {
                    $cnt++;
                    if ($cnt > 15000)
                        throw new Exception("$errorBeg too big a loop count in adjust lat array $errorEnd");
                    $this->lat[$jj + 1] = $this->lat[$jj];
                    $this->lon[$jj + 1] = $this->lon[$jj];
                    $this->seq[$jj + 1] = $this->seq[$jj];
                    $this->line_id[$jj + 1] = $this->line_id[$jj];
                    $this->milePostsAlong[$jj + 1] = $this->milePostsAlong[$jj];
                    $this->distAlongMeters[$jj + 1] = $this->distAlongMeters[$jj];
                    $this->lineTrailId[$jj + 1] = $this->lineTrailId[$jj];
                } // end for ($jj = $lastPoint; $jj >= $ii; $jj--)  - The shift loop
                if ($debugMakeMile)
                    $msg .= "I#725 replacing the values at <strong>$iiNewHere</strong> $eol";
                $this->lat[$iiNewHere] = $latitudeMid;
                $this->lon[$iiNewHere] = $longitudeMid;
                $this->milePostsAlong[$iiNewHere] = $begPost;
                $this->distAlongMeters[$iiNewHere] = freewheeling_calculations::distanceMeters(
                    $this->lat[$iiNewHere - 1],
                    $this->lon[$iiNewHere - 1],
                    $latitudeMid,
                    $longitudeMid
                ) +
                    $this->distAlongMeters[$iiNewHere - 1];
                $this->distAlongMeters[$iiNewHere + 1] = freewheeling_calculations::distanceMeters(
                    $latitudeMid,
                    $longitudeMid,
                    $this->lat[$iiNewHere + 1],
                    $this->lon[$iiNewHere + 1]
                ) +
                    $this->distAlongMeters[$iiNewHere];
                //$lineId[$ii] = $lineId[$ii}  // use the same.
                if ($debugMakeMile) {
                    $msg .= $this->dumpMilageTableByPoints($ii - 3, $ii + 4, "I#178 (after) upshift for the insert", $iiNewHere);
                    $msg .= $this->dumpMilageTableByPoints($lastPoint - 3, $lastPoint + 5, "i#1719 check end not messed up");
                } // leave the database line string alone.
                $msg .= FreewheelingEditDistance::insertMilePostIntoDatabase(
                    $latitudeMid,
                    $longitudeMid,
                    $begPost,
                    "Estimated Milepost $begPost",
                    $this->trailId
                );
                // above also adjusts the icon lat,lng, but not the milage along table
                return $msg;
            } // end for $ii = 0; $ii < $lastPoint; $ii++   The search for a place loop
            if ($this->between($this->milePostsAlong[0], $begPost, $this->milePostsAlong[$lastPoint])) {
                // if the given milepost is between the first and last milepost, we should have found a place to insert it
                $msg .= "$errorBeg E#3751 milepost $begPost is between " . $this->milePostsAlong[0] . " and " . $this->milePostsAlong[$lastPoint] .
                    " but could not find a place to insert it $errorEnd";
                throw new Exception($msg);
            }
            // this should not happen because we should have found a place to insert the milepost
            throw new Exception("$errorBeg E#3752could not find a place to insert milepost $begPost $errorEnd");
        } // end try
        catch (Exception $ex) {
            $msg .= $ex->getMessage();
            throw new Exception("$msg  $errorBeg  E#376 in makeOneMile $errorEnd");
        }
    } // end makeOneMile

    /*
    public function GivenMileageFindLatLon($milepost, &$latitude, &$longitude)
    {
        // given a milepost, find the lat,lng point on the line
        //  if the milepost is not on the line, return the nearest point on the line
        global $eol, $errorBeg, $errorEnd;
        $msg = "";
        $debugMiles = false;
        $debugMatch = false;
        $debugFind = false;
        $debugFind = false;
		$latCount = $this->get_latCount();
        for ($ii = 0; $ii < $latCount; $ii++) {
            if ($debugFind) $msg .= "I#1631 $ii $this->milePostsAlong[$ii] $this->lat[$ii] $this->lng[$ii] $eol";
            if ($this->milePostsAlong[$ii] == $milepost) { // found it as equal
                $latitude = $this->lat[$ii];
                $longitude = $this->lng[$ii];
                if ($debugFind) $msg .= "I#1633 found milepost $milepost at $latitude, $longitude $eol";
                return $msg;
            }
            if ($milePostsAlong[$ii] > $milepost) {
                $ratio = ($milepost - $milePostsAlong[$ii - 1]) / ($milePostsAlong[$ii] - $milePostsAlong[$ii - 1]);
                $latitude = $lat[$ii - 1] + ($lat[$ii] - $lat[$ii - 1]) * $ratio;
                $longitude = $lng[$ii - 1] + ($lng[$ii] - $lng[$ii - 1]) * $ratio;
                if ($debugFind) $msg .= "I#1637 found milepost $milepost between $ii-1 and $ii at $latitude, $longitude $eol";
                return $msg;
            }
        }
        $milepostEnd = $milePostsAlong[count($lat) - 1];
        $msg .= "$errorBeg E#1638 milepost $milepost not found in milePostsAlong $eol
                    milepost[0] = $milePostsAlong[0], milePostsAlong[end] = $milepostEnd $errorEnd";
        $msg .= self::dumpMilageTableBetweenMiles($milepost - 1, $milepost + 1, "GivenMileageFindLatLon");
        throw new Exception($msg);
    } // end GivenMileageFindLatLon
*/
    public  function assignMilageToNulls()
    {
        global $eol, $errorBeg, $errorEnd;;
        global $wpdbExtra;
        $msg = "";
        $debugNull = rrwParam::isDebugMode("debugNull");
        // now assign milage to points that are off the line
        try {
            $trailId = $this->trailId;
            $selections = " trailId = '$trailId' ";
            // now assign milage to points that are on the line
            if ($debugNull) $msg .= $this->dumpMilageTableBetweenMiles(0, 10, "I#1684 assignMilageToNulls before assigning milage to nulls");
            $sqlNullNonMilage = "update $wpdbExtra->icons set milepost = null where not onMap like '%milepost%' and $selections";
            $nonToAssign = $wpdbExtra->query($sqlNullNonMilage);
            $sqlNull = "select iconId, iconName, iconStyle, latitude, longitude, milepost, onMap, milepostPrefix
			            from $wpdbExtra->icons
                        where (milepost is null or milepost = '')
                                and $selections";
            if ($debugNull) $msg .= "E#1683 assign Milage To Nulls:: $sqlNull $eol";
            $recsNulls = $wpdbExtra->get_resultsA($sqlNull);
            if ((0 == $wpdbExtra->num_rows) || is_null($recsNulls))
                return "$msg I#1687 No points with a null mileage, nothing to do in assignMilageToNulls$eol ";
            if ($debugNull) $msg .= "I#1666 there are " . count($recsNulls) . " points with a null mileage $eol";
            foreach ($recsNulls as $recNull) {
                $milepost = $recNull["milepost"];
                if ($debugNull) $msg .= "I#1728 working on $recNull[iconName] with milepost $milepost $eol";
                if ($milepost !== null)
                    continue;
                $iconId = $recNull["iconId"];
                $iconName = $recNull["iconName"];
                $onMap = $recNull["onMap"];
                $latitude = $recNull["latitude"];
                $longitude = $recNull["longitude"];
                $milepostPrefix = $recNull["milepostPrefix"];
                if ($debugNull) $msg .= "I#1688 working on $iconName, $onMap, $latitude, $longitude, $milepost $eol";
                // point is the existing point on the line that is nearest to this point
                $result = $this->givenLatLonFindNearestMatchPoint($latitude, $longitude, $iconName, $msg);
                if ($debugNull) $msg .= $result->display("E#1664 found a point in a assign-Milage-to-Nulls of ");
                $matchPoint = $result->matchPoint;
                if ($matchPoint < 0) {
                    $msg .= "$errorBeg E#1649 '$iconName' is not near the line. No match point found $errorEnd";
                    throw new Exception($msg);
                }
                $distanceMeters = distanceMeters($latitude, $longitude, $this->lat[$matchPoint], $this->lon[$matchPoint]);
                if ($debugNull) $msg .= $this->dumpMilageTableByPoints($matchPoint - 2, $matchPoint + 2, "assignMilageToNulls", $matchPoint);
                if ($distanceMeters < 3 || ($distanceMeters < 75 && stripos($onMap, "offLine") !== false)) {
                    if ($debugNull) $msg .=  "I#1661 '$iconName' is $distanceMeters meters from the line at match point $matchPoint, so assigning milepost" .
                        $this->milePostsAlong[$matchPoint] . $eol;
                    $sqlMile = "update $wpdbExtra->icons set milepost = " . $this->milePostsAlong[$matchPoint] . " where iconId = $iconId ";
                    $wpdbExtra->query($sqlMile);
                    continue;
                } // point is too far from the line to assign a milepost
                $msg .= "$errorBeg E#1702 '$iconName' with $onMap is $distanceMeters meters from the line, which is too far to assign a milepost $errorEnd";
                // on to the next icon
            }  // end foreach ($recNulls as $recNull) for non off line points
        } // end try
        catch (Exception $ex) {
            $msg .= "$errorBeg  E#1663 thrown out of assignMilageToNulls: " . $ex->getMessage() .  $errorEnd;
            throw new Exception("$msg");
        }
        return $msg;
    } // end assignMilageToNulls


    public  function dumpMilageTableBetweenMiles($startMile, $endMile, $title)
    {
        global $eol, $errorBeg, $errorEnd;
        global $meters2Mile;

        $msg = "";
        $debugMiles = true;
        // need to handle decreasing milepost values, for not print all
        if ($debugMiles) $msg .= "dumpMilageTableBetweenMiles( $startMile, $endMile, $title) $eol ";
        if ($startMile > $endMile) {
            $msg .= "$errorBeg E#1670 wrong order $startMile is greater than $endMile";
            return $msg;
        }
        $needHeader = true;
        for ($ii = 1; $ii < count($this->milePostsAlong); $ii++) {
            if (is_null($this->milePostsAlong[$ii]))
                continue; // ignore empty.
            if ($this->milePostsAlong[$ii] >= $startMile && $this->milePostsAlong[$ii]  <= $endMile) {
                if ($needHeader) {
                    $msg .= "<h3>Milage Table Between $startMile and $endMile $eol</h3><table>";
                    $msg .= rrwFormat::HeaderRow(
                        "Count",
                        "Latitude",
                        "Longitude",
                        "Meters",
                        "Miles",
                        "milepost"
                    );
                    $needHeader = false;
                    $DistAlongMetersPast = $this->distAlongMeters[$ii];
                    $latPast = $this->lat[$ii];
                    $lngPast = $this->lon[$ii];
                    $milePostsAlongPast = $this->milePostsAlong[$ii];
                }
                $msg .= rrwFormat::CellRow(
                    " ",
                    " ",
                    $this->lat[$ii] - $latPast,
                    "",
                    $this->lon[$ii] - $lngPast,
                    "",
                    $this->distAlongMeters[$ii] - $DistAlongMetersPast,
                    " ",
                    ($this->distAlongMeters[$ii] - $DistAlongMetersPast) / $meters2Mile,
                    " ",
                    $this->milePostsAlong[$ii] - $milePostsAlongPast
                );
                $msg .= rrwFormat::CellRow(
                    $ii,
                    $this->lat[$ii],
                    "",
                    $this->lon[$ii],
                    "",
                    $this->distAlongMeters[$ii],
                    "",
                    $this->distAlongMeters[$ii] / $meters2Mile,
                    "",
                    $this->milePostsAlong[$ii]
                );
            }
        }
        if (!$needHeader) $msg .= "</table>";
        return $msg;
    }
    public function dumpMilageTableByLatitude($latitude, $tolerance = 0, $title = "")
    {
        global $eol, $errorBeg, $errorEnd;
        $distRoundTo = 5;

        // collection of points
        try {
            $msg = "";
            $debugWho = false;
            if ($debugWho)
                $msg .= rrwFormat::backtrace("I#1726 dumpMilageTableByLatitude( $latitude, $tolerance, $title) called from");
            if (! is_array($this->lat)) {
                $msg .= "$errorBeg E#1074 lat is not an array $errorEnd";
                return $msg;
            }
            if (count($this->lat) <= 0 || count($this->milePostsAlong) <= 0) {
                $msg .= "$errorBeg E#1076 lat is empty $errorEnd";
                return $msg;
            }
            if ($tolerance <= 0)
                $tolerance = 0.00001;

            for ($ii = 0; $ii < count($this->lat); $ii++) {
                if (abs($this->lat[$ii] - $latitude) <= $tolerance) {
                    break;
                }
            }
            $start = $ii;
            for (; $ii < count($this->lat); $ii++) {
                if (abs($this->lat[$ii] - $latitude) > $tolerance) {
                    break;
                }
            }
            $end = $ii;
            $msg .= $this->dumpMilageTableByPoints($start, $end, $title, -1);
        } catch (Exception $ex) {
            $msg .= "$errorBeg E#1706 bottom of dumpMilageTableByLatitude
: " . $ex->getMessage() . $errorEnd;
        }
        return $msg;
    } // end dumpMilageTableByLatitude

    public function dumpMilageTableByPoints($start = 0, $end = 32000, $title = "", $matchPoint = -1)
    {
        global $eol, $errorBeg, $errorEnd;
        $distRoundTo = 5;

        // collection of points
        try {
            $msg = "";
            $debugWho = false;
            if ($debugWho)
                $msg .= rrwFormat::backtrace("I#1727 dumpMilageTableByPoints( $start, $end, $title, $matchPoint) called from");
            if (! is_array($this->lat)) {
                $msg .= "$errorBeg E#1074 lat is not an array $errorEnd";
                return $msg;
            }
            if (count($this->lat) <= 0 || count($this->milePostsAlong) <= 0) {
                $msg .= "$errorBeg E#1076 lat is empty $errorEnd";
                return $msg;
            }
            if ($end > count($this->lat))
                $end = count($this->lat);
            if ($start < 0)
                $start = 0;
            if ($start > $end - 1)
                $start = $end - 1;
            $tableHeader = rrwFormat::HeaderRow(
                "Count",
                "Latitude",
                "Longitude",
                "lineWork Meters",
                "delta lineWork Meters",
                "lineWork Miles",
                "delta Line Work Miles",
                "milepost",
                "delta milesPostAlong",
                "lineId",
                "lineName"
            );
            if (! empty($title))
                $title = " $title $eol";
            $msg .= "<h3>$title</h3>";
            $msg .= "<table>$tableHeader";
            $color = "white";
            $displayMetersPast = Round($this->distAlongMeters[$start], $distRoundTo);
            $displayPostPast = rrwFormat::milePostText($this->milePostsAlong[$start]);
            $displayMilesPast = $this->distAlongMeters[$start] / freeWheel::metersPerMile;
            for ($ii = $start; $ii < $end; $ii++) {
                if (array_key_exists($ii, $this->distAlongMeters)) {
                    $displayMeters = Round($this->distAlongMeters[$ii], $distRoundTo);
                    $displayMiles = Round($this->distAlongMeters[$ii] / freeWheel::metersPerMile, $distRoundTo);
                } else {
                    $displayMeters = "no meters";
                    $displayMiles = "";
                }
                if (array_key_exists($ii, $this->milePostsAlong)) {
                    $displayPost = rrwFormat::milePostText($this->milePostsAlong[$ii]);
                } else {
                    $displayPost = "";
                }
                if ($ii == $matchPoint) {
                    $color = "#fadbd8";
                    $displayII = "<strong>$ii</strong>";
                } else {
                    $displayII = $ii;
                }
                $msg .= rrwFormat::CellRow(
                    $color,
                    $displayII,
                    $this->lat[$ii],
                    $this->lon[$ii],
                    $displayMeters,
                    $displayMeters - $displayMetersPast,
                    $displayMiles,
                    $displayMiles - $displayMilesPast,
                    $displayPost,
                    "$displayPost - $displayPostPast",
                    $this->line_id[$ii],
                    $this->lineTrailId[$ii]
                );
                $color = rrwUtil::colorSwap($color);
                $displayMetersPast = $displayMeters;
                $displayPostPast = $displayPost;
                $displayMilesPast = $displayMiles;
            }
            $msg .= "$tableHeader</table>";
        } catch (Exception $ex) {
            $msg .= "$errorBeg E#1707 bottom of dumpMilageTableByPoints: " . $ex->getMessage() . $errorEnd;
            throw new Exception($msg);
        }
        return $msg;
    } // end dumpMilageTableByPoints
    /**
     * Asserts that all mile-related icons for the current trail lie exactly on the line geometry.
     *
     * This method retrieves all icons with names starting with 'mile' or 'milepost' for the current trail and
     *        checks if their latitude and longitude match any point on the line geometry.

     *
     * @throws Exception If any mile-related icon is not on the line.
     *
     * @return string A message indicating any intermediate results or an empty string if all checks pass.
     */
    public function checkMilePointsOnLine()
    {
        global $eol, $errorBeg, $errorEnd;
        global $wpdbExtra;
        $msg = " ";
        $trailOrGroupId = $this->trailId;
        $sqlWhere = $wpdbExtra->sqlWhereIcons($trailOrGroupId);

        $sqlIcons = "select iconName, latitude, longitude
                        from $wpdbExtra->icons where $sqlWhere and (iconName like 'mile%' or onMap like 'milepost%')
                        order by milepost ";
        $recIcons = $wpdbExtra->get_resultsA($sqlIcons);
        foreach ($recIcons as $recIcon) {
            $latitude = $recIcon["latitude"];
            $longitude = $recIcon["longitude"];
            $checkResults = $this->GivenLatLonFindExactMatch($latitude, $longitude);
            if (! $checkResults->linePointExists) {
                $iconName = $recIcon["iconName"];
                $msgError = "$errorBeg E#1671 $iconName was not on line $errorEnd";
                throw new Exception($msgError);
            }
        } // end foreach
        return $msg;
    } // end checkMilePointsOnLine

    public function between($leftRight, $look, $rightLeft): bool
    {
        //$value = ( $min <= $look && $look <= $max ) || ( $max <= $look && $look <= $min );
        //  debug
        //   print "($min <= $look && $look <= $max) || ($max <= $look && $look <= $min)";
        //   print ($value ? "true" : "false") . " -- " . ($min - $look) . "," . ($look - $max) . "$eol";
        return ($leftRight <= $look && $look <= $rightLeft) || ($rightLeft <= $look && $look <= $leftRight);
    } // end between
} // end class lineTable
/**
 * Represents a found match point for a point into a line table,
 * capturing the matched point index, coordinates, existence flag,
 * description
 * coordinates could be the coordinates of the $ii point,
 *      or the coordinates of the point to be inserted after the $ii point,
 *      depending on the context of the search and the match.
 */
class lineTableInsert
{
    public int    $matchPoint;      // Index of the matched point in the existing line.
    public float  $lat;       // //Latitude of the new point (decimal degrees).
    public float  $lon;      //Longitude of the new point (decimal degrees).
    public bool   $linePointExists;  //Whether the new point already exists in the dataset.
    public string $description;  //Description or note about the insertion context.
    public float  $distAlongMeters; // Distance along the line in meters at the match point.
    public float  $milePostsAlong; // Milepost value at the match point.
    public function __construct()
    {
        $this->matchPoint = -1;
        $this->lat = 0;
        $this->lon = 0;
        $this->linePointExists = false;
        $this->distAlongMeters = -100000;
        $this->milePostsAlong = 100000;
        $this->description = "initialized lineTableInsert";
    }
    public function display($description): string
    {
        global $eol;
        $msg = "lineTableInsert: $description: $eol
            &nbsp; &nbsp; matchPoint = " . $this->matchPoint . " $eol
            &nbsp; &nbsp; this lat = " . $this->lat . " $eol
            &nbsp; &nbsp;  this lon = " . $this->lon . " $eol
            &nbsp; &nbsp; linePointExists = " . ($this->linePointExists ? "true" : "false") . " $eol
            &nbsp; &nbsp; distAlongMeters = " . $this->distAlongMeters . $eol;

        return $msg;
    }   // end resultsDisplay
}       // end class lineTableInsert